/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: fdobj.c,v 2.81 2001/03/27 06:59:24 cmalek Exp $
** Program: xdphys
**  Module: fdobj.c
**  Author: mazer
** Descrip: filedata object module
**
** Revision History (most recent last)
**
** Wed Jun  9 19:44:58 1993 mazer
**  created
**
** Fri Jun 11 19:39:09 1993 mazer
**  finished, more or less..
**  
** Wed Dec  1 13:07:10 1993 mazer
**  vufile() deleted and FDO_File_New modifed to supply
**  the same functionality.. (when parent == NULL)
**
** 96.fall bjarthur
**   basically, completely rewrote.
**
** 97.1 bjarthur
**   fixed bug if Fit menu button
**
** 97.4 bjarthur
**   changed DOWLDIRNEW to XDPHYSDIR
**
** 97.7 bjarthur
**   changed DumpText to have a default extention of .ext to be compatible
**   with matlab.  also changed all the comment characters from ; to % for
**   the same reason.
**
*******************************************************************/

#ifdef SANAL

#include "xdphyslib.h"
#include "FileNom.h"
#include "FileNomP.h"

extern int alphasort();

static void CloseCB(Widget, XtPointer, XtPointer);
static void HelpCB(Widget, XtPointer, XtPointer);
static void SetFlags(FDO*);
static void UpdateCB(Widget, XtPointer, XtPointer);
static void DestTogCB(Widget, XtPointer, XtPointer);
static void KeepCB(Widget, XtPointer, XtPointer);
static void UnKeep(FDO*);
static void DumpPrinter(Widget, XtPointer, XtPointer);
static void DumpPS(Widget, XtPointer, XtPointer);
static void DumpFIG(Widget, XtPointer, XtPointer);
static void DumpText(Widget, XtPointer, XtPointer);
static void DumpSTDOUT(Widget, XtPointer, XtPointer);
static void DoXDprint(Widget, XtPointer, XtPointer);
static void DoXDprintRayleigh(Widget, XtPointer, XtPointer);
static void ZoomCB(Widget, XtPointer, XtPointer);
static void UnZoomCB(Widget, XtPointer, XtPointer);
static void SelectFileCB(Widget, XtPointer, XtPointer);
static void SelectLineCB(Widget, XtPointer, XtPointer);
static void ClearCB(Widget, XtPointer, XtPointer);
static void RightviewCB(Widget, XtPointer, XtPointer);
static void LeftviewCB(Widget, XtPointer, XtPointer);
static void UpviewCB(Widget, XtPointer, XtPointer);
static void DownviewCB(Widget, XtPointer, XtPointer);
static void Duplicate(Widget, XtPointer, XtPointer);
static void RereadDirCB(Widget, XtPointer, XtPointer);
static void ClickCB(Widget, XtPointer, XtPointer);

static FDO *FDObj_Init(FILEDATA *, FDObj_ViewType*);
static void FDObj_UpdateCore(FDO *);
static void Free_Data_All(FDO*);
static int SameModule(FILEDATA*,FILEDATA*);
static float adjed_dist(float, float, AtPointCallbackData*, double, double,
      double, double);
static int SupportedModule(FILEDATA*);
static void SensitizeButtons(FDO*,FILEDATA*);
static char *compute_pat(FDO*, int);

char *FDObj_Colors[] = { "blue", "red", "darkgreen", "orange", "violet",
      "yellow" };
int FDObj_numColors = 6;

AtPlotMark FDObj_Mark[] = { AtMarkSQUARE, AtMarkCIRCLE,
      AtMarkDIAMOND, AtMarkTRIANGLE1 };
int FDObj_numMark = 4;

AtPlotLine FDObj_Line[] = { AtLineSOLID, AtLineDASHED,
      AtLineDOTTED, AtLineDOTTED2, AtLineDOTTED3 };
int FDObj_numLine = 6;

FDO *FDObj_FileBrowser_New(char *orig_wildcard, Widget parent,
			FDObj_ViewType *view)
{
	char *wildcard;
	FDO *fdo;
	struct direct **namelist;
	int i, num;
	extern int scandir();
	FDObj_ViewType def_view = { 0, 0 };

	/* this introduces is a SMALL memory leak: */
	wildcard = strsave(orig_wildcard);

	num = scandir(".", &namelist, NULL, alphasort);
	for (fdo = NULL, i = 0; i < num; i++) {
		if (fdo == NULL && match(wildcard, namelist[i]->d_name))
			fdo = FDObj_File_New(namelist[i]->d_name, parent,
					     (view == NULL) ? (&def_view) : view);
		free(namelist[i]);
	}
	free(namelist);

	if (fdo == NULL)
		fdo =
		    FDObj_New(NULL, parent, (view == NULL) ? (&def_view) : view, 0);

	fdo->filenom = XtVaCreateManagedWidget("fileNominator",
					       fileNominatorWidgetClass, fdo->form,
					       XtNfromVert, NULL, XtNfromHoriz,
					       fdo->graph, XtNshowDotFiles, False,
					       XtNselectAction, "Plot",
					       XtNcancelAction, "Reread", XtNpattern,
					       wildcard, XtNnumberRows, 18, NULL);

	XtAddCallback(fdo->filenom, XtNselectCallback, SelectFileCB, fdo);
	XtAddCallback(fdo->filenom, XtNcancelCallback, RereadDirCB, fdo);

	return (fdo);
}

static FDO *FDObj_Init(FILEDATA *fd, FDObj_ViewType *view)
{
	FDO *fdo;

	assert((fdo = (FDO*)calloc(sizeof(FDO), 1))!=NULL);

	fdo->baw = 0;
	fdo->no_grid = 0;
	fdo->legend = 0;
	fdo->log_x = 0;
	fdo->log_y = 0;
	fdo->no_errs = 0;
	fdo->normalize = 0;
	fdo->show_raw_points = 0;
	fdo->no_text_header = 0;

	fdo->destroy_old = 1;
	fdo->just_zoomed = 0;
	fdo->selected = NULL;
	fdo->point_tag = NULL;
	fdo->to_tty = 0;
	fdo->view.lr = view->lr;
	fdo->view.ud = view->ud;

	assert((fdo->sum_code = (int*)malloc(1*sizeof(int)))!=NULL);
	fdo->sum_code[0] = MU_CODE;

	assert((fdo->fds = (FILEDATA**)malloc(1*sizeof(FILEDATA*)))!=NULL);
	fdo->fds[0] = fd;
	fdo->nfds = (fdo->fds[0]==NULL) ? 0 : 1;

  if(!fdo->no_X) {
		assert((fdo->atlines = (Widget**)malloc(1*sizeof(Widget*)))!=NULL);
		assert((fdo->num_atlines = (int*)malloc(1*sizeof(int)))!=NULL);
		fdo->atlines[0]=NULL;
		fdo->num_atlines[0]=0; }

	if (fdo->fds[0]) {
		fdo->epoch = FD_GF(fdo->fds[0], "epoch");
		fdo->dur = FD_GF(fdo->fds[0], "dur");
		fdo->delay = FD_GF(fdo->fds[0], "delay"); }

	fdo->meta_num_all = 0;
	fdo->ndata_all = NULL;
	fdo->xdata_all = NULL;
	fdo->ydata_all = NULL;

  assert((fdo->flag_menu = (MENU_ENTRY*)malloc(11*sizeof(MENU_ENTRY)))!=NULL);
  fdo->flag_menu[0].label_str = "flags";
  fdo->flag_menu[0].callback_fn = NULL;
  fdo->flag_menu[0].callback_data = NULL;
  fdo->flag_menu[1].label_str = "b&w";
  fdo->flag_menu[1].callback_fn = MENU_NOCB;
  fdo->flag_menu[1].callback_data = &(fdo->baw);
  fdo->flag_menu[2].label_str = "no_grid";
  fdo->flag_menu[2].callback_fn = MENU_NOCB;
  fdo->flag_menu[2].callback_data = &(fdo->no_grid);
  fdo->flag_menu[3].label_str = "legend";
  fdo->flag_menu[3].callback_fn = MENU_NOCB;
  fdo->flag_menu[3].callback_data = &(fdo->legend);
  fdo->flag_menu[4].label_str = "log_x";
  fdo->flag_menu[4].callback_fn = MENU_NOCB;
  fdo->flag_menu[4].callback_data = &(fdo->log_x);
  fdo->flag_menu[5].label_str = "log_y";
  fdo->flag_menu[5].callback_fn = MENU_NOCB;
  fdo->flag_menu[5].callback_data = &(fdo->log_y);
  fdo->flag_menu[6].label_str = "no_errs";
  fdo->flag_menu[6].callback_fn = MENU_NOCB;
  fdo->flag_menu[6].callback_data = &(fdo->no_errs);
  fdo->flag_menu[7].label_str = "normalize";
  fdo->flag_menu[7].callback_fn = MENU_NOCB;
  fdo->flag_menu[7].callback_data = &(fdo->normalize);
  fdo->flag_menu[8].label_str = "raw_points";
  fdo->flag_menu[8].callback_fn = MENU_NOCB;
  fdo->flag_menu[8].callback_data = &(fdo->show_raw_points);
  fdo->flag_menu[9].label_str = "no_text_header";
  fdo->flag_menu[9].callback_fn = MENU_NOCB;
  fdo->flag_menu[9].callback_data = &(fdo->no_text_header);
  fdo->flag_menu[10].label_str = NULL;
  fdo->flag_menu[10].callback_fn = NULL;
  fdo->flag_menu[10].callback_data = NULL;

	return(fdo);
}

FDO *FDObj_New(FILEDATA *fd, Widget parent, FDObj_ViewType *view, int no_X)
{
	FDO *fdo;
	Widget tmp;
	Widget doc;

	fdo = FDObj_Init(fd, view);

  if(no_X) {
    fdo->no_X = 1;
    return(fdo); }

	fdo->form = XtVaCreateManagedWidget("FDObj_form", formWidgetClass, parent,
        NULL);

	fdo->close = button_new(fdo->form,"close","C", CloseCB,
		(XtPointer) fdo, NULL, NULL);
	doc = button_new(fdo->form,"Help","Help", HelpCB,
		(XtPointer) fdo, fdo->close, NULL);
	tmp = button_new(fdo->form,"clr","Clr",ClearCB,
		(XtPointer) fdo,doc,NULL);
	fdo->between = button_new(fdo->form,"redraw","Redraw",
		UpdateCB, (XtPointer) fdo, tmp, NULL);
	tmp = button_new(fdo->form,"reset","Unzm",  UnZoomCB,
		(XtPointer) fdo, fdo->between, NULL);
	fdo->over = button_new(fdo->form,"over","@New", DestTogCB, 
		(XtPointer) fdo, tmp, NULL);
	fdo->keep = button_new(fdo->form,"keep","Keep", KeepCB, 
		(XtPointer) fdo, fdo->over, NULL);
	fdo->dup = button_new(fdo->form,"dup","Dup", Duplicate, 
		(XtPointer) fdo, fdo->keep, NULL);

  fdo->flags = menu_new(fdo->form, "Flags", fdo->flag_menu, fdo->dup, NULL);
	tmp = make_parms(fdo->form,"parms","Parms",fdo->flags,NULL);

	tmp = button_new(fdo->form,"prev","<-",LeftviewCB, (XtPointer) fdo,
		NULL, doc);
	tmp = button_new(fdo->form,"next","->",RightviewCB, (XtPointer) fdo,
		tmp, doc);
	fdo->down = button_new(fdo->form,"down","\\/",DownviewCB, (XtPointer) fdo,
		tmp, doc);
	fdo->up = button_new(fdo->form,"up","/\\",UpviewCB, (XtPointer) fdo,
		fdo->down, doc);
	fdo->vsm = menubutton_new(fdo->form,"Var",fdo->up,doc,&(fdo->vsmpsh));
	fdo->fit = menubutton_new(fdo->form,"Fit",fdo->vsm,doc,&(fdo->fitpsh));

	fdo->dump = menubutton_new(fdo->form,"Dump",fdo->fit,doc,&fdo->dumppsh);
	menubutton_add(fdo->dumppsh,"to Printer",DumpPrinter,(XtPointer) fdo);
	menubutton_add(fdo->dumppsh,"to Postscript file",DumpPS,(XtPointer) fdo);
	menubutton_add(fdo->dumppsh,"to XFig file",DumpFIG,(XtPointer) fdo);
	menubutton_add(fdo->dumppsh,"to text file",DumpText,(XtPointer) fdo);
	menubutton_add(fdo->dumppsh,"to stdout",DumpSTDOUT,(XtPointer) fdo);
	menubutton_add(fdo->dumppsh,"xdprint ",DoXDprint,(XtPointer) fdo);
	menubutton_add(fdo->dumppsh,"xdprint (rayleigh)",DoXDprintRayleigh, (XtPointer) fdo);

	tmp = button_new(fdo->form,"com","Com",FDObj_CommentsCB,fdo,fdo->dump,doc);

	fdo->clickinfo = XtVaCreateManagedWidget("clickinfo", labelWidgetClass,
		fdo->form, XtNlabel, "            ", XtNfromVert, doc,
		XtNfromHoriz, tmp, XtNresizable, True, XtNborderWidth,0, NULL);

	fdo->graph = XtVaCreateManagedWidget("graph", atPlotterWidgetClass,
		fdo->form, XtNwidth, PercentDisplayWidth(35, 400), XtNheight,
		PercentDisplayHeight(35, 350), XtNshowLegend, False, XtNfromHoriz,
		NULL, XtNfromVert, fdo->fit, NULL);

	XtAddCallback(fdo->graph, XtNdragCallback, ZoomCB, fdo);
	XtAddCallback(fdo->graph, XtNselectCallback, SelectLineCB, fdo);
	XtAddCallback(fdo->graph, XtNclickCallback, ClickCB, fdo);

	fdo->xaxis = XtVaCreateWidget("xaxis", atQuadAxisWidgetClass, fdo->graph,
		XtNvertical, False, XtNlabel, "undef", NULL);
	fdo->yaxis = XtVaCreateWidget("yaxis", atQuadAxisWidgetClass, fdo->graph,
		XtNvertical, True, XtNlabel, "undef", NULL);

	XtVaSetValues(fdo->graph, XtNxAxis, fdo->xaxis, XtNyAxis, fdo->yaxis, 
		NULL);

	XtInstallAllAccelerators(fdo->graph, fdo->form);

	return(fdo);
}

static void FDObj_UpdateCore(FDO *fdo)
{
  int j, l;

  fdo->sum_code[fdo->nfds-1]=lookupParms_int("parms.spikemask");

  if(!fdo->no_X) {
    for(l=0; l<fdo->nfds; l++) {
      for(j=0; j<fdo->num_atlines[l]; j++) {
        if(fdo->atlines[l][j]!=NULL)
          XtDestroyWidget(fdo->atlines[l][j]); }
      FREE(fdo->atlines[l]);
      fdo->num_atlines[l]=0; } }

  Free_Data_All(fdo);
}

void FDObj_Update_ForceView(FDO *fdo, FILE *fptr,int view)
{
  int l;
  char tmp[512];

	if(fdo->nfds>0) {
  	FDObj_UpdateCore(fdo);
		SetFlags(fdo);
		SensitizeButtons(fdo,fdo->fds[fdo->nfds-1]); }

  for(l=0; l<fdo->nfds; l++) {
    if(fdo->fds[l]==NULL) continue;
    if(fdo->fds[l]->trashed) {
      strcpy(tmp,fdo->fds[l]->filename);
      FD_free(fdo->fds[l]);
      fdo->fds[l]=FD_read(tmp,0); }

    doFDOdoplotMethod((void*)fdo,fdo->fds[l],view,l,fptr); }
}

void FDObj_Update(FDO *fdo, FILE *fptr)
{
	int l;
	char tmp[512];

	if((fdo->nfds>0) && (fdo->fds[fdo->nfds-1]==NULL))
		UnKeep(fdo);

	if(fdo->nfds>0) {
		FDObj_UpdateCore(fdo);
		SetFlags(fdo);
		SensitizeButtons(fdo,fdo->fds[fdo->nfds-1]); }

	for(l=0; l<fdo->nfds; l++) {
		if(fdo->fds[l]==NULL) continue;
		if(fdo->fds[l]->trashed) {
			strcpy(tmp,fdo->fds[l]->filename);
			FD_free(fdo->fds[l]);
			fdo->fds[l]=FD_read(tmp,0); 
		}

		doFDOplotMethod((void*)fdo,fdo->fds[l],&(fdo->view),l,fptr); 
	}
}

FDO *FDObj_File_New(char *fname, Widget parent, FDObj_ViewType *view)
{
	FILEDATA *fd;
	FDO *fdo;
	Widget psh, form;
	FDObj_ViewType def_view = {0,0};

	if (parent == NULL) {
		assert((parent = psh = top_new(TopLevel, "FDO"))!=NULL);
	} else
		psh = NULL;

	if ((fd = FD_read(fname,0)) != NULL) {
		fdo = FDObj_New(fd, parent, (view==NULL) ? (&def_view) : view, 0);
		FDObj_Update(fdo,NULL); 
	} else
		fdo = NULL;

	if (psh) {
		if (fdo) {
			popup(psh);
			XtAddCallback(fdo->form, XtNdestroyCallback, ClosePShellCB, psh); }
		else {
			XtDestroyWidget(psh); 
		} 
	}

	return(fdo);
}

void FDObj_Add(FDO *fdo, FILEDATA *fd)
{
	if(((fdo->nfds==1) && (!fdo->destroy_old)) ||
			(fdo->nfds>1)) {
		if(!SameModule(fd,fdo->fds[0])) {
			beep(0);
			return; } }
	if(fdo->nfds>0) {
		if(!fdo->destroy_old)
			KeepCB(NULL, fdo, NULL);
		if(fdo->fds[fdo->nfds-1]!=NULL)
			FD_free(fdo->fds[fdo->nfds-1]); }
	else
		++fdo->nfds;

	fdo->fds[fdo->nfds-1] = fd;
	fdo->epoch = FD_GF(fdo->fds[fdo->nfds-1], "epoch");
	fdo->dur = FD_GF(fdo->fds[fdo->nfds-1], "dur");
	fdo->delay = FD_GF(fdo->fds[fdo->nfds-1], "delay");
	fdo->sum_code[fdo->nfds-1] = MU_CODE;
	UnZoomCB(NULL,fdo,0);

  if(!fdo->no_X) {
		if (fdo->point_tag != NULL) {
			XtDestroyWidget(fdo->point_tag);
			fdo->point_tag = NULL;
 			XtVaSetValues(fdo->clickinfo, XtNlabel, "", NULL); } }

	FDObj_Update(fdo,NULL);
}

static void Duplicate(Widget g, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo_old = (FDO *) fdoptr;
  FDO *fdo_new;
  Widget psh;
  int i;

  assert((psh = top_new(TopLevel, "FDO"))!=NULL);
  fdo_new=FDObj_New(NULL,psh,&(fdo_old->view),0);

  FREE(fdo_new->fds);
  fdo_new->fds = fdo_old->fds;
  fdo_new->nfds = fdo_old->nfds;
  assert((fdo_old->fds = (FILEDATA**)malloc(fdo_new->nfds*
        sizeof(FILEDATA*)))!=NULL);
  for(i=0; i<fdo_new->nfds; i++)
		if(fdo_new->fds[i]!=NULL)
  	  fdo_old->fds[i] = FD_read(fdo_new->fds[i]->filename,0);

  FREE(fdo_new->atlines);
  fdo_new->atlines = fdo_old->atlines;
  FREE(fdo_new->num_atlines);
  fdo_new->num_atlines = fdo_old->num_atlines;
  assert((fdo_old->atlines = (Widget**)malloc(fdo_new->nfds*
        sizeof(Widget*)))!=NULL);
  assert((fdo_old->num_atlines = (int*)malloc(fdo_new->nfds*
        sizeof(int)))!=NULL);
  for(i=0; i<fdo_new->nfds; i++) {
    fdo_old->atlines[i]=NULL;
    fdo_old->num_atlines[i]=0; }

  fdo_new->ndata_all = fdo_old->ndata_all;
  fdo_old->ndata_all = NULL;
  fdo_new->xdata_all = fdo_old->xdata_all;
  fdo_old->xdata_all = NULL;
  fdo_new->ydata_all = fdo_old->ydata_all;
  fdo_old->ydata_all = NULL;
  fdo_new->meta_num_all = fdo_old->meta_num_all;
  fdo_old->meta_num_all = 0;

  FREE(fdo_new->sum_code);
  fdo_new->sum_code = fdo_old->sum_code;
  assert((fdo_old->sum_code = (int*)malloc(fdo_new->nfds*sizeof(int)))!=NULL);
  for(i=0; i<fdo_new->nfds; i++)
    fdo_old->sum_code[i] = fdo_new->sum_code[i];

	fdo_new->epoch = fdo_old->epoch;
	fdo_new->dur = fdo_old->dur;
	fdo_new->delay = fdo_old->delay;

  fdo_new->baw = fdo_old->baw;  
  fdo_new->no_grid = fdo_old->no_grid;  
  fdo_new->legend = fdo_old->legend;  
  fdo_new->log_x = fdo_old->log_x; 
  fdo_new->log_y = fdo_old->log_y; 
  fdo_new->no_errs = fdo_old->no_errs; 

  popup(psh);
  XtAddCallback(fdo_new->form, XtNdestroyCallback, ClosePShellCB, psh);
  FDObj_Update(fdo_old,NULL);
  FDObj_Update(fdo_new,NULL);
}

void FDObj_AddLine(FDO *fdo, int l, float *xdata, float *ydata, float *zdata,
			int ndata, AtDataType datatype, WidgetClass class, XtArgVal type,
			XtArgVal mark, XtArgVal color, XtArgVal legend)
{
  int curr;
	double guage = 60.0, offset = 1.0;

  if(fdo->num_atlines[l]==0)
    assert((fdo->atlines[l] = (Widget*)malloc(sizeof(Widget)))!=NULL);
  else
    assert((fdo->atlines[l] = (Widget*)realloc(fdo->atlines[l],
          (fdo->num_atlines[l]+1)*sizeof(Widget)))!=NULL);
  curr = (fdo->num_atlines[l])++;
  fdo->atlines[l][curr] = XtVaCreateWidget("line", class,
        fdo->graph, NULL);
  AtQuadLinePlotAttachData(fdo->atlines[l][curr],
        xdata, datatype, sizeof(float), ydata, datatype, sizeof(float),
        zdata, datatype, sizeof(float), NULL, datatype, sizeof(float), 0,ndata);
  if(fdo->baw) {
    XtVaSetValues(fdo->atlines[l][curr], XtNplotType, type,
            XtNplotMark, FDObj_Mark[l%FDObj_numMark],
            XtNmarkGuage, &guage, NULL); }
	else {
		if(class==atQuadLinePlotWidgetClass)
      XtVaSetValues(fdo->atlines[l][curr], XtNplotType, type,
            XtNplotColor, color, XtNmarkGuage, &guage,
            XtNplotMark, mark, NULL);
		else if(class==atQuadBarPlotWidgetClass)
      XtVaSetValues(fdo->atlines[l][curr],
            XtNbarOffset, &offset, XtNfillColor, color, NULL); }

  if(legend)
    XtVaSetValues(fdo->atlines[l][curr], XtNlegendName, legend, NULL);
}

void FDObj_Close(FDO *fdo)
{
  int i,j;

  if(fdo->nfds > 0) {
    for(i=0; i<fdo->nfds; i++) {
      FD_free(fdo->fds[i]);
      if(!fdo->no_X) {
        for(j=0; j<fdo->num_atlines[i]; j++)
          XtDestroyWidget(fdo->atlines[i][j]);
        FREE(fdo->atlines[i]); } } }
  FREE(fdo->fds);
  if(!fdo->no_X) {
    FREE(fdo->atlines);
    FREE(fdo->num_atlines); }
  fdo->nfds=0;
  Free_Data_All(fdo);

  if(!fdo->no_X)
    XtDestroyWidget(fdo->form);

  FREE(fdo->flag_menu);
  FREE(fdo);
}

static void CloseCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
  parms_close_hook();
	FDObj_Close(fdo);
}

static void HelpCB(Widget w, XtPointer foo, XtPointer call_data)
{
  char tmp[100],*file;

  sprintf(tmp,"$(XDPHYSDIR)/doc/xdview.doc");
  file = envexpand(tmp);
  if (!probefile(file)) {
    alert("Can't find help file.");
    return; }
  help(file,"xdview");
}

static void SetFlags(FDO *fdo)
{
  if(fdo->no_X) return;

  XtVaSetValues(fdo->graph, XtNshowLegend, fdo->legend, NULL);

	if(fdo->log_x)
    XtVaSetValues(fdo->xaxis, XtNroundEndpoints, False,
          XtNaxisTransform, AtTransformLOGARITHMIC, NULL); 
	else
    XtVaSetValues(fdo->xaxis, XtNroundEndpoints, False,
          XtNaxisTransform, AtTransformLINEAR, NULL); 
	if(fdo->log_y)
    XtVaSetValues(fdo->yaxis, XtNroundEndpoints, False,
          XtNaxisTransform, AtTransformLOGARITHMIC, NULL); 
	else
    XtVaSetValues(fdo->yaxis, XtNroundEndpoints, False,
          XtNaxisTransform, AtTransformLINEAR, NULL); 

	if(fdo->no_grid) {
    XtVaSetValues(fdo->xaxis, /*XtNdrawOrigin, False,*/ XtNdrawGrid, False, NULL);
    XtVaSetValues(fdo->yaxis, /*XtNdrawOrigin, False,*/ XtNdrawGrid, False, NULL); }
	else {
    XtVaSetValues(fdo->xaxis, /*XtNdrawOrigin, True,*/ XtNdrawGrid, True, NULL);
    XtVaSetValues(fdo->yaxis, /*XtNdrawOrigin, True,*/ XtNdrawGrid, True, NULL);  }
}

static void UpdateCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;

  FDObj_Update(fdo,NULL);
}

static void DestTogCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;

  fdo->destroy_old = !fdo->destroy_old;
  if (fdo->destroy_old) {
    XtVaSetValues(w, XtNlabel, "@New", NULL);
    XtVaSetValues(fdo->graph, XtNshowLegend, False, NULL);
  } else {
    XtVaSetValues(w, XtNlabel, "Over", NULL);
    XtVaSetValues(fdo->graph, XtNshowLegend, True, NULL);
  }
}

static void KeepCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;

  if(fdo->fds[fdo->nfds-1]!=NULL) {
    fdo->nfds++;
    assert((fdo->fds = (FILEDATA**)realloc(fdo->fds,
          sizeof(FILEDATA*)*(fdo->nfds)))!=NULL);
    if(!fdo->no_X) {
      assert((fdo->atlines = (Widget**)realloc(fdo->atlines,
            sizeof(Widget*)*(fdo->nfds)))!=NULL);
      assert((fdo->num_atlines = (int*)realloc(fdo->num_atlines,
            sizeof(int)*(fdo->nfds)))!=NULL);
      fdo->atlines[fdo->nfds-1]=NULL;
      fdo->num_atlines[fdo->nfds-1]=0; }
    assert((fdo->sum_code = (int*)realloc(fdo->sum_code,
          sizeof(int)*(fdo->nfds)))!=NULL);
    fdo->fds[fdo->nfds-1]=NULL; }
}

static void UnKeep(FDO *fdo)
{
  fdo->nfds--;
  assert((fdo->fds = (FILEDATA**)realloc(fdo->fds,
        sizeof(FILEDATA*)*(fdo->nfds)))!=NULL);
  assert((fdo->atlines = (Widget**)realloc(fdo->atlines,
        sizeof(Widget*)*(fdo->nfds)))!=NULL);
  assert((fdo->num_atlines = (int*)realloc(fdo->num_atlines,
        sizeof(int)*(fdo->nfds)))!=NULL);
  assert((fdo->sum_code = (int*)realloc(fdo->sum_code,
        sizeof(int)*(fdo->nfds)))!=NULL);
}

static void ClearCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  int i,j;

  for(i=0; i<fdo->nfds-1; i++) {
    FD_free(fdo->fds[i]);
    for(j=0; j<fdo->num_atlines[i]; j++)
      XtDestroyWidget(fdo->atlines[i][j]); }
  fdo->fds[0] = fdo->fds[fdo->nfds-1];
  fdo->atlines[0] = fdo->atlines[fdo->nfds-1];
  fdo->num_atlines[0] = fdo->num_atlines[fdo->nfds-1];
  fdo->nfds=1;

  assert((fdo->fds = (FILEDATA**)realloc(fdo->fds,1*sizeof(FILEDATA*)))!=NULL);
  assert((fdo->atlines = (Widget**)realloc(fdo->atlines,
        1*sizeof(Widget*)))!=NULL);
  assert((fdo->num_atlines = (int*)realloc(fdo->num_atlines,
        1*sizeof(int)))!=NULL);

  fdo->selected = NULL;
  if (fdo->point_tag != NULL) {
    XtDestroyWidget(fdo->point_tag);
    fdo->point_tag = NULL;
    XtVaSetValues(fdo->clickinfo, XtNlabel, "", NULL); }

  FDObj_Update(fdo,NULL);
}

static void RightviewCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  fdo->view.lr++;
  if (fdo->point_tag != NULL)
    XtVaSetValues(fdo->point_tag, XtNdisplayed, False, NULL);
  UnZoomCB(NULL,fdo,0);
  fdo->selected = NULL;
  FDObj_Update(fdo,NULL);
}

static void LeftviewCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  fdo->view.lr--;
  if (fdo->point_tag != NULL)
    XtVaSetValues(fdo->point_tag, XtNdisplayed, False, NULL);
  UnZoomCB(NULL,fdo,0);
  fdo->selected = NULL;
  FDObj_Update(fdo,NULL);
}

static void UpviewCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  fdo->view.ud--;
  if (fdo->point_tag != NULL)
    XtVaSetValues(fdo->point_tag, XtNdisplayed, False, NULL);
  UnZoomCB(NULL,fdo,0);
  fdo->selected = NULL;
  FDObj_Update(fdo,NULL);
}

static void DownviewCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  fdo->view.ud++;
  if (fdo->point_tag != NULL)
    XtVaSetValues(fdo->point_tag, XtNdisplayed, False, NULL);
  UnZoomCB(NULL,fdo,0);
  fdo->selected = NULL;
  FDObj_Update(fdo,NULL);
}

static void ZoomCB(Widget w, XtPointer fdoptr, XtPointer atrectptr)
{
  FDO *fdo = (FDO *) fdoptr;
  AtRectangleCallbackData *atrect = (AtRectangleCallbackData *) atrectptr;

  if (atrect->x11 < atrect->x12) {
    XtVaSetValues(fdo->xaxis, XtNmin, &atrect->x11, XtNmax, &atrect->x12, NULL);
  	XtVaSetValues(fdo->xaxis, XtNautoScale, False, NULL); }
 
  else {
    XtVaSetValues(fdo->xaxis, XtNmax, &atrect->x11, XtNmin, &atrect->x12, NULL);
  	XtVaSetValues(fdo->xaxis, XtNautoScale, False, NULL); }

  if (atrect->y11 < atrect->y12) {
    XtVaSetValues(fdo->yaxis, XtNmin, &atrect->y11, XtNmax, &atrect->y12, NULL);
  	XtVaSetValues(fdo->yaxis, XtNautoScale, False, NULL); }
  else {
    XtVaSetValues(fdo->yaxis, XtNmax, &atrect->y11, XtNmin, &atrect->y12, NULL);
  	XtVaSetValues(fdo->yaxis, XtNautoScale, False, NULL); }
  fdo->just_zoomed = 1;
}

static void UnZoomCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;

  if(fdo->no_X) return;

 	XtVaSetValues(fdo->xaxis, XtNautoScale, True, NULL);
 	XtVaSetValues(fdo->yaxis, XtNautoScale, True, NULL);
}


/* this distance routine stuff is an almost direct copy of the
** copy in atgraph.c -- just duplicated to keep the possiblity
** of making this module standalone..
*/

#define DIST(i,j) \
    ((fdo->xdata_all[i][j]-atpoint->x1) * (fdo->xdata_all[i][j]-atpoint->x1) + \
     (fdo->ydata_all[i][j]-atpoint->y1) * (fdo->ydata_all[i][j]-atpoint->y1))

static float adjed_dist(float x, float y, AtPointCallbackData *atpoint,
      double xmin, double xmax, double ymin, double ymax)
{
  float px = atpoint->x1;
  float py = atpoint->y1;

  x = (x - xmin) / (xmax - xmin);
  y = (y - ymin) / (ymax - ymin);
  px = (atpoint->x1 - xmin) / (xmax - xmin);
  py = (atpoint->y1 - ymin) / (ymax - ymin);
  return((x - px) * (x - px) + (y - py) * (y - py));
}

static void ClickCB(Widget w, XtPointer fdoptr, XtPointer atpointptr)
{
  FDO *fdo = (FDO *) fdoptr;
  AtPointCallbackData *atpoint = (AtPointCallbackData *) atpointptr;

  int i, j, nth, mth, view_adj;
  double min, f, x, y;
  char buf[100];
  double xmin, xmax, ymin, ymax;

  if (fdo->just_zoomed)
    fdo->just_zoomed = 0;
  else {
    if (fdo->xaxis && fdo->yaxis) {
      view_adj = 1;
      XtVaGetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax, NULL);
      XtVaGetValues(fdo->yaxis, XtNmin, &ymin, XtNmax, &ymax, NULL); }
    else
      view_adj = 0;
    if (fdo->meta_num_all > 0) {
      nth = mth = 0;
      if (view_adj) {
				min = adjed_dist(fdo->xdata_all[0][0], fdo->ydata_all[0][0],
							atpoint, xmin, xmax, ymin, ymax); }
      else
				min = DIST(0,0);
			for (i = 0; i < fdo->meta_num_all; i++) {
				for (j = (i==0) ? 1 : 0; j < fdo->ndata_all[i]; j++) {
					if (view_adj) {
						f = adjed_dist(fdo->xdata_all[i][j], fdo->ydata_all[i][j],
									atpoint, xmin, xmax, ymin, ymax); }
					else {
						f = DIST(0,0); }
				if (f < min) {
					min = f;
					nth = i;
					mth = j; } } }
      if (fdo->to_tty) {
		printf("%g\t%g\n", fdo->xdata_all[nth][mth], fdo->ydata_all[nth][mth]);
		fflush(stdout); 
	  }

      sprintf(buf, "(%.2f, %.2f)", x = fdo->xdata_all[nth][mth],
            y = fdo->ydata_all[nth][mth]);
      XtVaSetValues(fdo->clickinfo,
		    XtNlabel, buf, NULL);

      if (fdo->point_tag == NULL)
				fdo->point_tag =
							XtVaCreateWidget("near", atTextPlotWidgetClass, fdo->graph, NULL);
			XtVaSetValues(fdo->point_tag, XtNlabel, "V", XtNdisplayed, True,
						XtNjustify, AtTextJUSTIFY_CENTER, XtNfontSize, AtFontBIGGEST,
						XtNfloatingX, &x, XtNfloatingY, &y, NULL); } }
}

static void SelectLineCB(Widget w, XtPointer fdoptr, XtPointer atselectptr)
{
  FDO *fdo = (FDO *) fdoptr;
  AtSelectCallbackData *atselect = (AtSelectCallbackData *) atselectptr;

  if (fdo->selected) {
    XtVaSetValues(fdo->selected, XtNlineWidth, 0, NULL);
  }
  fdo->selected = atselect->widget;
  if (atselect->reason == AtSelectSELECTED && fdo->selected) {
    XtVaSetValues(fdo->selected, XtNlineWidth, 3, NULL);
  }
}

static void RereadDirCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FileNominatorSetDirectory(w, NULL);

}


#endif /* SANAL */

void FDObj_CommentsCB(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
  char *buf,*buf2,*tmp;
  int i,size,count=0,add;

  size=1024;
  assert((buf2=(char*)calloc(size,sizeof(char)))!=NULL);

  if(fdo->nfds>0) {
    tmp = "========================================\n";
    strcpy(buf2, tmp);  count += strlen(tmp); }

  for(i=0; i<fdo->nfds; i++) {
    if(fdo->fds[i]==NULL) continue;

    buf = vars2ascii(fdo->fds[i]->params);

    add = strlen(buf)+strlen(fdo->fds[i]->comments);
    if((count+add) > (size-512)) {
      assert((buf2=(char*)realloc(buf2,(count+add+1024)*sizeof(char)))!=NULL);
      size = count+add+1024; }

    tmp = fdo->fds[i]->comments;
    strcat(buf2, tmp);  count += strlen(tmp);
    tmp = "----------------------------------------\n";
    strcat(buf2, tmp);  count += strlen(tmp);
    strcat(buf2, buf);  count += strlen(buf);

    tmp = "========================================\n";
    strcat(buf2, tmp);  count += strlen(tmp);

    free(buf); 
  }

	if(!fdo->to_tty)
		pop_text("Comments", buf2, strlen(buf2)+1, False);
	else
		printf("%s",buf2);
}

void FDObj_PrintRasters(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
  spike_t **rasters;
  int nrasters,i,j;
  int epoch;
  int *pres_order,*inv_pres_order;
  int *depints;

  assert(fdo->nfds==1);

  rasters = FD_get_rastlist(fdo->fds[0]);
  nrasters = FD_get_nrasters(fdo->fds[0]);
  epoch = FD_GI(fdo->fds[0],"Epoch");
  depints = FD1_get_depints(fdo->fds[0]);

  pres_order = FD_get_pres_order(fdo->fds[0]);
  FD_inverse_pres_order(pres_order,&inv_pres_order,nrasters);

  printf("%d %d 0 0\n",nrasters,epoch*10000);

  for(i=0; i<nrasters; i++)
    if (N_SPIKES(rasters[inv_pres_order[i]]) == 0) {
		/* Print something to show that we really did get no spikes
		 * for this trial */
	    printf("%d -1 %d %d\n",i, 
            SPIKE_CHAN(rasters[inv_pres_order[i]],j),
			depints[i]);
	} else {
		for(j=0; j<N_SPIKES(rasters[inv_pres_order[i]]); j++)
			printf("%d %d %d %d\n",i, 
				SPIKE_TIME(rasters[inv_pres_order[i]],j),
				SPIKE_CHAN(rasters[inv_pres_order[i]],j),
				depints[i]);
	}
}

static void DumpPrinter(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;

  AtPlotterGeneratePostscript(NULL, NULL, fdo->graph, 0.8, 0);
	
/*
  system("lpr /tmp/xdview.ps");
  unlink("/tmp/xdview.ps");
*/
}

static void DumpPS(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  char *p;

  if ((p = fileBox(".ps", "w", "Save PS to:")) == NULL)
    return;
  if(probefile(p))
    if(pop_box("File exists!","Overwrite","Cancel",NULL)==2)
      return;

  AtPlotterGeneratePostscript(p, "w", fdo->graph, 0.8, 0);
}

static void DumpFIG(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  char *p;

  if ((p = fileBox(".fig", "w", "Save FIG to:")) == NULL)
    return;
  if(probefile(p))
    if(pop_box("File exists!","Overwrite","Cancel",NULL)==2)
      return;
  XPS_SetFileOutput(1);
  XPS_SetFilename(p);
  free(p);
  XtUnmapWidget(fdo->graph);
  XPS_PrepareFIG(XtDisplay(fdo->graph), XtWindow(fdo->graph));
  XtMapWidget(fdo->graph);
  dloop_empty();
  XPS_FinishFIG();
  XtUnmapWidget(fdo->graph);
  XtMapWidget(fdo->graph);
}

static void DumpText(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
	char *p;
	FILE *fptr;

	if ((p = fileBox(".txt", "w", "Save TEXT to:")) == NULL)
		return;
	if (probefile(p))
		if (pop_box("File exists!", "Overwrite", "Cancel", NULL) == 2)
			return;
	assert((fptr = fopen(p, "w")) != NULL);

	if (fdo->no_text_header != 1) {
		fprintf(fptr, ";; xdphys txt ver %s\n", XDPHYS_VERSION);
	}
	FDObj_Update(fdo, fptr);
	fclose(fptr);
}

static void DoXDprint(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  char p[MAXPATHLEN + 100];
  
  if (fdo->fds != NULL) {
	  if (fdo->fds[0]->filename != NULL) {
		  sprintf(p,"exec xdprint %s | lpr", fdo->fds[0]->filename);
		  system_noblock(p);
	  }
  }
}

static void DoXDprintRayleigh(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;
  char p[MAXPATHLEN + 100];
  
  if (fdo->fds != NULL) {
	  if (fdo->fds[0]->filename != NULL) {
		  sprintf(p,"exec xdprint -r %s | lpr", fdo->fds[0]->filename);
	  system_noblock(p);
	  }
  }
}

static void DumpSTDOUT(Widget w, XtPointer fdoptr, XtPointer call_data)
{
  FDO *fdo = (FDO *) fdoptr;

	printf(";; xdphys txt ver %s\n",XDPHYS_VERSION);
  FDObj_Update(fdo,stdout);
}


static int SameModule(FILEDATA *one, FILEDATA*two)
{
  if(!strcmp(one->modname,two->modname))
    return(1);
	if(!strcmp(one->modname,"txt") || !strcmp(two->modname,"txt"))
    return(1);
  return(0);
}

static char *compute_pat(FDO *fdo, int l)
{
  static char pat[10];

  if (fdo->sum_code[l] == MU_CODE) {
    sprintf(pat,"MU"); }
  else if (fdo->sum_code[l] == TTL_CODE) {
    sprintf(pat,"TTL"); }
  else if (fdo->sum_code[l] == OUTLIER_CODE) {
    sprintf(pat,"OUT"); }
  else {
    sprintf(pat, "#%02d", fdo->sum_code[l]); }

  return(pat);
}

static int check_type(char *mod)
{
	if(!strcmp(mod,"ana") ||
	      !strcmp(mod,"cal") || !strcmp(mod,"fr") ||
	      !strcmp(mod,"bw") || !strcmp(mod,"int"))
		return(0);

  return(1);
}

XtArgVal FDObj_Legend(FDO *fdo, int l)
{
	static char ret_val[128];
	char *pat;

	sprintf(ret_val, "%s", shortname(fdo->fds[l]->filename));
	if(check_type(fdo->fds[l]->modname) &&
        !strcmp(FD_GV(fdo->fds[l],"detect.mode"),"ss")) {
		pat=compute_pat(fdo,l);
		sprintf(ret_val+strlen(ret_val), "/%s", pat); }
	return((XtArgVal) ret_val);
}

char *FDObj_Title(FDO *fdo, char *ti, int l)
{
	static char tbuf[100];
	char *pat;

	sprintf(tbuf, "@b(%s) - %s",
			fdo->fds[l] ? shortname(fdo->fds[l]->filename) : "???", ti);
	if(check_type(fdo->fds[l]->modname) &&
	      !strcmp(FD_GV(fdo->fds[l],"detect.mode"),"ss")) {
		pat=compute_pat(fdo,l);
		sprintf(tbuf+strlen(tbuf), " - @b(%s)@-(%d)",
				pat, fdo->fds[l] ? (-1+fdo->fds[l]->channels[0]) : 0); }
	return(tbuf);
}

void FDObj_Add_Data_All(FDO *fdo, float *xdata, float *ydata, int ndata)
{
  if(fdo->meta_num_all==0) {
    assert((fdo->ndata_all = (int*)malloc(1*sizeof(int)))!=NULL);
    assert((fdo->xdata_all = (float**)malloc(1*sizeof(float*)))!=NULL);
    assert((fdo->ydata_all = (float**)malloc(1*sizeof(float*)))!=NULL); }
  else {
    assert((fdo->ndata_all = (int*)realloc(fdo->ndata_all,
          (1+fdo->meta_num_all)*sizeof(int)))!=NULL);
    assert((fdo->xdata_all = (float**)realloc(fdo->xdata_all,
          (1+fdo->meta_num_all)*sizeof(float*)))!=NULL);
    assert((fdo->ydata_all = (float**)realloc(fdo->ydata_all,
          (1+fdo->meta_num_all)*sizeof(float*)))!=NULL); }
  fdo->meta_num_all++;
  fdo->ndata_all[fdo->meta_num_all-1] = ndata;
  fdo->xdata_all[fdo->meta_num_all-1] = xdata;
  fdo->ydata_all[fdo->meta_num_all-1] = ydata;
}

static void SelectFileCB(Widget w, XtPointer fdoptr, XtPointer nameptr)
{
  FDO *fdo = (FDO *) fdoptr;

  FileNominatorStruct *name = (FileNominatorStruct *) nameptr;
  FILEDATA *fd;
  char *fname;

  assert((fname = (char*)malloc(strlen(name->directoryPart) +
        strlen(name->filenamePart) + 1))!=NULL);
  strcpy(fname, name->directoryPart);
  strcat(fname, name->filenamePart);
  if (!probefile(fname) && is_pattern(name->filenamePart)) {
    XtVaSetValues(w, XtNpattern, name->filenamePart, NULL);
    FileNominatorSetDirectory(w, NULL); }
  else if((fd = FD_read(fname,0)) != NULL)
		FDObj_Add(fdo, fd);
  else
    beep(0);

  free(fname);
}

static int SupportedModule(FILEDATA *fd)
{
  if(!strcmp(fd->modname,"txt"))
    return(1);
  if(!strcmp(fd->progname,"dowl") || !strcmp(fd->progname,"xdphys")) {
    if(!strcmp(fd->modname,"abi") || !strcmp(fd->modname,"itd") ||
       !strcmp(fd->modname,"iid") || !strcmp(fd->modname,"bf") ||
       !strcmp(fd->modname,"freq") || !strcmp(fd->modname,"bja")) {
      return(1); }
    else {
      return(0); } }
  else if(!strcmp(fd->progname,"calib") || !strcmp(fd->progname,"xcalibur")) {
    if(!strcmp(fd->modname,"int")) {
      return(1); }
    else {
      return(0); } }

  return(0);
}

static void SensitizeButtons(FDO *fdo, FILEDATA *fd)
{
  if(fdo->no_X) return;

  if(!SupportedModule(fd)) {
    XtSetSensitive(fdo->over, False);
    XtSetSensitive(fdo->keep, False); }
  else {
    XtSetSensitive(fdo->over, True);
    XtSetSensitive(fdo->keep, True); }
  XtSetSensitive(fdo->up, False);
  XtSetSensitive(fdo->down, False);
  XtSetSensitive(fdo->fit, False);
  XtSetSensitive(fdo->vsm, False);
}

static void Free_Data_All(FDO *fdo)
{
  int i;

  for(i=0; i<fdo->meta_num_all; i++) {
		FREE(fdo->xdata_all[i]);
		FREE(fdo->ydata_all[i]); }
 	FREE(fdo->ndata_all);
  fdo->ndata_all=NULL;
  FREE(fdo->xdata_all);
  fdo->xdata_all=NULL;
  FREE(fdo->ydata_all);
  fdo->ydata_all=NULL;
  fdo->meta_num_all = 0;
}

void CombineDataFN(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
  FDO *fdobj = (FDO*) fdobjptr;
  char *p1,*p2;
  FILE *fptr;

  p1 = pop_dialog("shift?", "");

  if ((p2 = fileBox(".txt", "w", "Save CombineData to:")) == NULL)
    return;
  if(probefile(p2))
    if(pop_box("File exists!","Overwrite","Cancel",NULL)==2)
      return;
  assert((fptr=fopen(p2,"w"))!=NULL);
  fprintf(fptr,";; xdphys txt ver %s\n",XDPHYS_VERSION);

  FD1_combine_data(fdobj,p1,fptr);
  fclose(fptr);

  KeepCB(NULL,fdobjptr,NULL);
  FDObj_Add(fdobj,FD_read(p2,0));
}

#if(0)
void LinearityStatsFN(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
  FDO *fdobj = (FDO*) fdobjptr;
  char *p1,*p2;
  FILE *fptr;

  p1 = pop_dialog("shift?", "");

  if ((p2 = fileBox(".txt", "w", "Save LinearityData to:")) == NULL)
    return;
  if(probefile(p2))
    if(pop_box("File exists!","Overwrite","Cancel",NULL)==2)
      return;
  assert((fptr=fopen(p2,"w"))!=NULL);
  fprintf(fptr,";; xdphys txt ver %s\n",XDPHYS_VERSION);

  FD1_linearity_stats(fdobj,p1,fptr);
  fclose(fptr);

  KeepCB(NULL,fdobjptr,NULL);
  FDObj_Add(fdobj,FD_read(p2,0));
}
#endif
