/*
 * ** This file is part of XDowl
 * ** Copyright (c) 1994 Jamie Mazer
 * ** California Institute of Technology
 * ** <mazer@asterix.cns.caltech.edu>
 */

/******************************************************************
**  RCSID: $Id: fd1.c,v 1.39 2001/03/27 06:59:24 cmalek Exp $
** Program: dowl
**  Module: plotfile.c
**  Author: mazer
** Descrip: high level interface to graph.c -- reads dowl datafiles
**
** Revision History (most recent last)
**
** 97.2 bjarthur
**  created from fd.c
**
** 97.4 bjarthur
**   added FD1_perhist1FN(), FD1_perhist2FN()
**
** 98.2 bjarthur
**  moved struct RawData definition into fd1.h to facilitate less duplication
**  of FD1_*FN() routines.  now mod's which have the same RawData like
**  bam,beats,cf, etc... can use some of the same plot FNs, like psth, isih,
**  and perhist.
**
** 98.5 bjarthur
**  added similarity_stats and isih_stats.
**  added nspikes in SumData struct, did NOT change fd2.c, or the bam,
**    beats, cf, or rover modules accordingly
**  modified _summarize() to realloc arrays to save space
**
** 99.6 bjarthur
**  removed linearity_stats
**
*******************************************************************/

#include "xdphyslib.h"

/* ------------------------------------------------------------------------
 * Global Variables
 * ------------------------------------------------------------------------ */
char dis[50][24];								/* used in perhist_stats */
float mps[50];
float vss[50];
float rays[50];

/* ------------------------------------------------------------------------
 * Typedefs
 * ------------------------------------------------------------------------ */
typedef struct {
	int unit;					/* unit id or "mask" */
	int ndata;					/* number of summary data points */
	char **depstrs;				/* depvar values (int) for matching means */
	/* note: the depstrs[n] here are actually owned
	 * by deplines, and should NOT be free'd*/
	float *depvals;				/* depvar values (int) for matching means */
	float *means;				/* mean responses (nrasters/reps) */
	float *stderrs;				/* std error of responses (nrasters/reps) */
	float *stddevs;				/* std dev of responses (nrasters/reps) */
	int *n;						/* # of data points means, etc are based on */
	int **nspikes;				/* nspikes[depval_idx][rep] = #spikes */
} FD1_SumData;

#define FD1_SUMDATA(data) ((FD1_SumData*)(data->sumdata))

/* ------------------------------------------------------------------------
 * Prototypes
 * ------------------------------------------------------------------------ */
static void FD1_sort(FILEDATA *);
static void FD1_summarize(FILEDATA *, int);
static float *copyVec(float *, int, float);
static void FD1_freeSumData(FILEDATA *);
static void compute_isih_stats(int, int **, int, float, int *, int *);

/* ------------------------------------------------------------------------
 * FUNCTIONS
 * ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------
 * FD1_curveFN
 * ------------------------------------------------------------------------ */
void FD1_curveFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	int i, j;
	char *ti;
	float f1;
	char *xlabel, *ylabel;
	float *px, *py, *pz, *sx, *sy, *sz;
	float *rx, *ry;
	int pn, sn, rn;
	int **nspikes, *n, ndata, npoints;
	char line[1000];
	char column[20];

	FD1_summarize(fd, fdo->sum_code[l]);
	if (fdo->normalize == 1) {
		ylabel = "% max response";
		f1 = FD1_SUMDATA(fd)->means[0];
		for (i = 1; i < FD1_SUMDATA(fd)->ndata; i++) {
			if (FD1_SUMDATA(fd)->means[i] > f1)
				f1 = FD1_SUMDATA(fd)->means[i];
		}
		f1 /= 100.0;								/* convert to percent */
	}
	else {
		ylabel = "nspikes";
		f1 = 1.0;										/* no scale factor.. */
	}
	xlabel = FD_GV(fd, "depvar");
	ti = "curve";

	FD_plotter_copy(FD1_SUMDATA(fd)->depvals, FD1_SUMDATA(fd)->means,
				!(fdo->no_errs) ? FD1_SUMDATA(fd)->stderrs : NULL, FD1_SUMDATA(fd)->ndata, 1,
				&px, &py, &pz, &pn, &sx, &sy, &sz, &sn);

	for (i = 0; i < pn; i++)
		py[i] /= f1;
	for (i = 0; i < sn; i++)
		sy[i] /= f1;

	if (fdo->show_raw_points == 1) {
		ndata = FD1_get_ndata(fd);
		n = FD1_get_n(fd);
		nspikes = FD1_get_nspikes(fd);
		rn = (FD1_RAWDATA(fd)->nrasters);
		assert((rx = (float *) calloc(rn, sizeof(float))) != NULL);
		assert((ry = (float *) calloc(rn, sizeof(float))) != NULL);
		npoints = 0;
		for (i = 1; i < ndata; i++) {
			for (j = 0; j < n[i]; j++) {
				rx[npoints] = FD1_SUMDATA(fd)->depvals[i];
				ry[npoints] = nspikes[i][j];
				npoints++;
			}
		}
		rn = npoints;
		assert((rx = (float *) realloc(rx, rn * sizeof(float))) != NULL);
		assert((ry = (float *) realloc(ry, rn * sizeof(float))) != NULL);
	}
	if (fptr != NULL) {
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "COMMENTS\n");
			fprintf(fptr, "%%%s, %s view\n",
						strrchr(fd->filename, '/') == NULL ? fd->filename :
						1 + (char *) strrchr(fd->filename, '/'), ti);
			fprintf(fptr, "%%col 1: %s\n", xlabel);
			fprintf(fptr, "%%col 2: %s\n", ylabel);
			if (!(fdo->no_errs))
				fprintf(fptr, "%%col 3: std error\n");
			fprintf(fptr, "%%col last: n\n");
			fprintf(fptr, "END_COMMENTS\n");
			fprintf(fptr, "PARAMS\n");
			fprintf(fptr, "Reps=%d\n", FD_GI(fd, "Reps"));
			fprintf(fptr, "spont=%e\t%e\t%d\n",
						FD1_SUMDATA(fd)->means[0], FD1_SUMDATA(fd)->stderrs[0],
						FD1_SUMDATA(fd)->n[0]);
			fprintf(fptr, "END_PARAMS\n");
			fprintf(fptr, "RASTERDATA\n");
			fprintf(fptr, "nlines=%d\n", FD1_SUMDATA(fd)->ndata - 1);
			if (fdo->no_errs) {
				for (i = 1; i < FD1_SUMDATA(fd)->ndata; i++)
					fprintf(fptr, "%e\t%e\t%d\n", FD1_SUMDATA(fd)->depvals[i],
								FD1_SUMDATA(fd)->means[i], FD1_SUMDATA(fd)->n[i]);
			}
			else {
				for (i = 1; i < FD1_SUMDATA(fd)->ndata; i++)
					fprintf(fptr, "%e\t%e\t%e\t%d\n", FD1_SUMDATA(fd)->depvals[i],
								FD1_SUMDATA(fd)->means[i], FD1_SUMDATA(fd)->stderrs[i],
								FD1_SUMDATA(fd)->n[i]);
			}
			fprintf(fptr, "END_RASTERDATA\n");
		}
		else {
			/* Dump a file suitable for importing into sigmaplot */
			if (fdo->show_raw_points == 1) {
				/* print column headings */
				line[0] = '\0';
				for (i = 2; i < ndata; i++) {
					sprintf(column, "%d\t", (int) FD1_SUMDATA(fd)->depvals[i - 1]);
					strcat(line, column);
				}
				fprintf(fptr, "%s\n", line);
				/* print nspikes */
				for (i = 0; i < FD_GI(fd, "reps"); i++) {
					line[0] = '\0';
					for (j = 2; j < ndata; j++) {
						if (i < n[j]) {
							sprintf(column, "%d\t", nspikes[j - 1][i]);
							strcat(line, column);
						}
						else {
							strcat(line, "-1\t");
						}
					}
					fprintf(fptr, "%s\n", line);
				}
			}
			else {
				if (fdo->no_errs) {
					for (i = 1; i < FD1_SUMDATA(fd)->ndata; i++)
						fprintf(fptr, "%e\t%e\n",
									FD1_SUMDATA(fd)->depvals[i],
									FD1_SUMDATA(fd)->means[i]);
				}
				else {
					for (i = 1; i < FD1_SUMDATA(fd)->ndata; i++)
						fprintf(fptr, "%e\t%e\t%e\n",
									FD1_SUMDATA(fd)->depvals[i],
									FD1_SUMDATA(fd)->means[i],
									FD1_SUMDATA(fd)->stderrs[i]);
				}
			}
		}
	}

	if (fdo->no_X)
		return;

	if (fdo->show_raw_points == 1) {
		FDObj_Add_Data_All(fdo, rx, ry, rn);
		FDObj_AddLine(fdo, l, rx, ry, NULL, rn, AtFloat,
					atQuadLinePlotWidgetClass, AtTypePOINTS, AtMarkCIRCLE,
					ConvertColor(fdo->graph, FDObj_Colors[l % FDObj_numColors]), NULL);
	}
	FDObj_Add_Data_All(fdo, px, py, pn);
	FDObj_AddLine(fdo, l, px, py, !(fdo->no_errs) ? pz : NULL, pn, AtFloat,
				atQuadLinePlotWidgetClass, AtTypeLINEPOINTS, AtMarkCIRCLE,
				ConvertColor(fdo->graph, FDObj_Colors[l % FDObj_numColors]),
				FDObj_Legend(fdo, l));
	FDObj_Add_Data_All(fdo, sx, sy, sn);
	FDObj_AddLine(fdo, l, sx, sy, !(fdo->no_errs) ? sz : NULL, sn, AtFloat,
				atQuadLinePlotWidgetClass, AtTypeLINEPOINTS, AtMarkCIRCLE,
				ConvertColor(fdo->graph, FDObj_Colors[l % FDObj_numColors]), NULL);

	XtVaSetValues(fdo->xaxis, XtNautoScale, True, NULL);
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
				XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

/* ------------------------------------------------------------------------
 * FD1_rasterFN
 * ------------------------------------------------------------------------ */
void FD1_rasterFN(FDO * fdo, FILEDATA * fd, int l, int raw, FILE * fptr)
{
	int i, j, k;
	char *ti;
	SA_SpikeRasterList *srl = NULL;
	char *xlabel, *ylabel;
	float *xdata, *ydata;
	int ndata, *rep;
	float low, high;
	float *ftmp, tmp2;
	int tmp;
	double xmin, xmax;
	float y;

	FD1_summarize(fd, fdo->sum_code[l]);
	srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
				FD1_RAWDATA(fd)->rastlist, FD_GI(fd, "epoch"), fdo->sum_code[l]);
	ndata = SA_TotalSpikesInSRL(srl);
	assert((xdata = (float *) calloc(ndata, sizeof(float))) != NULL);
	assert((ydata = (float *) calloc(ndata, sizeof(float))) != NULL);

	xlabel = "time (ms)";
	ylabel = raw ? "raster # (pres order)" : "dep int (no SR)";
	ti = raw ? "raw_raster" : "raster";

	if (fptr != NULL) {
		fprintf(fptr, "COMMENTS\n");
		fprintf(fptr, "%%%s, %s view\n",
					strrchr(fd->filename, '/') == NULL ? fd->filename :
					1 + (char *) strrchr(fd->filename, '/'), ti);
		fprintf(fptr, "%%col 1: %s\n", xlabel);
		fprintf(fptr, "%%col 2: %s\n", ylabel);
		fprintf(fptr, "END_COMMENTS\n");
		fprintf(fptr, "PARAMS\n");
		fprintf(fptr, "END_PARAMS\n");
		fprintf(fptr, "RASTERDATA\n");
		fprintf(fptr, "nlines=%d\n", ndata);
	}
	assert((rep = (int *) calloc(FD1_SUMDATA(fd)->ndata, sizeof(int))) != NULL);
	for (k = i = 0; i < srl->nrasters; i++) {
		if (raw) {
			y = FD1_RAWDATA(fd)->pres_order[i];
		}
		else {
			if (FD1_RAWDATA(fd)->depints[i] == SPONT)
				continue;
			tmp2 = (float) FD1_RAWDATA(fd)->depints[i];
			ftmp = (float *) bsearch(&tmp2, FD1_SUMDATA(fd)->depvals, FD1_SUMDATA(fd)->ndata,
						sizeof(float), floatcompare);
			high = ((ftmp + 1) < (FD1_SUMDATA(fd)->depvals + FD1_SUMDATA(fd)->ndata)) ?
						(*(ftmp + 1)) : 2 * (*ftmp) - (*(ftmp - 1));
			high = (*ftmp) + 0.5 * (high - (*ftmp));
			low = ((ftmp - 1) > FD1_SUMDATA(fd)->depvals) ?
						(*(ftmp - 1)) : 2 * (*ftmp) - (*(ftmp + 1));
			low = (*ftmp) + 0.5 * (low - (*ftmp));
			tmp = ftmp - FD1_SUMDATA(fd)->depvals;
			y = low + (high - low) * (float) (rep[tmp] %
						FD1_SUMDATA(fd)->n[tmp]) / (float) FD1_SUMDATA(fd)->n[tmp];
			rep[tmp]++;
		}
		for (j = 0; j < srl->sr[i]->nspikes; j++) {
			xdata[k] = srl->sr[i]->tm[j];
			ydata[k] = y;
			if (fptr != NULL)
				fprintf(fptr, "%e\t%e\n", xdata[k], ydata[k]);
			k++;
		}
	}
	if (fptr != NULL)
		fprintf(fptr, "END_RASTERDATA\n");

	SA_FreeSpikeRasterList(srl);
	free(rep);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, xdata, ydata, k);
	FDObj_AddLine(fdo, l, xdata, ydata, NULL, k,
				AtFloat, atQuadLinePlotWidgetClass,
				AtTypePOINTS, AtMarkVBAR, ConvertColor(fdo->graph,
							FDObj_Colors[l % FDObj_numColors]), FDObj_Legend(fdo, l));

	xmin = 0.0;
	xmax = (double) FD_GI(fd, "epoch");
	XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax, NULL);
	XtVaSetValues(fdo->xaxis, XtNautoScale, False, NULL);
	if (raw) {
		xmin = 0.0;
		xmax = (double) srl->nrasters;
		XtVaSetValues(fdo->yaxis, XtNmin, &xmin, XtNmax, &xmax, NULL);
		XtVaSetValues(fdo->yaxis, XtNautoScale, False, NULL);
	}
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
				XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

/* ------------------------------------------------------------------------
 * FD1_psthFN
 * ------------------------------------------------------------------------ */
void FD1_psthFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	char ti[32];
	SA_Histogram *h;
	SA_SpikeRasterList *srl = NULL;
	float binsize;
	char *xlabel, *ylabel;
	float *xdata, *ydata;
	int ndata, i;
	float adFc;
	double xmin, xmax;
	float tmin = 0.0;
	float tmax = fdo->epoch;
	int nbins;

	adFc=(float) FD_GI(fd,"adFc");
	srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
				FD1_RAWDATA(fd)->rastlist, FD_GI(fd, "epoch"), fdo->sum_code[l]);
	binsize = lookupParms_float("parms.psth_bin"); /* binsize is in msec */

	if (binsize <= 0.0) {
		fprintf(stderr, "xdview: psth_bin <= 0.0 \n"
						"xdview: making psth_bin = 1.0\n");
		binsize = 1.0;
	}

	/* No point in making the binsize less than one sampling period */
	if (binsize < (1000.0/adFc)) {
		fprintf(stderr, "xdview: psth_bin < sampling period\n"
						"xdview: making psth_bin = sampling period\n");
		binsize =  1000.0/adFc;
	}

	if ((lookupParms_int("parms.psth_use_start_end"))) {
		tmin = lookupParms_float("parms.psth_start");
		tmax = lookupParms_float("parms.psth_end");

		if (tmin < 0.0) tmin = 0.0;
		if (tmin > fdo->epoch) tmin = fdo->epoch;
		if (tmax < tmin) tmax = tmin + 1.0;
		if (tmax > fdo->epoch) tmax = fdo->epoch;
	}
		
	nbins= (int) (0.5 +(tmax-tmin)/binsize);

	h = SA_Psth(srl, tmin, tmax, nbins, lookupParms_int("parms.psth_triangle"));
	ndata = h->n;
	xdata = h->x;
	ydata = h->y;
	free(h);

	ndata++;
	assert((xdata = (float *) realloc(xdata, ndata * sizeof(float))) != NULL);
	assert((ydata = (float *) realloc(ydata, ndata * sizeof(float))) != NULL);
	xdata[ndata - 1] = 2 * xdata[ndata - 2] - xdata[ndata - 3];
	ydata[ndata - 1] = ydata[ndata - 2];

	sprintf(ti,"psth (binwidth=%5.3fmsec)",(tmax - tmin) / (float) nbins);
	xlabel = "time (ms)";
	ylabel = "nspikes";

	if (fptr != NULL) {
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "COMMENTS\n");
			fprintf(fptr, "%%%s, %s view\n",
						strrchr(fd->filename, '/') == NULL ? fd->filename :
						1 + (char *) strrchr(fd->filename, '/'), ti);
			fprintf(fptr, "%%col 1: %s\n", xlabel);
			fprintf(fptr, "%%col 2: %s\n", ylabel);
			fprintf(fptr, "END_COMMENTS\n");
			fprintf(fptr, "PARAMS\n");
			fprintf(fptr, "END_PARAMS\n");
			fprintf(fptr, "RASTERDATA\n");
			fprintf(fptr, "nlines=%d\n", ndata - 1);
		}
		for (i = 0; i < ndata - 1; i++)
			fprintf(fptr, "%e\t%e\n", xdata[i], ydata[i]);
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "END_RASTERDATA\n");
		}
	}
	SA_FreeSpikeRasterList(srl);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, xdata, ydata, ndata);
	FDObj_AddLine(fdo, l, xdata, ydata, NULL, ndata,
				AtFloat, atQuadBarPlotWidgetClass,
				-1, -1, ConvertColor(fdo->graph,
							FDObj_Colors[l % FDObj_numColors]), FDObj_Legend(fdo, l));

	xmin = (double) tmin;
	xmax = (double) tmax;
	XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax, NULL);
	XtVaSetValues(fdo->xaxis, XtNautoScale, False, NULL);
	/*XtVaSetValues(fdo->yaxis, XtNmin, &xmin, NULL);*/
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
				XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

/* ------------------------------------------------------------------------
 * FD1_isihFN
 * ------------------------------------------------------------------------ */
void FD1_isihFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	char *ti;
	SA_Histogram *h;
	SA_SpikeRasterList *srl = NULL;
	float f1, f2;
	char *xlabel, *ylabel;
	float *xdata, *ydata;
	int ndata, i;
	double xmin, xmax;

	srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
				FD1_RAWDATA(fd)->rastlist, FD_GI(fd, "epoch"), fdo->sum_code[l]);
	if ((f1 = lookupParms_float("parms.isih_max")) <= 0)
		f1 = 10.0;
	if ((f2 = lookupParms_float("parms.isih_bin")) <= 0.0)
		f2 = 1.0;
	h = SA_Isih(srl, 0.0, f1, (int) (0.5 + f1 / f2));
	ndata = h->n;
	xdata = h->x;
	ydata = h->y;
	free(h);

	ndata++;
	assert((xdata = (float *) realloc(xdata, ndata * sizeof(float))) != NULL);
	assert((ydata = (float *) realloc(ydata, ndata * sizeof(float))) != NULL);
	xdata[ndata - 1] = 2 * xdata[ndata - 2] - xdata[ndata - 3];
	ydata[ndata - 1] = ydata[ndata - 2];

	ti = "isih";
	xlabel = "time (ms)";
	ylabel = "nspikes";

	if (fptr != NULL) {
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "COMMENTS\n");
			fprintf(fptr, "%%%s, %s view\n",
						strrchr(fd->filename, '/') == NULL ? fd->filename :
						1 + (char *) strrchr(fd->filename, '/'), ti);
			fprintf(fptr, "%%col 1: %s\n", xlabel);
			fprintf(fptr, "%%col 2: %s\n", ylabel);
			fprintf(fptr, "END_COMMENTS\n");
			fprintf(fptr, "PARAMS\n");
			fprintf(fptr, "END_PARAMS\n");
			fprintf(fptr, "RASTERDATA\n");
			fprintf(fptr, "nlines=%d\n", ndata - 1);
		}
		for (i = 0; i < ndata - 1; i++)
			fprintf(fptr, "%e\t%e\n", xdata[i], ydata[i]);
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "END_RASTERDATA\n");
		}
	}
	SA_FreeSpikeRasterList(srl);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, xdata, ydata, ndata);
	FDObj_AddLine(fdo, l, xdata, ydata, NULL, ndata,
				AtFloat, atQuadBarPlotWidgetClass,
				-1, -1, ConvertColor(fdo->graph,
							FDObj_Colors[l % FDObj_numColors]), FDObj_Legend(fdo, l));

	xmin = 0.0;
	xmax = (double) lookupParms_float("parms.isih_max");
	XtVaSetValues(fdo->xaxis, XtNmin, &xmin, XtNmax, &xmax, NULL);
	XtVaSetValues(fdo->xaxis, XtNautoScale, False, NULL);
	XtVaSetValues(fdo->yaxis, XtNmin, &xmin, NULL);
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
				XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

#if(0)
/* ------------------------------------------------------------------------
 * FD1_autocorrFN
 * ------------------------------------------------------------------------ */
void FD1_autocorrFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	char *ti;
	SA_Histogram *h;
	SA_SpikeRasterList *srl = NULL;
	float f1;
	char *xlabel, *ylabel;
	float *xdata, *ydata;
	int ndata;

	srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
				FD1_RAWDATA(fd)->rastlist, FD_GI(fd, "epoch"), fdo->sum_code[l]);
	if ((f1 = lookupParms_float("parms.ach_bin")) <= 0.0)
		f1 = 1.0;
	h = SA_Correlate(srl, srl, 1, -fdo->epoch, fdo->epoch,
				(int) (0.5 + (2 * fdo->epoch / f1)));
	ndata = h->n;
	xdata = h->x;
	ydata = h->y;
	free(h);

	SA_FreeSpikeRasterList(srl);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, xdata, ydata, ndata);
	FDObj_AddLine(fdo, l, xdata, ydata, ndata, AtFloat,
				AtPlotLINES, -1, ConvertColor(fdo->graph,
							FDObj_Colors[l % FDObj_numColors]), FDObj_Legend(fdo, l));

	ti = "autocor";
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
				XtNshowTitle, True, NULL);
	xlabel = "time (ms)";
	ylabel = "nspikes";
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

/* ------------------------------------------------------------------------
 * FD1_densityFN
 * ------------------------------------------------------------------------ */
void FD1_densityFN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr)
{
	int i;
	char *ti;
	SA_SpikeRasterList *srl = NULL;
	SA_SpikeDensityList *sdlp = NULL;
	float f1, f2;
	char *xlabel, *ylabel;
	float *xdata, *ydata;
	int ndata;

	srl = FD_to_SpikeRasterList(FD1_RAWDATA(fd)->nrasters,
				FD1_RAWDATA(fd)->rastlist, FD_GI(fd, "epoch"), fdo->sum_code[l]);
	sdlp = NULL;
	if ((f1 = lookupParms_float("parms.sdens_sigma")) <= 0.0)
		f1 = 3.0;
	if ((f2 = lookupParms_float("parms.sdens_iters")) <= 0.0)
		f2 = 6.0;
	SA_CalcSpikeDensityList(srl, &sdlp, f1, (int) (0.5 + f2));
	ndata = sdlp->dur;
	assert((xdata = (float *) calloc(ndata, sizeof(float))) != NULL);
	assert((ydata = (float *) calloc(ndata, sizeof(float))) != NULL);
	for (i = 0; i < sdlp->dur; i++) {
		xdata[i] = i;
		ydata[i] = sdlp->mean[i];
	}

	SA_FreeSpikeDensityList(sdlp);
	SA_FreeSpikeRasterList(srl);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, xdata, ydata, ndata);
	FDObj_AddLine(fdo, l, xdata, ydata, ndata, AtFloat,
				AtPlotLINES, -1, ConvertColor(fdo->graph,
							FDObj_Colors[l % FDObj_numColors]), FDObj_Legend(fdo, l));

	ti = "density";
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
				XtNshowTitle, True, NULL);
	xlabel = "time (ms)";
	ylabel = "?nspikes/sec?";
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}
#endif

/* ------------------------------------------------------------------------
 * FD1_perhist1FN
 * ------------------------------------------------------------------------ */
void FD1_perhist1FN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr,
			float period, int nbins, int type)
{
	int i;
	float *depvals, *means;
	float mean_phase, vector_strength, rayleigh;
	int n;
	char *ti;
	char *xlabel, *ylabel;
	float *px, *py, *pz, *sx, *sy, *sz;
	int pn, sn;

	assert((type >= 0) && (type <= 2));

	FD_perhist1FN(fdo, fd, FD_GI(fd, "Spont_Stims"),
				FD1_RAWDATA(fd)->nrasters, FD1_RAWDATA(fd)->pres_order,
				FD1_RAWDATA(fd)->rastlist, FD1_RAWDATA(fd)->depints,
				&depvals, &means, &n, &mean_phase, &vector_strength, &rayleigh,
				period, nbins, type);

	switch (type) {
		case (PERHIST_MP):
			ylabel = "mean phase (cycles)";
			ti = "mean_phase";
			break;
		case (PERHIST_VS):
			ylabel = "vector strength";
			ti = "vector_strength";
			break;
		case (PERHIST_RT):
			ylabel = "rayleigh test (p-val)";
			ti = "rayleigh_test";
			break;
	}
	xlabel = FD_GV(fd, "depvar");

	FD_plotter_copy(depvals, means, NULL, n, (period == 0.0) ? 0 : 1,
				&px, &py, &pz, &pn, &sx, &sy, &sz, &sn);

	if (fptr != NULL) {
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "COMMENTS\n");
			fprintf(fptr, "%%%s, %s view\n",
						strrchr(fd->filename, '/') == NULL ? fd->filename :
						1 + (char *) strrchr(fd->filename, '/'), ti);
			fprintf(fptr, "%%col 1: %s\n", xlabel);
			fprintf(fptr, "%%col 2: %s\n", ylabel);
			fprintf(fptr, "END_COMMENTS\n");
			fprintf(fptr, "PARAMS\n");
			fprintf(fptr, "END_PARAMS\n");
			fprintf(fptr, "RASTERDATA\n");
			fprintf(fptr, "nlines=%d\n", n);
		}
		for (i = 0; i < n; i++)
			fprintf(fptr, "%e\t%e\n", depvals[i], means[i]);
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "END_RASTERDATA\n");
		}
	}
	free(means);
	free(depvals);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, px, py, pn);
	FDObj_AddLine(fdo, l, px, py, pz, pn,
				AtFloat, atQuadLinePlotWidgetClass,
				AtTypeLINEPOINTS, AtMarkCIRCLE, ConvertColor(fdo->graph,
							FDObj_Colors[l % FDObj_numColors]), FDObj_Legend(fdo, l));
	if (period != 0.0) {
		FDObj_Add_Data_All(fdo, sx, sy, sn);
		FDObj_AddLine(fdo, l, sx, sy, sz, sn,
					AtFloat, atQuadLinePlotWidgetClass,
					AtTypeLINEPOINTS, AtMarkCIRCLE, ConvertColor(fdo->graph,
								FDObj_Colors[l % FDObj_numColors]), FDObj_Legend(fdo, l));
	}
	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
				XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}

/* ------------------------------------------------------------------------
 * FD1_perhist2FN
 * ------------------------------------------------------------------------ */
void FD1_perhist2FN(FDO * fdo, FILEDATA * fd, int l, FILE * fptr,
			float period, int nbins)
{
/* fdo->no_errs=1 here means it's a true histogram, ie spikes are simply summed
 * and no division by # trials or # cycles occurrs.  fdo->no_errs=0 here means 
 * that a division by # cycles occurrs.  note that this is different from
 * the old perhist, which divides by # trials */

	int i;
	char ti[128], *xlabel, *ylabel;
	float *depvals, *means, *stderrs, *stddevs;
	int *n;
	float mean_phase, vector_strength, rayleigh;
	float *px, *py, *pz;
	int pn;

	FD_perhist2FN(fdo, fd,
				FD1_RAWDATA(fd)->nrasters, FD1_RAWDATA(fd)->pres_order,
				FD1_RAWDATA(fd)->rastlist, FD1_RAWDATA(fd)->depints,
				&depvals, &means, &stderrs, &stddevs, &n,
				&mean_phase, &vector_strength, &rayleigh,
				period, nbins);

	mps[l] = mean_phase;
	vss[l] = vector_strength;
	rays[l] = rayleigh;

	if (fdo->vsm_data == VSM_ALL_CODE)
		sprintf(dis[l], "all");
	else if (fdo->vsm_data == SPONT)
		sprintf(dis[l], "SPONT");
	else
		sprintf(dis[l], "%d", fdo->vsm_data);

	xlabel = "phase (cycles)";
	ylabel = "nspikes/cycle";
	sprintf(ti, "period_histogram(%s)", dis[l]);

	FD_plotter_copy(depvals, means, (!fdo->no_errs) ? stderrs : NULL,
				nbins, 0, &px, &py, &pz, &pn, NULL, NULL, NULL, NULL);

	if (fptr != NULL) {
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "COMMENTS\n");
			fprintf(fptr, "%%%s, %s view\n",
						strrchr(fd->filename, '/') == NULL ? fd->filename :
						1 + (char *) strrchr(fd->filename, '/'), ti);
			fprintf(fptr, "%%col 1: %s\n", xlabel);
			fprintf(fptr, "%%col 2: %s\n", ylabel);
			if (!fdo->no_errs)
				fprintf(fptr, "%%col 3: std error\n");
			fprintf(fptr, "END_COMMENTS\n");
			fprintf(fptr, "PARAMS\n");
			fprintf(fptr, "END_PARAMS\n");
			fprintf(fptr, "RASTERDATA\n");
			fprintf(fptr, "nlines=%d\n", nbins);
		}
		if (!fdo->no_errs) {
			for (i = 0; i < nbins; i++)
				fprintf(fptr, "%e\t%e\t%e\n", depvals[i], means[i], stderrs[i]);
		}
		else {
			for (i = 0; i < nbins; i++)
				fprintf(fptr, "%e\t%e\n", depvals[i], means[i]);
		}
		if (fdo->no_text_header != 1) {
			fprintf(fptr, "END_RASTERDATA\n");
		}
	}
	free(means);
	free(depvals);
	free(stderrs);
	free(stddevs);

	if (fdo->no_X)
		return;

	FDObj_Add_Data_All(fdo, px, py, pn);
	FDObj_AddLine(fdo, l, px, py, pz, pn,
				AtFloat, atQuadLinePlotWidgetClass,
				AtTypeLINEPOINTS, AtMarkCIRCLE, ConvertColor(fdo->graph,
							FDObj_Colors[l % FDObj_numColors]), FDObj_Legend(fdo, l));

	XtVaSetValues(fdo->graph, XtNtitle, FDObj_Title(fdo, ti, l),
				XtNshowTitle, True, NULL);
	XtVaSetValues(fdo->xaxis, XtNlabel, xlabel, NULL);
	XtVaSetValues(fdo->yaxis, XtNlabel, ylabel, NULL);
}


/* ------------------------------------------------------------------------
 * FD1_similarity_stats
 * ------------------------------------------------------------------------ */
static void my_chisquare(
			float *x1, float *y1, float *e1, int n1,
			float *x2, float *y2, float *e2, int n2,
			float *chisq, float *pval)
{
	int i;
	float tmp, chisq1, chisq2, pval1, pval2, min;

	if (e1 != NULL) {
		min = 10000.0;
		for (i = 0; i < n1; i++)
			if ((e1[i] < min) && (e1[i] != 0.0))
				min = e1[i];
		chisq1 = 0.0;
		for (i = 0; i < n1; i++) {
			tmp = lin_interpolate(x1[i], n2, x2, y2);
			tmp -= y1[i];
			tmp /= (e1[i] == 0.0) ? min : e1[i];
			tmp *= tmp;
			chisq1 += tmp;
		}
		pval1 = gammq(0.5 * n1, 0.5 * chisq1);
	}

	if (e2 != NULL) {
		min = 10000.0;
		for (i = 1; i < n2; i++)
			if ((e2[i] < min) && (e2[i] != 0.0))
				min = e2[i];
		chisq2 = 0.0;
		for (i = 0; i < n2; i++) {
			tmp = lin_interpolate(x2[i], n1, x1, y1);
			tmp -= y2[i];
			tmp /= (e2[i] == 0.0) ? min : e2[i];
			tmp *= tmp;
			chisq2 += tmp;
		}
		pval2 = gammq(0.5 * n2, 0.5 * chisq2);
	}

	if (e1 == NULL) {
		(*chisq) = chisq2;
		(*pval) = pval2;
	}
	else if (e2 == NULL) {
		(*chisq) = chisq1;
		(*pval) = pval1;
	}
	else {
		(*chisq) = (chisq1 + chisq2) / 2.0;
		(*pval) = (pval1 + pval2) / 2.0;
	}
}

static void compute_ave_diff(
			float *x1, float *y1, int n1,
			float *x2, float *y2, int n2,
			float *diff, float *diff_perc)
{
	int i;
	float tmp, max = -1.0, ret_val = 0.0;

	for (i = 0; i < n1; i++) {
		if (y1[i] > max)
			max = y1[i];
		tmp = lin_interpolate(x1[i], n2, x2, y2);
		tmp -= y1[i];
		ret_val += tmp;
	}
	for (i = 0; i < n2; i++) {
		tmp = lin_interpolate(x2[i], n1, x1, y1);
		tmp -= y2[i];
		ret_val -= tmp;
	}

	ret_val /= (n1 + n2);
	(*diff) = ret_val;
	(*diff_perc) = ret_val / max;
}

static void compute_stats(
			float *x1, float *y1, float *e1, int n1,
			float *x2, float *y2, float *e2, int n2,
			float *chisq, float *pval, float *diff, float *diff_perc)
{
	my_chisquare(x1, y1, e1, n1, x2, y2, e2, n2, chisq, pval);
	compute_ave_diff(x1, y1, n1, x2, y2, n2, diff, diff_perc);
}

#if(0)													/*keep me */
static float compute_repeated_ttests(FILEDATA * fd1, FILEDATA * fd2)
{
	float pval1, pval2;
	int i, n;

	assert(FD1_SUMDATA(fd1)->ndata == FD1_SUMDATA(fd2)->ndata);
	n = FD1_SUMDATA(fd1)->ndata;

	pval1 = pval2 = 1.0;
	for (i = 1; i < FD1_SUMDATA(fd1)->ndata; i++) {
		pval1 *= permutation_test_dp(
					FD1_SUMDATA(fd1)->nspikes[i], FD1_SUMDATA(fd1)->n[i],
					FD1_SUMDATA(fd2)->nspikes[i], FD1_SUMDATA(fd2)->n[i],
					-1);
		pval2 *= permutation_test_dp(
					FD1_SUMDATA(fd1)->nspikes[i], FD1_SUMDATA(fd1)->n[i],
					FD1_SUMDATA(fd2)->nspikes[i], FD1_SUMDATA(fd2)->n[i],
					1);
	}
	pval1 = pow(pval1, 1.0 / (-1.0 + n));
	pval2 = pow(pval2, 1.0 / (-1.0 + n));
	return (2.0 * MY_MIN(pval1, pval2));
}
#endif

extern int txt_get_n(FILEDATA *);
extern float *txt_get_x(FILEDATA *);
extern float *txt_get_y(FILEDATA *);
extern float *txt_get_e(FILEDATA *);

void FD1_similarity_stats(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
	int i, j, count, size;
	char *buf, buf2[128], *tmp;
	char *name, foo[64];
	float chisq, pval, diff, diff_perc;

	assert((buf = (char *) malloc(1024 * sizeof(char))) != NULL);
	size = 1024;
	count = 0;

	name = "chi square";
	tmp = "files                             chi^2 pval    ave diff       ave diff perc\n";
	strcpy(buf, tmp);
	count += strlen(tmp);
	tmp = "-----                             ----------    --------       -------------\n";
	strcat(buf, tmp);
	count += strlen(tmp);

	for (i = 0; i < (fdo->nfds - 1); i++) {
		for (j = i + 1; j < fdo->nfds; j++) {

			if ((!strcmp(fdo->fds[i]->modname, "txt")) &&
						(!strcmp(fdo->fds[j]->modname, "txt"))) {
				compute_ave_diff(
							txt_get_x(fdo->fds[i]), txt_get_y(fdo->fds[i]),
							txt_get_n(fdo->fds[i]),
							txt_get_x(fdo->fds[j]), txt_get_y(fdo->fds[j]),
							txt_get_n(fdo->fds[j]),
							&diff, &diff_perc);
				chisq = pval = -1;
			}
			else if (!strcmp(fdo->fds[i]->modname, "txt")) {
				compute_stats(
							txt_get_x(fdo->fds[i]), txt_get_y(fdo->fds[i]),
							NULL, txt_get_n(fdo->fds[i]),
							1 + FD1_SUMDATA(fdo->fds[j])->depvals,
							1 + FD1_SUMDATA(fdo->fds[j])->means,
							1 + FD1_SUMDATA(fdo->fds[j])->stddevs,
							-1 + FD1_SUMDATA(fdo->fds[j])->ndata,
							&chisq, &pval, &diff, &diff_perc);
			}
			else if (!strcmp(fdo->fds[j]->modname, "txt")) {
				compute_stats(
							1 + FD1_SUMDATA(fdo->fds[i])->depvals,
							1 + FD1_SUMDATA(fdo->fds[i])->means,
							1 + FD1_SUMDATA(fdo->fds[i])->stddevs,
							-1 + FD1_SUMDATA(fdo->fds[i])->ndata,
							txt_get_x(fdo->fds[j]), txt_get_y(fdo->fds[j]),
							NULL, txt_get_n(fdo->fds[j]),
							&chisq, &pval, &diff, &diff_perc);
			}
			else {
				compute_stats(
							1 + FD1_SUMDATA(fdo->fds[i])->depvals,
							1 + FD1_SUMDATA(fdo->fds[i])->means,
							1 + FD1_SUMDATA(fdo->fds[i])->stddevs,
							-1 + FD1_SUMDATA(fdo->fds[i])->ndata,
							1 + FD1_SUMDATA(fdo->fds[j])->depvals,
							1 + FD1_SUMDATA(fdo->fds[j])->means,
							1 + FD1_SUMDATA(fdo->fds[j])->stddevs,
							-1 + FD1_SUMDATA(fdo->fds[j])->ndata,
							&chisq, &pval, &diff, &diff_perc);
			}

#ifdef __linux__
			count += sprintf(buf2, "%-13s, %-13s  %e  %e  %e\n",
						strrchr(fdo->fds[i]->filename, '/') == NULL ? fdo->fds[i]->filename :
						1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
						strrchr(fdo->fds[j]->filename, '/') == NULL ? fdo->fds[j]->filename :
						1 + (char *) strrchr(fdo->fds[j]->filename, '/'),
						pval, diff, diff_perc);
#else
			count += strlen(sprintf(buf2, "%-13s, %-13s  %e  %e  %e\n",
									strrchr(fdo->fds[i]->filename, '/') == NULL ? fdo->fds[i]->filename :
									1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
									strrchr(fdo->fds[j]->filename, '/') == NULL ? fdo->fds[j]->filename :
									1 + (char *) strrchr(fdo->fds[j]->filename, '/'),
									pval, diff, diff_perc));
#endif
			if (fdo->to_tty)
				printf("%s", buf2);
			strcat(buf, buf2);
			if ((size - count) < 512) {
				assert((buf = (char *) realloc(buf, (size + 1024) * sizeof(char))) != NULL);
				size += 1024;
			}
		}
	}

	if (!fdo->to_tty) {
		sprintf(foo, "%s stats", name);
		pop_text(foo, buf, strlen(buf) + 1, False);
	}
}


#define PROB_NBINS 50
/* ------------------------------------------------------------------------
 * FD1_prob_distribution
 * ------------------------------------------------------------------------ */
void FD1_prob_distribution(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
	int **prob_bins = NULL;
	int i, j;
	int nreps;
	FILE *fp = NULL;

	nreps = FD_GI(fdo->fds[0], "reps");

	assert((prob_bins = (int **) calloc(FD1_SUMDATA(fdo->fds[0])->ndata,
										sizeof(int *))) != NULL);
	for (i = 0; i < FD1_SUMDATA(fdo->fds[0])->ndata; i++)
		assert((prob_bins[i] = (int *) calloc(PROB_NBINS, sizeof(int))) != NULL);

	for (i = 0; i < FD1_SUMDATA(fdo->fds[0])->ndata; i++)
		for (j = 0; j < nreps; j++)
			if (FD1_SUMDATA(fdo->fds[0])->nspikes[i][j] < PROB_NBINS)
				prob_bins[i][FD1_SUMDATA(fdo->fds[0])->nspikes[i][j]]++;

	/* write the MATLAB file */

	if (w == NULL)
		fp = stdout;
	else
		fp = fopen2("load_probs.m", "w");
	for (i = 0; i < FD1_SUMDATA(fdo->fds[0])->ndata; i++)
		fprintf(fp, "%%%d: %f\n", i + 1, FD1_SUMDATA(fdo->fds[0])->depvals[i]);
	fprintf(fp, "global depvars\n");
	fprintf(fp, "global prob\n");
	fprintf(fp, "depvars = [ ");
	for (i = 0; i < FD1_SUMDATA(fdo->fds[0])->ndata; i++)
		fprintf(fp, "%f ", FD1_SUMDATA(fdo->fds[0])->depvals[i]);
	fprintf(fp, "];\n");
	fprintf(fp, "prob = [ ");
	for (i = 0; i < FD1_SUMDATA(fdo->fds[0])->ndata; i++) {
		fprintf(fp, "[ ");
		for (j = 0; j < PROB_NBINS; j++)
			fprintf(fp, "%d ", prob_bins[i][j]);
		fprintf(fp, "]\n");
	}
	fprintf(fp, "];\n");
	if (w != NULL)
		fclose(fp);
	for (i = 0; i < FD1_SUMDATA(fdo->fds[0])->ndata; i++)
		free(prob_bins[i]);
	free(prob_bins);
}

#if(0)
/* ------------------------------------------------------------------------
 * FD1_linearity_stats
 * ------------------------------------------------------------------------ */
void FD1_linearity_stats(FDO * fdo, char *p, FILE * fptr)
{
	int i, j, k, off;
	float min, max, min2, max2, tmpf;
	float **g_y, *g_x;
	int g_n;
	char *q;
	float qf[100];

	if (fdo->nfds < 2)
		return;

	for (i = 1; i < fdo->nfds; i++) {
		if (strcmp(p, "")) {
			q = strtok(((i == 1) ? p : NULL), ",");
			sscanf(q, "%f", qf + i - 1);
		}
		else
			qf[i - 1] = 0.0;
	}

	min = FD1_SUMDATA(fdo->fds[0])->depvals[1];
	max = FD1_SUMDATA(fdo->fds[0])->depvals[FD1_SUMDATA(fdo->fds[0])->ndata - 1];
	for (i = 1; i < fdo->nfds; i++) {
		min2 = -qf[i - 1] + FD1_SUMDATA(fdo->fds[i])->depvals[1];
		max2 = -qf[i - 1] + FD1_SUMDATA(fdo->fds[i])->depvals[
					FD1_SUMDATA(fdo->fds[i])->ndata - 1];
		min = MY_MAX(min, min2);
		max = MY_MIN(max, max2);
	}

	assert((g_y = (float **) malloc((fdo->nfds - 1) * sizeof(float *))) != NULL);
	for (i = 1; i < fdo->nfds; i++) {
		assert((g_y[i - 1] = (float *) malloc(FD1_SUMDATA(fdo->fds[0])->ndata *
											sizeof(float))) != NULL);
		for (k = 0, j = 1; j < FD1_SUMDATA(fdo->fds[0])->ndata; j++) {
			tmpf = FD1_SUMDATA(fdo->fds[0])->depvals[j];
			if ((tmpf < min) || (tmpf > max))
				continue;
			g_y[i - 1][k++] = lin_interpolate(tmpf + qf[i - 1], FD1_SUMDATA(fdo->fds[i])->ndata,
						FD1_SUMDATA(fdo->fds[i])->depvals, FD1_SUMDATA(fdo->fds[i])->means);
		}
	}

	assert((g_x = (float *) malloc(FD1_SUMDATA(fdo->fds[0])->ndata * sizeof(float))) != NULL);
	for (g_n = 0, off = -1, k = 0, i = 1; i < FD1_SUMDATA(fdo->fds[0])->ndata; i++) {
		tmpf = FD1_SUMDATA(fdo->fds[0])->depvals[i];
		if ((tmpf < min) || (tmpf > max))
			continue;
		if (off == -1)
			off = i;
		g_x[k++] = FD1_SUMDATA(fdo->fds[0])->depvals[i];
		g_n++;
	}

	FD_linearity_stats(
				FD1_SUMDATA(fdo->fds[0])->depvals + off, FD1_SUMDATA(fdo->fds[0])->means + off,
				FD1_SUMDATA(fdo->fds[0])->stddevs + off, g_x, g_y, g_n,
				fdo->nfds - 1, fdo, fptr);

	for (i = 0; i < fdo->nfds - 1; i++)
		free(g_y[i]);
	free(g_y);
	free(g_x);
}
#endif


/* ------------------------------------------------------------------------
 * FD1_isih_stats
 * ------------------------------------------------------------------------ */
void FD1_isih_stats_menu(FDO * fdobj)
{
	if (fdobj->no_X)
		return;

	menubutton_clear(fdobj->fit, &(fdobj->fitpsh));
	XtSetSensitive(fdobj->fit, False);

	menubutton_add(fdobj->fitpsh, "stats",
				FD1_isih_stats, fdobj);
	XtSetSensitive(fdobj->fit, True);
}

static void compute_isih_stats(int nrasters, int **rastlist,
			int code, float crit, int *total, int *bad)
{
	int i, j;
	float last;

	(*total) = (*bad) = 0;

	for (i = 0; i < nrasters; i++) {
		if (rastlist[i] == NULL)
			continue;
    last = -1;
		for (j = 0; j < N_SPIKES(rastlist[i]); j++) {
			if (matchEvent(SPIKE_CHAN(rastlist[i], j), code)) {
				if (last == -1) {
					last = SPIKE_MSEC(rastlist[i], j);
					continue;
				}
				(*total) += 1;
				if ((-last + SPIKE_MSEC(rastlist[i], j)) < crit)
					(*bad) += 1;
				last = SPIKE_MSEC(rastlist[i], j);
			}
		}
	}
}

void FD1_isih_stats(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
	int i, count, size;
	char *buf, buf2[128], *tmp;
	int total, bad, sum_total, sum_bad;
	float crit;

	crit = lookupParms_float("parms.isih_percent");
	sum_total = sum_bad = 0;

	assert((buf = (char *) malloc(1024 * sizeof(char))) != NULL);
	size = 1024;
	count = 0;

	tmp = "file            total    bad     %\n";
	strcpy(buf, tmp);
	count += strlen(tmp);
	tmp = "----            ------   ---     -\n";
	strcat(buf, tmp);
	count += strlen(tmp);

	for (i = 0; i < fdo->nfds; i++) {
		compute_isih_stats(FD1_RAWDATA(fdo->fds[i])->nrasters,
					FD1_RAWDATA(fdo->fds[i])->rastlist,
					fdo->sum_code[i], crit, &total, &bad);
		sum_total += total;
		sum_bad += bad;

#ifdef __linux__
		count += sprintf(buf2, "%-13s   %6d   %4d    %6.3f\n",
					strrchr(fdo->fds[i]->filename, '/') == NULL ? fdo->fds[i]->filename :
					1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
					total, bad, ((float) bad / (float) total) * 100.0);
#else
		count += strlen(sprintf(buf2, "%-13s   %6d   %4d    %6.3f\n",
								strrchr(fdo->fds[i]->filename, '/') == NULL ? fdo->fds[i]->filename :
								1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
								total, bad, ((float) bad / (float) total) * 100.0));
#endif
		if (fdo->to_tty)
			printf("%s", buf2);
		strcat(buf, buf2);
		if ((size - count) < 512) {
			assert((buf = (char *) realloc(buf, (size + 1024) * sizeof(char))) != NULL);
			size += 1024;
		}
	}

#ifdef __linux__
	count += sprintf(buf2, "TOTAL            %6d   %4d    %6.3f\n",
				sum_total, sum_bad, ((float) sum_bad / (float) sum_total) * 100.0);
#else
	count += strlen(sprintf(buf2, "TOTAL            %6d   %4d    %6.3f\n",
							sum_total, sum_bad, ((float) sum_bad / (float) sum_total) * 100.0));
#endif
	if (fdo->to_tty)
		printf("%s", buf2);
	strcat(buf, buf2);
	if ((size - count) < 512) {
		assert((buf = (char *) realloc(buf, (size + 1024) * sizeof(char))) != NULL);
		size += 1024;
	}

	if (!fdo->to_tty)
		pop_text("stats", buf, strlen(buf) + 1, False);
}


/* ------------------------------------------------------------------------
 * FD1_perhist_stats
 * ------------------------------------------------------------------------ */
void FD1_perhist_stats_menu(FDO * fdobj)
{
	if (fdobj->no_X)
		return;

	menubutton_clear(fdobj->fit, &(fdobj->fitpsh));
	XtSetSensitive(fdobj->fit, False);

	menubutton_add(fdobj->fitpsh, "stats",
				FD1_perhist_stats, fdobj);
	XtSetSensitive(fdobj->fit, True);
}

void FD1_perhist_stats(Widget w, XtPointer fdoptr, XtPointer call_data)
{
	FDO *fdo = (FDO *) fdoptr;
	int i, count, size;
	char *buf, buf2[128], *tmp;

	assert((buf = (char *) malloc(1024 * sizeof(char))) != NULL);
	size = 1024;

	tmp = "file            depvar    MP     VS     Rayleigh\n";
	strcpy(buf, tmp);
	count += strlen(tmp);
	tmp = "----            ------    --     --     --------\n";
	strcat(buf, tmp);
	count += strlen(tmp);

	for (i = 0; i < fdo->nfds; i++) {
#ifdef __linux__
		count += sprintf(buf2, "%-13s   %6s  %5.3f   %4.2f    %5.3f\n",
					strrchr(fdo->fds[i]->filename, '/') == NULL ? fdo->fds[i]->filename :
					1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
					dis[i], mps[i], vss[i], rays[i]);
#else
		count += strlen(sprintf(buf2, "%-13s   %6s  %5.3f   %4.2f    %5.3f\n",
								strrchr(fdo->fds[i]->filename, '/') == NULL ? fdo->fds[i]->filename :
								1 + (char *) strrchr(fdo->fds[i]->filename, '/'),
								dis[i], mps[i], vss[i], rays[i]));
#endif
		strcat(buf, buf2);
		if ((size - count) < 512) {
			assert((buf = (char *) realloc(buf, (size + 1024) * sizeof(char))) != NULL);
			size += 1024;
		}
	}

	pop_text("stats", buf, strlen(buf) + 1, False);
}

/* ------------------------------------------------------------------------
 * FD1_contra_ipsi
 * ------------------------------------------------------------------------ */
void FD1_contra_ipsi(FILEDATA * fd)
{
	int i;
	char *p, *q;

	if (as_bool(getpref("contraipsi", "no"))) {
		if (FD_GF(fd, "ML") < 0.0) {
			for (i = 0; i < FD1_RAWDATA(fd)->nrasters; i++) {
				if (FD1_RAWDATA(fd)->depints[i] != SPONT)
					FD1_RAWDATA(fd)->depints[i] *= -1;
			}
			for (i = 0; i < FD1_SUMDATA(fd)->ndata; i++) {
				if (FD1_SUMDATA(fd)->depvals[i] != SPONT)
					FD1_SUMDATA(fd)->depvals[i] *= -1.0;
			}
		}
		p = FD_GV(fd, "depvar");
		q = (char *) malloc(strlen(p) + 200);

		sprintf(q, "C leads I        %s        I leads C", p);
		setvar("depvar", q, fd->params);

		free(q);
		fd->trashed = 1;
	}
}

/* ------------------------------------------------------------------------
 * FD1_sort
 * 
 * sorts trials by depints 
 * ------------------------------------------------------------------------ */
static void FD1_sort(FILEDATA * data)
{
	int i, j, min, idx;

	for (i = 0; i < (FD1_RAWDATA(data)->nrasters - 1); i++) {
		idx = i;
		min = FD1_RAWDATA(data)->depints[i];
		for (j = i + 1; j < FD1_RAWDATA(data)->nrasters; j++) {
			if (FD1_RAWDATA(data)->depints[j] < min) {
				idx = j;
				min = FD1_RAWDATA(data)->depints[j];
			}
		}
		if (idx != i) {
			SWAP(FD1_RAWDATA(data)->pres_order[i],
						FD1_RAWDATA(data)->pres_order[idx], int);
			if (FD1_RAWDATA(data)->ana != NULL)
				SWAP(FD1_RAWDATA(data)->ana[i],
							FD1_RAWDATA(data)->ana[idx], xword **);
			if (FD1_RAWDATA(data)->chan_ids != NULL)
				SWAP(FD1_RAWDATA(data)->chan_ids[i],
							FD1_RAWDATA(data)->chan_ids[idx], int *);
			SWAP(FD1_RAWDATA(data)->rastlist[i],
						FD1_RAWDATA(data)->rastlist[idx], spike_t *);
			SWAP(FD1_RAWDATA(data)->deplines[i],
						FD1_RAWDATA(data)->deplines[idx], char *);
			SWAP(FD1_RAWDATA(data)->depints[i],
						FD1_RAWDATA(data)->depints[idx], int);
		}
	}
}

/* ------------------------------------------------------------------------
 * FD1_summarize
 * 
 * summarize() is a little tricky because there are two options for
 * spontaneous trials.  if Spont_Stims=1, then one spont trial is
 * present for every repetition and the spont rate is calculated
 * from those trials.  if Spont_Stims=0, then
 * independent spont trials are not present, and the spont rate is
 * calculated from the Delay portion of all the trials *
 * ------------------------------------------------------------------------ */
static void FD1_summarize(FILEDATA * fd, int unitid)
{
	int nreps;          /* number of reps (ideally) */
	float t_begin;        /* driven begins at in ms (t_ in ticks) */
	float t_end;          /* driven ends at in ms (t_ in ticks) */
	float t_spont_begin;  /* spont begins at in ms (t_ in ticks) */
	float t_spont_end;    /* spont ends at in ms (t_ in ticks) */
	int use_delay;      /* use the pre-stim delay as spontaneous */
	int driven;         /* response during driven period */
	float *stattmp;     /* temp buf for means,stderrs,stddevs */
	int *stattmpi;
	int last;
	char *last_line;
	int i, j, k;
	int spont_stims;
	float *spont_resp;
	int *spont_respi;
	int tmp;

	FD1_freeSumData(fd);

	tmp = (1 + FD1_RAWDATA(fd)->nrasters);

	assert((FD1_SUMDATA(fd) = (FD1_SumData *) malloc(sizeof(FD1_SumData))) != NULL);
	assert((FD1_SUMDATA(fd)->n = (int *) malloc(tmp * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->means = (float *) malloc(tmp * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->stderrs = (float *) malloc(tmp * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->stddevs = (float *) malloc(tmp * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->depvals = (float *) malloc(tmp * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->depstrs = (char **) malloc(tmp * sizeof(char *))) != NULL);
	assert((FD1_SUMDATA(fd)->nspikes = (int **) malloc(tmp * sizeof(int *))) != NULL);

	/* pre-calculate all analysis parameters */

	FD_calc_params(fd, &spont_stims, &use_delay,
				&t_begin, &t_end, &t_spont_begin, &t_spont_end);
	nreps = FD_GI(fd, "reps");

	/* now summarize the data */
	k = 0;
	if ((!spont_stims) || use_delay) {
		k = 1;

		assert((spont_resp = (float *) malloc(
											(FD1_RAWDATA(fd)->nrasters) * sizeof(float))) != NULL);
		assert((spont_respi = (int *) malloc(
											(FD1_RAWDATA(fd)->nrasters) * sizeof(int))) != NULL);

		for (i = 0; i < FD1_RAWDATA(fd)->nrasters; i++) {
			spont_resp[i] = countRaster(t_spont_begin, t_spont_end,
						FD1_RAWDATA(fd)->rastlist[i], unitid);
			spont_resp[i] *= ((t_end - t_begin) /
						(t_spont_end - t_spont_begin));
			spont_respi[i] = (int) (spont_resp[i] + 0.5);
		}

		FD1_SUMDATA(fd)->n[0] = i;
		FD1_SUMDATA(fd)->means[0] = mean(spont_resp, i);
		FD1_SUMDATA(fd)->stderrs[0] = stderror(spont_resp, i);
		FD1_SUMDATA(fd)->stddevs[0] = stddev(spont_resp, i);
		FD1_SUMDATA(fd)->depvals[0] = (float) SPONT;
		FD1_SUMDATA(fd)->depstrs[0] = "depvar=-6666 <SPONT>";
		FD1_SUMDATA(fd)->nspikes[0] = spont_respi;

		free(spont_resp);
	}

	for (i = 0; i < FD1_RAWDATA(fd)->nrasters;) {

		assert((stattmp = (float *) malloc(nreps * sizeof(float))) != NULL);
		assert((stattmpi = (int *) malloc(nreps * sizeof(int))) != NULL);

		last = FD1_RAWDATA(fd)->depints[i];
		last_line = FD1_RAWDATA(fd)->deplines[i];

		if (use_delay && (last == SPONT)) {
			i++;
			continue;
		}

		for (j = 0; i < FD1_RAWDATA(fd)->nrasters &&
					FD1_RAWDATA(fd)->depints[i] == last; i++, j++) {
			driven = countRaster(t_begin, t_end, FD1_RAWDATA(fd)->rastlist[i],
						unitid);
			stattmp[j] = (float) driven;
			stattmpi[j] = driven;
		}

		FD1_SUMDATA(fd)->n[k] = j;
		FD1_SUMDATA(fd)->means[k] = mean(stattmp, j);
		FD1_SUMDATA(fd)->stderrs[k] = stderror(stattmp, j);
		FD1_SUMDATA(fd)->stddevs[k] = stddev(stattmp, j);
		FD1_SUMDATA(fd)->depvals[k] = (float) last;
		FD1_SUMDATA(fd)->depstrs[k] = last_line;
		FD1_SUMDATA(fd)->nspikes[k] = stattmpi;

		free(stattmp);
		k++;

/*
		if (j != nreps)
      fprintf(stderr, "; note trial %d had %d of %d samples\n", last, j, nreps);
*/
	}

	assert((FD1_SUMDATA(fd)->n = (int *) realloc(FD1_SUMDATA(fd)->n,
										k * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->means = (float *) realloc(FD1_SUMDATA(fd)->means,
										k * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->stderrs = (float *) realloc(FD1_SUMDATA(fd)->stderrs,
										k * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->stddevs = (float *) realloc(FD1_SUMDATA(fd)->stddevs,
										k * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->depvals = (float *) realloc(FD1_SUMDATA(fd)->depvals,
										k * sizeof(float))) != NULL);
	assert((FD1_SUMDATA(fd)->depstrs = (char **) realloc(FD1_SUMDATA(fd)->depstrs,
										k * sizeof(char *))) != NULL);
	assert((FD1_SUMDATA(fd)->nspikes = (int **) realloc(FD1_SUMDATA(fd)->nspikes,
										k * sizeof(int *))) != NULL);

	FD1_SUMDATA(fd)->ndata = k;
	FD1_SUMDATA(fd)->unit = unitid;
}

/* ------------------------------------------------------------------------
 * FD1_reader
 * ------------------------------------------------------------------------ */
int FD1_reader(FILE * fp, FILEDATA * data, int ana_flag)
{
	int nrasters, act_nrasters, i, j, ana_nchans;
	char *p, *q;
	spike_t **rastlist;
	char **strings;
	int *ints, *pres_order;
	float **avepmk;
	xword ***ana;
	xword **tmp_ana;
	int *chan_ids;
	int *all_chan_ids;

	assert((data->rawdata = calloc(1, sizeof(FD1_RawData))) != NULL);
	data->sumdata = NULL;

	if ((p = skipTo(fp, "nrasters=")) == NULL ||
				sscanf(p, "nrasters=%d", &nrasters) != 1) {
		fclose(fp);
		FD_free(data);
		return (0);
	}

	if (ana_flag) {
		if (FD_GI(data,"file.version")>=3)
			ana_nchans=FD_GI(data,"ana.nchans");
		else
			ana_nchans=1;

		assert((ana = (xword ***) calloc(nrasters, sizeof(xword *))) != NULL);
		assert((all_chan_ids = (int **) calloc(nrasters, sizeof(int *))) != NULL);
	}

	assert((rastlist = (spike_t **) calloc(nrasters, sizeof(spike_t *))) != NULL);
	assert((strings = (char **) calloc(nrasters, sizeof(char *))) != NULL);
	assert((ints = (int *) calloc(nrasters, sizeof(int))) != NULL);
	assert((pres_order = (int *) calloc(nrasters, sizeof(int))) != NULL);
	assert((avepmk = (float **) calloc(nrasters, sizeof(float *))) != NULL);

	for (i = act_nrasters = 0; i < nrasters; i++) {
		if ((p = skipTo(fp, "depvar=")) == NULL)
			break;
		rastlist[act_nrasters] = readRaster(fp, data, avepmk + act_nrasters);
		if ((q = rindex(p, '\n')) != NULL)
			*q = 0;
		if (rastlist[act_nrasters] != NULL) {
			pres_order[act_nrasters] = act_nrasters;
			strings[act_nrasters] = strsave(p);
			sscanf(p, "depvar=%d", &ints[act_nrasters]);
			if (ana_flag) {
				if (readAnalog(fp, data, &tmp_ana, &chan_ids)!=0) {
					assert((ana[i] = (xword **) calloc(ana_nchans, sizeof(xword *))) != NULL);
					for (j=0;j<ana_nchans;j++)  
						ana[i][j]=tmp_ana[j];
					all_chan_ids[i]=chan_ids;
				} else {
					ana[i]=NULL;
					all_chan_ids[i]=NULL;
				}
			}
			act_nrasters++;
		}
	}

	if (act_nrasters < nrasters) {
       fprintf(stderr, ";Warning:  %s is data light\n", basename(data->filename));
		if (ana_flag) {
			assert((ana = (xword ***) realloc(ana, act_nrasters * sizeof(xword **))) != NULL);
			assert((chan_ids = (int **) realloc(chan_ids, act_nrasters * sizeof(int **))) != NULL);
		}
		assert((rastlist = (spike_t **) realloc(rastlist, act_nrasters * sizeof(spike_t *))) != NULL);
		assert((strings = (char **) realloc(strings, act_nrasters * sizeof(char *))) != NULL);
		assert((ints = (int *) realloc(ints, act_nrasters * sizeof(int))) != NULL);
		assert((pres_order = (int *) realloc(pres_order, act_nrasters * sizeof(int))) != NULL);
		assert((avepmk = (float **) realloc(avepmk, act_nrasters * sizeof(float *))) != NULL);
	}

	FD1_RAWDATA(data)->ana_nchans = ana_flag ? ana_nchans : 0;
	FD1_RAWDATA(data)->ana = ana_flag ? ana : NULL;
	FD1_RAWDATA(data)->chan_ids = ana_flag ? all_chan_ids : NULL;
	FD1_RAWDATA(data)->rastlist = rastlist;
	FD1_RAWDATA(data)->deplines = strings;
	FD1_RAWDATA(data)->depints = ints;
	FD1_RAWDATA(data)->pres_order = pres_order;
	FD1_RAWDATA(data)->avepmk = avepmk;
	FD1_RAWDATA(data)->nrasters = act_nrasters;

	FD1_sort(data);

	data->channels = NULL;
	FD_howmany_units(&(data->channels), FD1_RAWDATA(data)->nrasters,
				FD1_RAWDATA(data)->rastlist);

	return (1);
}

/* ------------------------------------------------------------------------
 * FD1_free
 * ------------------------------------------------------------------------ */

void FD1_freeRawData(FILEDATA * data)
{
	int i,j;

	for (i = 0; i < FD1_RAWDATA(data)->nrasters; i++) {
		if (FD1_RAWDATA(data)->ana != NULL) {
			for (j=0; j<FD1_RAWDATA(data)->ana_nchans; j++)
				FREE(FD1_RAWDATA(data)->ana[i][j]);
			FREE(FD1_RAWDATA(data)->ana[i]);
		}
		if (FD1_RAWDATA(data)->chan_ids != NULL) 
			FREE(FD1_RAWDATA(data)->chan_ids[i]);
		if (FD1_RAWDATA(data)->rastlist != NULL)
			FREE(FD1_RAWDATA(data)->rastlist[i]);
		if (FD1_RAWDATA(data)->deplines != NULL)
			FREE(FD1_RAWDATA(data)->deplines[i]);
		if (FD1_RAWDATA(data)->avepmk != NULL)
			FREE(FD1_RAWDATA(data)->avepmk[i]);
	}

	FREE(FD1_RAWDATA(data)->pres_order);
	FREE(FD1_RAWDATA(data)->ana);
	FREE(FD1_RAWDATA(data)->chan_ids);
	FREE(FD1_RAWDATA(data)->rastlist);
	FREE(FD1_RAWDATA(data)->deplines);
	FREE(FD1_RAWDATA(data)->depints);
	FREE(FD1_RAWDATA(data)->avepmk);

	FREE(FD1_RAWDATA(data));
}

static void FD1_freeSumData(FILEDATA * data)
{
	int i;

	if (FD1_SUMDATA(data) != NULL) {
		FREE(FD1_SUMDATA(data)->depstrs);		/*vals were freed by free'ing deplines array */
		FREE(FD1_SUMDATA(data)->depvals);
		FREE(FD1_SUMDATA(data)->means);
		FREE(FD1_SUMDATA(data)->stderrs);
		FREE(FD1_SUMDATA(data)->stddevs);
		FREE(FD1_SUMDATA(data)->n);
		for (i = 0; i < FD1_SUMDATA(data)->ndata; i++)
			FREE(FD1_SUMDATA(data)->nspikes[i]);
		FREE(FD1_SUMDATA(data)->nspikes);
		FREE(FD1_SUMDATA(data));
	}
}

int FD1_free(FILEDATA * data)
{
	FD1_freeRawData(data);
	FD1_freeSumData(data);

	return (1);
}

/* ------------------------------------------------------------------------
 * FD1_get_*
 * ------------------------------------------------------------------------ */
float *FD1_get_means(FILEDATA * fd)
{
	return (FD1_SUMDATA(fd)->means);
}

float *FD1_get_depvals(FILEDATA * fd)
{
	return (FD1_SUMDATA(fd)->depvals);
}

char **FD1_get_depstrs(FILEDATA * fd)
{
      return (FD1_SUMDATA(fd)->depstrs);
}

float *FD1_get_stderrs(FILEDATA * fd)
{
	return (FD1_SUMDATA(fd)->stderrs);
}

int FD1_get_ndata(FILEDATA * fd)
{
	return (FD1_SUMDATA(fd)->ndata);
}

int *FD1_get_n(FILEDATA * fd)
{
	return (FD1_SUMDATA(fd)->n);
}
int **FD1_get_nspikes(FILEDATA * fd)
{
	return (FD1_SUMDATA(fd)->nspikes);
}

int *FD1_get_depints(FILEDATA * fd)
{
	return (FD1_RAWDATA(fd)->depints);
}

int FD1_get_nrasters(FILEDATA * fd)
{
	return (FD1_RAWDATA(fd)->nrasters);
}

int *FD1_get_pres_order(FILEDATA * fd)
{
	return (FD1_RAWDATA(fd)->pres_order);
}

spike_t **FD1_get_rastlist(FILEDATA * fd)
{
	return (FD1_RAWDATA(fd)->rastlist);
}

int FD1_get_ana_nchans(FILEDATA * fd)
{
	return (FD1_RAWDATA(fd)->ana_nchans);
}

xword ***FD1_get_ana(FILEDATA * fd)
{
	return (FD1_RAWDATA(fd)->ana);
}

int **FD1_get_chan_ids(FILEDATA * fd)
{
	return (FD1_RAWDATA(fd)->chan_ids);
}

/* ------------------------------------------------------------------------
 * FD1_peak_stats
 * ------------------------------------------------------------------------ */
void FD1_peak_stats(Widget w, XtPointer fdobjptr, XtPointer call_data)
{
	FDO *fdobj = (FDO *) fdobjptr;
	int i, *ndata;
	float **x, **y, *spont;
	char **file;

	assert((ndata = (int *) malloc(fdobj->nfds * sizeof(int))) != NULL);
	assert((x = (float **) malloc(fdobj->nfds * sizeof(float *))) != NULL);
	assert((y = (float **) malloc(fdobj->nfds * sizeof(float *))) != NULL);
	assert((spont = (float *) malloc(fdobj->nfds * sizeof(float))) != NULL);
	assert((file = (char **) malloc(fdobj->nfds * sizeof(char *))) != NULL);

	for (i = 0; i < fdobj->nfds; i++) {
		ndata[i] = FD1_SUMDATA(fdobj->fds[i])->ndata - 1;
		x[i] = FD1_SUMDATA(fdobj->fds[i])->depvals + 1;
		y[i] = FD1_SUMDATA(fdobj->fds[i])->means + 1;
		spont[i] = FD1_SUMDATA(fdobj->fds[i])->means[0];
		file[i] = fdobj->fds[i]->filename;
	}

	FD_peak_stats(fdobj->nfds, ndata, x, y, spont, file, fdobj->to_tty,
				!strcmp(fdobj->fds[0]->modname, "itd") ? 1 : 0);

	free(ndata);
	free(x);
	free(y);
	free(spont);
	free(file);
}

static float *copyVec(float *in, int n, float shift)
{
	float *out;
	int i;

	assert((out = (float *) malloc(n * sizeof(float))) != NULL);
	for (i = 0; i < n; i++)
		out[i] = in[i] - shift;

	return (out);
}

/* ------------------------------------------------------------------------
 * FD1_combine_data
 * ------------------------------------------------------------------------ */
void FD1_combine_data(FDO * fdobj, char *p, FILE * fptr)
{
	int i, n, tn, t2n;
	float *x, *y, *tx, *ty, *t2x, *t2y;
	float spont;
	char *q;
	float qf;

	if (fdobj->nfds < 2)
		return;

	fprintf(fptr, "COMMENTS\n");
	fprintf(fptr, "%%a combine w/ %s of ", p);

	for (i = n = 0, spont = 0.0; i < fdobj->nfds; i++) {
		if (strcmp(p, "")) {
			q = strtok(((i == 0) ? p : NULL), ",");
			sscanf(q, "%f", &qf);
		}
		else
			qf = 0.0;

		t2n = FD1_get_ndata(fdobj->fds[i]) - 1;
		t2y = FD1_get_means(fdobj->fds[i]) + 1;
		t2x = copyVec(FD1_get_depvals(fdobj->fds[i]) + 1, t2n, qf);

		FD_combine_data(x, y, n, t2x, t2y, t2n, &tx, &ty, &tn);
		spont += FD1_get_means(fdobj->fds[i])[0];

		free(t2x);
		if (i > 0) {
			free(x);
			free(y);
		}

		x = tx;
		y = ty;
		n = tn;
	}

	spont /= fdobj->nfds;

	for (i = 0; i < n; i++)
		y[i] -= (spont * (fdobj->nfds - 1));

	for (i = 0; i < fdobj->nfds; i++)
		fprintf(fptr, "%s, ",
					strrchr(fdobj->fds[i]->filename, '/') == NULL ? fdobj->fds[i]->filename :
					1 + (char *) strrchr(fdobj->fds[i]->filename, '/'));
	fprintf(fptr, "\n");
	fprintf(fptr, "%%col 1: x[i]\n");
	fprintf(fptr, "%%col 2: y[i]\n");
	fprintf(fptr, "END_COMMENTS\n");
	fprintf(fptr, "PARAMS\n");
	fprintf(fptr, "spont=%e\t%e\n", spont, 0.0);
	fprintf(fptr, "END_PARAMS\n");
	fprintf(fptr, "RASTERDATA\n");
	fprintf(fptr, "nlines=%d\n", n);
	for (i = 0; i < n; i++)
		fprintf(fptr, "%e\t%e\n", x[i], y[i]);
	fprintf(fptr, "END_RASTERDATA\n");
}
