/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: dynraster.c,v 2.51 1999/03/12 01:57:59 bjarthur Exp $
** Program: dowl
**  Module: dynraster.c
**  Author: mazer
** Descrip: dynamic raster displays
**
** Revision History (most recent last)
**
** Mon Apr  4 11:43:31 1994 mazer
**  started with rasterplot.c
**
*******************************************************************/

#include "xdphyslib.h"

void dynraster_add(
     DRASTER *r,
     int  max,					/* maximum possible timestamp */
     spike_t *spikelist,		/* spike time struct */
     int mask,
     int begin,                 /* stimulus begin/end in milliseconds */
     int end)		
{
  int nspikes = (int) N_SPIKES(spikelist);
  int i, j;
  double xmax = (double) max;
  spike_t time;

  if (r->nlines > 0 && r->nextline == 0 && r->ndata > 0) {
    r->ndata = 0;
    r->nmarks = 0;
  }

  if (nspikes > 0) {
	  j = r->ndata + nspikes;

	  r->xdata = (float *)realloc(r->xdata, sizeof(float) * j);
	  r->ydata = (float *)realloc(r->ydata, sizeof(float) * j);

	  assert(r->xdata != NULL);
	  assert(r->ydata != NULL);

	  XtVaSetValues(r->xaxis, XtNmax, &xmax, NULL);

	  for (i = 0; i < nspikes; i++) {
		if(matchEvent(SPIKE_CHAN(spikelist, i),mask)) {
		  r->ydata[r->ndata] = r->nlines - r->nextline;
		  time = SPIKE_TIME(spikelist,i);
		  r->xdata[r->ndata] = ((float) time / 1e4);
		  r->ndata++;
		}
	  }

	  r->xdata = (float *)realloc(r->xdata, sizeof(float) * r->ndata);
	  r->ydata = (float *)realloc(r->ydata, sizeof(float) * r->ndata);

	  assert(r->xdata != NULL);
	  assert(r->ydata != NULL);

	  AtQuadLinePlotAttachData(r->spikes,
				 r->xdata, AtFloat, sizeof(float),
				 r->ydata, AtFloat, sizeof(float),
				 NULL, AtFloat, sizeof(float),
				 NULL, AtFloat, sizeof(float), 0, r->ndata);

	  if (begin != end) {
		j = r->nmarks + 2;
		r->xmarks = (float *)realloc(r->xmarks, sizeof(float) * j);
		r->ymarks = (float *)realloc(r->ymarks, sizeof(float) * j);

		r->ymarks[r->nmarks] = r->nlines - r->nextline;
		r->xmarks[r->nmarks] = (float)begin;
		r->nmarks++;

		r->ymarks[r->nmarks] = r->nlines - r->nextline;
		r->xmarks[r->nmarks] = (float)end;
		r->nmarks++;

		AtQuadLinePlotAttachData(r->spikes,
				   r->xmarks, AtFloat, sizeof(float),
				   r->ymarks, AtFloat, sizeof(float),
				   NULL, AtFloat, sizeof(float),
				   NULL, AtFloat, sizeof(float), 0, r->nmarks);
	  }
  }

  if (r->nlines > 0) {
    if (++r->nextline >= r->nlines)
      r->nextline = 0;
  }
}

void dynraster_clear(DRASTER *r)
{
  r->nextline = 0;

  r->ndata = 1;
  if (!r->xdata) {
	  r->xdata = (float *)calloc(r->ndata,sizeof(float));
  } else {
	  r->xdata = (float *)realloc(r->xdata, sizeof(float) * r->ndata);
  }
  if (!r->ydata) {
	  r->ydata = (float *)calloc(r->ndata,sizeof(float));
  } else {
	  r->ydata = (float *)realloc(r->ydata, sizeof(float) * r->ndata);
  }

  AtQuadLinePlotAttachData(r->spikes,
			 r->xdata, AtFloat, sizeof(float),
			 r->ydata, AtFloat, sizeof(float),
			 NULL, AtFloat, sizeof(float),
			 NULL, AtFloat, sizeof(float), 0, r->ndata);

  r->nmarks = 1;
  if (!r->xmarks) {
	  r->xmarks = (float *)calloc(r->nmarks,sizeof(float));
  } else {
	  r->xmarks = (float *)realloc(r->xmarks,sizeof(float) * r->nmarks);
  }
  if (!r->ymarks) {
	  r->ymarks = (float *)calloc(r->nmarks,sizeof(float));
  } else {
	  r->ymarks = (float *)realloc(r->ymarks, sizeof(float) * r->nmarks);
  }
  AtQuadLinePlotAttachData(r->spikes,
			 r->xmarks, AtFloat, sizeof(float),
			 r->ymarks, AtFloat, sizeof(float),
			 NULL, AtFloat, sizeof(float),
			 NULL, AtFloat, sizeof(float), 0, r->nmarks);
}

DRASTER *dynraster_new(Widget parent, Dimension width, Dimension height,
      int nlines)
{
  double min, max;
  DRASTER *d = (DRASTER *) calloc(sizeof(DRASTER), 1);

  d->nlines = nlines;
  d->nextline = 0;

  d->ndata = 1;
  d->xdata = (float *)calloc(1, sizeof(float));
  d->ydata = (float *)calloc(1, sizeof(float));

  d->nmarks = 1;
  d->xmarks = (float *)calloc(1, sizeof(float));
  d->ymarks = (float *)calloc(1, sizeof(float));

#if 0
  d->ndata = 0;
  d->xdata = NULL;
  d->ydata = NULL;

  d->nmarks = 0;
  d->xmarks = NULL;
  d->ymarks = NULL;
#endif 

  d->r = XtVaCreateManagedWidget("raster", atPlotterWidgetClass, parent,
			    XtNwidth, width,
			    XtNheight, height,
			    XtNshowLegend, False,
			    NULL);
  min = 0.0;
  max = 1.0;
  d->xaxis = XtVaCreateWidget("xaxis", atQuadAxisWidgetClass, d->r,
		     XtNvertical, False,
		     XtNmin, &min,
		     XtNmax, &max,
		     XtNroundEndpoints, False,
		     XtNautoScale, False,
		     NULL);

  d->yaxis = XtVaCreateWidget("yaxis", atQuadAxisWidgetClass, d->r,
		     XtNvertical, True,
		     NULL);
  if (nlines >= 0) {
    min = 0.0;
    max = nlines;
    XtVaSetValues(d->yaxis,
		  XtNautoScale, False,
		  XtNmin, &min, XtNmax, &max,
		  XtNautoScale, False, XtNroundEndpoints, False,
		  XtNdrawNumbers, False,
		  XtNdrawFrame, False,
		  NULL);
  }

  XtVaSetValues(d->r, XtNxAxis, d->xaxis, XtNyAxis, d->yaxis, NULL);

  d->spikes = XtVaCreateWidget("spikes", atQuadLinePlotWidgetClass, d->r,
		     XtNplotType, AtTypePOINTS,
		     XtNplotMark, AtMarkVBAR,
		     NULL);

  AtQuadLinePlotAttachData(d->spikes,
			 d->xdata, AtFloat, sizeof(float),
			 d->ydata, AtFloat, sizeof(float),
			 NULL, AtFloat, sizeof(float),
			 NULL, AtFloat, sizeof(float), 0, d->ndata);

  d->marks = XtVaCreateWidget("marks", atQuadLinePlotWidgetClass, d->r,
		     XtNplotType, AtTypePOINTS,
		     XtNplotMark, AtMarkVBAR,
		     NULL);
  AtQuadLinePlotAttachData(d->marks,
			 d->xmarks, AtFloat, sizeof(float),
			 d->ymarks, AtFloat, sizeof(float),
			 NULL, AtFloat, sizeof(float),
			 NULL, AtFloat, sizeof(float), 0, d->nmarks);

  return(d);
}
