/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: clock.c,v 2.45 1999/01/26 21:02:35 cmalek Exp $
** Program: dowl
**  Module: clock.c
**  Author: mazer
** Descrip: digital clock "gadget"
**
** Revision History (most recent last)
**
*******************************************************************/

#include "xdphyslib.h"
#include "led_on.bm"
#include "led_off.bm"

static Pixmap led_on_pix;
static Pixmap led_off_pix;
static int led_pix_init = 0;

static void UpdateCB(XtPointer, XtIntervalId*);
static void UpdateLedCB(XtPointer, XtIntervalId*);
static void set_clk_label(struct clock_info*, int);
static void ClockCB(Widget, XtPointer, XtPointer);

static void set_clk_label(struct clock_info *info, int nodash)
{
  time_t elapsed;
  char buf[50];
  int h, m, s;

  if (info->running || nodash) {
    elapsed = time((long *)0) - info->startat;
    h = elapsed / (60 * 60);
    m = (elapsed % (60 * 60)) / 60;
    s = elapsed % 60;
    if (nodash) {
      if (info->opt_label) {
	if (info->interval < 1000)
	  sprintf(buf, "%s %02dh%02dm%02d", info->opt_label, h % 100, m, s);
	else
	  sprintf(buf, "%s %02dh%02d", info->opt_label, h % 100, m);
      } else {
	if (info->interval < 1000)
	  sprintf(buf, "%02dh%02dm%02d", h % 100, m, s);
	else
	  sprintf(buf, "%02dh%02d", h % 100, m);
      }
    } else {
      if (info->opt_label) {
	if (info->interval < 1000)
	  sprintf(buf, "%s %02d:%02d:%02d", info->opt_label, h % 100, m, s);
	else
	  sprintf(buf, "%s %02d:%02d", info->opt_label, h % 100, m);
      } else {
	if (info->interval < 1000)
	  sprintf(buf, "%02d:%02d:%02d", h % 100, m, s);
	else
	  sprintf(buf, "%02d:%02d", h % 100, m);
      }
    }
    if (info->alarm_time > 0) {
      if (!info->alarm_trigd && (info->alarm_time < elapsed)) {
	beep(50);
	notify("Clock Alarm: %s", info->name);
	info->alarm_trigd = 1;
	info->led_timeout =
	  XtAppAddTimeOut(XtWidgetToApplicationContext(info->w),
			  500, UpdateLedCB, info);
      }
    }
  } else {
    if (info->opt_label) {
      if (info->interval < 1000)
	sprintf(buf, "%s --:--:--", info->opt_label);
      else
	sprintf(buf, "%s --:--", info->opt_label);
    } else {
      if (info->interval < 1000)
	sprintf(buf, "--:--:--");
      else
	sprintf(buf, "--:--");
    }
  }
  if (strcmp(buf, info->last_buf) != 0) {
    XtVaSetValues(info->w, XtNlabel, buf, NULL);
    strcpy(info->last_buf, buf);
  }
}

static void UpdateCB(XtPointer info_blah, XtIntervalId *id)
{
  struct clock_info *info = (struct clock_info*)info_blah;

  set_clk_label(info, 0);
  info->timeout = XtAppAddTimeOut(XtWidgetToApplicationContext(info->w),
				  info->interval, UpdateCB, info);
}

static void UpdateLedCB(XtPointer info_blah, XtIntervalId *id)
{
  struct clock_info *info = (struct clock_info*)info_blah;

  if (info->led_state)
    XtVaSetValues(info->w, XtNleftBitmap, led_off_pix, NULL);
  else
    XtVaSetValues(info->w, XtNleftBitmap, led_on_pix, NULL);
  info->led_state = !(info->led_state);
  info->led_timeout = XtAppAddTimeOut(XtWidgetToApplicationContext(info->w),
				      500, UpdateLedCB, info);
}


static void ClockCB(Widget w, XtPointer info_blah, XtPointer client_data)
{
  struct clock_info *info = (struct clock_info*)info_blah;

  if (w != NULL && info->clktype == CLOCK_SYSTEM) {
    if (info->running) {
      if (pop_box("System clock: HALT?", "No", "Yes", NULL) == 1)
	return;
    } else {
      if (pop_box("System clock: START?", "No", "Yes", NULL) == 1)
	return;
    }
  }

  if (info->running) {
    XtRemoveTimeOut(info->timeout);
    info->running = 0;
    setvar_int(info->name, 0, info->svartable);
    set_clk_label(info, 1);
    info->alarm_trigd = 0;
    if (info->led_timeout) {
      XtRemoveTimeOut(info->led_timeout);
      info->led_timeout = 0;
      XtVaSetValues(info->w, XtNleftBitmap, led_off_pix, NULL);
    }
  } else {
    info->startat = time((long *)0);
    info->timeout = XtAppAddTimeOut(XtWidgetToApplicationContext(info->w),
				    info->interval, UpdateCB, info);
    info->running = 1;
    setvar_int(info->name, info->startat, info->svartable);
    set_clk_label(info, 0);
    info->alarm_trigd = 0;
  }
  if (info->statefile)
    savevars(info->statefile, info->svartable);
}

void clock_reset(struct clock_info *info, int restart)
{
  if (info->running)
    ClockCB(NULL, info, NULL);
  if (restart == CLOCK_RESTART)
    ClockCB(NULL, info, NULL);
  info->alarm_trigd = 0;
  if (info->led_timeout) {
    XtRemoveTimeOut(info->led_timeout);
    info->led_timeout = 0;
    XtVaSetValues(info->w, XtNleftBitmap, led_off_pix, NULL);
  }
}

int clock_read(struct clock_info *info, int *hourp, int *minp, int *secp)
{
  time_t elapsed;
  extern time_t time();

  if (info->running) {
    elapsed = time((long *)0) - info->startat;
    *hourp = elapsed / (60 * 60);
    *minp = (elapsed % (60 * 60)) / 60;
    *secp = elapsed % 60;
    return(1);
  } else {
    return(0);
  }
}

Widget clock_new(Widget parent, char *name, Widget horiz, Widget vert,
      char *statefile, SVAR_TABLE *svartable, int clktype, void **handle,
      char *label, int alarm_secs, int interval)
{
  extern Widget TopLevel;
  struct clock_info *info =
    (struct clock_info *)malloc(sizeof(struct clock_info));

  if (led_pix_init == 0) {
    led_pix_init = 1;
    led_on_pix = makePixmap(led_on_bits, led_on_width, led_on_height);
    led_off_pix = makePixmap(led_off_bits, led_off_width, led_off_height);
  }

  if (svartable == NULL)
    svartable = global_svar_table;

  if (interval > 0)
    info->interval = interval;
  else
    info->interval = 1000;

  info->running = 0;
  info->w = button_new(parent, name ? name : "clock", name ? name : "clock", 
		       ClockCB, info, horiz, vert);
  info->clktype = clktype;
  info->name = strsave(name ? name : "clock");
  info->statefile = statefile ? strsave(statefile) : NULL;
  info->svartable = svartable;
  info->opt_label = label;
  info->alarm_time = alarm_secs;
  info->alarm_trigd = 0;
  info->led_timeout = 0;
  info->led_state = 0;
  strcpy(info->last_buf, "");

  if (info->alarm_time > 0) {
    XtVaSetValues(info->w, XtNleftBitmap, led_off_pix, NULL);
  }
  
  if ((info->startat = getvar_int(info->name, info->svartable)) != 0) {
    info->running = 1;
    set_clk_label(info, 0);
    info->timeout = XtAppAddTimeOut(XtWidgetToApplicationContext(info->w),
				    info->interval, UpdateCB, info);
  } else
    set_clk_label(info, 0);
  if (handle != NULL)
    *handle = (void *)info;
  return(info->w);
}
