/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: analogfns.c,v 2.60 2001/03/27 06:59:23 cmalek Exp $
** Program: dowl
**  Module: analogfns.c
**  Author: jamie
** Descrip: analog save funcions
**
** Revision History (most recent last)
**  
** Tue Nov 30 21:57:30 1993 mazer
**  creation date
**
** Thu Oct  6 18:09:24 1994 mazer
**  added handling of ana-decimate factor..
**
** Sun Nov 13 16:01:40 1994 mazer
**  added appendAnalog() function to take a fileNAME instead
**  of a filePOINTER...
**
** Fri Nov 18 13:30:16 1994 mazer
**  added support for analog calibration data in wave header..
**
** Wed Nov 23 14:53:39 1994 mazer
**  accidentally saved calibration scale data using relative
**  offsets.. changed NOW .. prev collected files are WRONG
**  but can probably be figured out off line..
**
** 99.04 bjarthur
**  changed writeAnalog to output ascii, instead of binary.
**  analog data is now save to the xdphys files, instead of
**  creating a separate .ana file.  takes more space, but
**  it's easier to keep track of things
**
*******************************************************************/

#ifndef __NO_XDPHYSLIB
#include "xdphyslib.h"
#else /* __NO_XDPHYSLIB */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <iserver.h>
#include "xdphysio.h"
#include "raster.h"
#include "svars.h"
#include "waveio.h"
#include "addacal.h"
#include "fopen2.h"
#endif /* __NO_XDPHYSLIB */
#include "analogfns.h"
#ifndef __linux__
#define	M_32_SWAP(a) {							\
	unsigned int _tmp = a;						\
	((char *)&a)[0] = ((char *)&_tmp)[3];				\
	((char *)&a)[1] = ((char *)&_tmp)[2];				\
	((char *)&a)[2] = ((char *)&_tmp)[1];				\
	((char *)&a)[3] = ((char *)&_tmp)[0];				\
}
#define	M_16_SWAP(a) {							\
	unsigned short _tmp = a;						\
	((char *)&a)[0] = ((char *)&_tmp)[1];				\
	((char *)&a)[1] = ((char *)&_tmp)[0];				\
}
#define	M_16_SWAP2(a) {							\
	unsigned short _tmp = a;						\
	((char *)&a)[0] = ((char *)&_tmp)[1];				\
	((char *)&a)[1] = ((char *)&_tmp)[0];				\
}
#else /* __linux__ */
#define	M_32_SWAP(a) 
#define	M_16_SWAP(a)
#endif /* not __linux__ */


static void readSingleAnalog(FILE *, int, xword *);

void writeBuffers(FILE *fp, char *depstr, char *filename, int rasternum)
{
	xword *adbuf, *lbuf, *rbuf;
	int nsamps, i, j;
	float tomv, offset;
	int decimate;

	decimate = GI("ana-decimate");
	if (LastVarUndefined || decimate < 1)
		decimate = 1;

	/* this just strips out the trailing newlines.. */
	if (depstr && (i = strlen(depstr)) > 0) {
		if (depstr[i - 1] == '\n')
			depstr[i - 1] = 0;
	}

	if ((fp != NULL) && ((adbuf = is_getADbuffer(&nsamps)) != NULL)) {
		nsamps /= decimate;

		assert((lbuf = (xword *) calloc(nsamps, sizeof(xword))) != NULL);
		assert((rbuf = (xword *) calloc(nsamps, sizeof(xword))) != NULL);
		for (j = i = 0; i < nsamps; i++, j += (2 * decimate)) {
			lbuf[i] = adbuf[j];
			rbuf[i] = adbuf[j + 1];
		}

		get_analog_conversion(&tomv, &offset, 0);
		writewave(fp, lbuf, is_adFc / decimate, nsamps, 1,
			  depstr, "writeBuffers()", tomv, offset, filename,
			  rasternum);

		get_analog_conversion(&tomv, &offset, 1);
		writewave(fp, rbuf, is_adFc / decimate, nsamps, 1,
			  depstr, "writeBuffers()", tomv, offset, filename,
			  rasternum);

		free(lbuf);
		free(rbuf);
	}
}

void writeAnalog(FILE * fp)
{
	xword **anabufs;
	int  i, j, chan;
	int nsamples;
	float tomv, offset;
	int decimate;
	char tmp[32];
	xword tmpword;
	char *chan_state;

	decimate = GI("ana-decimate");
	if (LastVarUndefined || decimate < 1)
		decimate = 1;

	assert(sizeof(xword) == 2);

	if (fp != NULL) {
		spiker_getAnalogBuffers(&nsamples, &anabufs);
		for (chan=0; chan<is_getADnchans(); chan++) {
			sprintf(tmp,"ana.channel%d",chan+1);
			chan_state=GS(tmp);
			if (chan_state !=NULL) {
				if (strncmp(chan_state,"on",2)==0) {
					get_analog_conversion(&tomv, &offset, chan);
					nsamples /= decimate;
					fprintf(fp,"TRACE\n");
					fprintf(fp,"channel=%d\n",chan+1);

					for (j = i = 0; i < nsamples; i++, j += (is_getADnchans() * decimate)) {
						tmpword=anabufs[chan][j];
						sprintf(tmp, "%08x", tmpword);
						fprintf(fp, "%s", tmp + 4);
						if (((i + 1) % 20) == 0)
							fprintf(fp, "\n");
					}

					if ((i % 20) != 0)
						fprintf(fp, "\n");
						
					fprintf(fp,"END_TRACE\n");
				}
			}
		}
		free(anabufs);
	}
}

int readAnalog(FILE * fp, FILEDATA * data, xword ***anabufs, int **chan_ids)
{
	int 	i;  
	int 	nsamps;
	char 	tmp[32];
	float 	adFc;
	float 	epoch;
	int 	pos;
	int 	nchans;
	int     decimate;

	adFc = (float) FD_GI(data, "adFc");
	epoch = ((float) FD_GI(data, "epoch")) / 1000.0;

	/* only file format >= 2.1 has ana.nchans, and 
	 * more than 1 analog trace per raster */
	if (FD_GI(data,"file.version")>=3)
		nchans=FD_GI(data,"ana.nchans");
	else
		nchans=1;

	assert((*anabufs=(xword **) calloc(nchans, sizeof(xword *)))!=NULL);
	assert((*chan_ids=(int *) calloc(nchans, sizeof(int)))!=NULL);
	for (i=0; i<nchans; i++) 
		assert(((*anabufs)[i] = (xword *) malloc(nsamps * 
			sizeof(xword*))) != NULL);

	strcpy(tmp, "00000000");
	decimate=GI("ana-decimate");
	if (LastVarUndefined || decimate < 1)
		decimate = 1;

	nsamps = (int) (adFc*epoch) / decimate;
	pos=ftell(fp);
	fread(tmp + 4, sizeof(char), 4, fp);
	if ((!strcmp(tmp,"0000depv")) || (!strcmp(tmp,"0000END_"))) {
		/* There are no traces associated with this raster */
		fseek(fp, pos, SEEK_SET);
		for (i=0; i<nchans; i++) {
			free((*anabufs)[i]);
			free((*anabufs));
			(*anabufs)=NULL;
			return(0);
		}
	} else {
		if (FD_GI(data,"file.version")>=3) {
			for (i=0; i<nchans; i++) {
				/* Read the "TRACE" marker */
				fgets(tmp, sizeof(tmp), fp);
				/* Read in the "channel=" line */
				fgets(tmp, sizeof(tmp), fp);
				sscanf(tmp,"channel=%d",&((*chan_ids)[i]));
				readSingleAnalog(fp,nsamps,(*anabufs)[i]);
				/* Read the "END_TRACE" marker */
				fgets(tmp, sizeof(tmp), fp);
			}
		} else {
			/* Cover backwards compatibility */
			readSingleAnalog(fp,nsamps,(*anabufs)[0]);
			(*chan_ids)[0]=1;
		}
	}

	return (nchans);
}

static void readSingleAnalog(FILE * fp, int nsamps, xword *buf)
{
	int i, tmpi;
	char tmp[32];

	for (i = 0; i < nsamps; i++) {
		fread(tmp + 4, sizeof(char), 4, fp);
		sscanf(tmp, "%x", &tmpi);
		/* Read in in little-endian (i.e. i386) format */
		M_16_SWAP(((short *)&tmpi)[2]);
		buf[i] = (short) tmpi;
		if (((i + 1) % 20) == 0)
			/* deal with the end of line character */
			fread(tmp + 4, sizeof(char), 1, fp);
		strcpy(tmp, "00000000");
	}
}

void get_analog_conversion(float *tomv, float *offset, int chan)
{
	/* get calibration data for analog channel */

	if (chan == 0) {
		if (ad_to_mv(1.0, tomv, NULL, AddaRelative) == 0)
			(*tomv) = (*offset) = 0.0;
		if (ad_to_mv(2.0, offset, NULL, AddaAbsolute) == 0)
			(*tomv) = (*offset) = 0.0;
	} else {
		if (ad_to_mv(1.0, NULL, tomv, AddaRelative) == 0)
			(*tomv) = (*offset) = 0.0;
		if (ad_to_mv(0.0, NULL, offset, AddaAbsolute) == 0)
			(*tomv) = (*offset) = 0.0;
	}
}
