/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: Wave.c,v 2.47 1998/11/20 22:17:00 bjarthur Exp $
** Program: dowl
**  Module: Wave.c
**  Author: mazer
** Descrip: waveform widget -- true widget for X/ps display lists
**
** Revision History (most recent last)
**
** Sat Mar 14 11:08:16 1992 mazer
**  creation date :: derrived from current version of Canvas.c
**
** Tue Nov  3 00:40:52 1992 mazer
**  fixed fencepost error in Redisplay() that made waveforms
**  have gaps when > width of window
**
** Wed Dec  9 14:22:55 1992 mazer
**  cleaned up some stuff in Redisplay() and switched things
**  to use X(PS)DrawLines instead of X(PS)DrawLine
**
** Tue Dec 15 18:21:07 1992 mazer
**  added simple menu interface for usual keyboard commands
**  (for Allison!)
**
** Sat Jan 23 19:47:08 1993 mazer
**  changed data to be xword's instead of ints
**
** Sun Jan 31 21:51:42 1993 mazer
**  move PrintAction into PostScrtip.c
**  renamed psWave() to XtWavePS() and made it public
**
** Fri Feb  5 15:51:27 1993 mazer
**  fixed big memory leak in Redisplay() where segments list
**  was realloc'd each call, but never free'd. Removed the help
**  panel -- this was the wrong way to do help anyway..
**
** Mon Sep 27 11:29:43 1993 mazer
**  - added a proper status line when wave.display_info is set
**  - forceupdate() added for code reuse.
**  - elminated (useless?) yzoom stuff
**  - fixed (I think) visible-scale mode
**
** Thu Oct  7 19:04:13 1993 mazer
**  Changed vcursor stuff to read hcursor .. I don't know why
**  I ever call horizontal line "vertical" cursors..
**
** Thu Dec  9 22:35:31 1993 mazer
**  removed copy-action since it's covered more generally 
**  by the PostScript module
**  
** Sun Mar  6 15:27:03 1994 mazer
**  moved postscript and fig output generators from PostScript.c
**  back here -- to avoid sucking in Canvas, AtPlotter etc..
**  
** Tue Nov  1 19:00:54 1994 mazer
**  Added XtWaveSetWaveformOffset()
**
** Tue Nov  8 18:13:15 1994 mazer
**  ChangeDisplayAction() won't propagate down links if it
**  ends in an error (ie XBell)..
**
** 97.4 bjarthur
**  moved si_label from ggems.c here and deleted ggems.c
**
*******************************************************************/

#include "xdphyslib.h"

#define mag(x) (((x) < 0) ? -(x) : (x))

static void GeneratePSAction(WaveWidget, XEvent*, String*, Cardinal*);
static void GenerateFIGAction(Widget, XEvent*, String*, Cardinal*);

static void privMenuCB(Widget, String, caddr_t);
static Widget privMenu(Widget, String, String[]);
static void Initialize(WaveWidget, WaveWidget);
static int avail_height(WaveWidget);
static void DrawCursors(WaveWidget, int, int, int, int);
static void DrawGrid(WaveWidget);
static char *si_label(float);
static void DrawYAxisLabels(WaveWidget, int, int);
static void Redisplay(WaveWidget, XEvent*, Region);
static void forceupdate(WaveWidget);
static void Destroy(WaveWidget);
static void Resize(WaveWidget);
static Boolean SetValues(WaveWidget, WaveWidget, WaveWidget);
static void InputAction(WaveWidget, XEvent*, String*, Cardinal*);
static void ToggleLinesAction(WaveWidget, XEvent*, String*, Cardinal*);
static void ChangeAutoScaleAction(WaveWidget, XEvent*, String*, Cardinal*);
static void SetCursorAction(WaveWidget, XEvent*, String*, Cardinal*);
static int xzoomPushPop(WaveWidget, int);
static void ChangeDisplayAction(WaveWidget, XEvent*, String*, Cardinal*);
static void ToggleAutoRangeAction(WaveWidget, XEvent*, String*, Cardinal*);
static void ToggleInfoAction(WaveWidget, XEvent*, String*, Cardinal*);
static void ToggleGridAction(WaveWidget, XEvent*, String*, Cardinal*);
static void UnlinkCB(WaveWidget, WaveWidget, caddr_t);
static char *getafilename(char*, char*);

static XtResource resources[] = {
#define offset(field) XtOffset(WaveWidget, wave.field)
  { XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
      offset(foreground_pixel), XtRString, XtDefaultForeground },
  { XtNgridColor, XtCGridColor, XtRPixel, sizeof(Pixel),
      offset(gridcolor_pixel), XtRString, XtDefaultForeground },
  { XtNcallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
      offset(input_callback), XtRCallback, (caddr_t) NULL },
  { XtNuseLines, XtCUseLines, XtRBoolean, sizeof(Boolean),
      offset(use_lines), XtRImmediate, (XtPointer) False },
  { XtNautoScale, XtCAutoScale, XtRBoolean, sizeof(Boolean),
      offset(auto_scale), XtRImmediate, (XtPointer) False },
  { XtNautoRange, XtCAutoRange, XtRBoolean, sizeof(Boolean),
      offset(auto_range), XtRImmediate, (XtPointer) True },
  { XtNshowGrid, XtCShowGrid, XtRBoolean, sizeof(Boolean),
      offset(show_grid), XtRImmediate, (XtPointer) False },
  { XtNwaveBegin, XtCWavePosition, XtRInt, sizeof(int),
      offset(wave_begin), XtRImmediate, (XtPointer) -1 },
  { XtNwaveEnd, XtCWavePosition, XtRInt, sizeof(int),
      offset(wave_end), XtRImmediate, (XtPointer) -1 },
  { XtNfc, XtCFc, XtRInt, sizeof(int),
      offset(fc), XtRImmediate, (XtPointer) -1 },
  { XtNdisplayInfo, XtCDisplayInfo, XtRBoolean, sizeof(Boolean),
      offset(display_info), XtRImmediate, (XtPointer) False },
  { XtNshowYaxis, XtCAxisView, XtRBoolean, sizeof(Boolean),
      offset(show_y_axis), XtRImmediate, (XtPointer) False },
  { XtNshowXaxis, XtCAxisView, XtRBoolean, sizeof(Boolean),
      offset(show_x_axis), XtRImmediate, (XtPointer) False },
  { XtNfont, XtCFont, XtRString, sizeof(String),
      offset(fontname), XtRString, "6x13" },

  /* XtNuserCursors determines whether or not use can change 'em */
  { XtNuserCursors, XtCSensitive, XtRBoolean, sizeof(Boolean),
      offset(user_cursors), XtRImmediate, (XtPointer) False },

  /* XtNcursor1 and XtNcursor2 are *vertical* cursor positions */
  { XtNcursor1, XtCWavePosition, XtRInt, sizeof(int),
      offset(cursor1), XtRImmediate, (XtPointer) -1 },
  { XtNcursor2, XtCWavePosition, XtRInt, sizeof(int),
      offset(cursor2), XtRImmediate, (XtPointer) -1 },

  
  /* XtNhcursor1 and XtNhcursor2 are *horizontal* cursor positions */
  { XtNhorizCursor1, XtCWavePosition, XtRInt, sizeof(int),
      offset(hcursor1), XtRImmediate, (XtPointer) -1 },
  { XtNhorizCursor2, XtCWavePosition, XtRInt, sizeof(int),
      offset(hcursor2), XtRImmediate, (XtPointer) -1 },

  /* XtNshowHcursor1 and XtNshowHcursor2 display visibility */
  { XtNshowHorizCursor1, XtCCursorView, XtRBoolean, sizeof(Boolean),
      offset(show_h_cursor1), XtRImmediate, (XtPointer) False },
  { XtNshowHorizCursor2, XtCCursorView, XtRBoolean, sizeof(Boolean),
      offset(show_h_cursor2), XtRImmediate, (XtPointer) False },

#undef offset
};

static void privMenuCB(Widget w, String dostring, caddr_t call_data)
{
  int i;
/*
  static void ChangeDisplayAction();
  static void ChangeAutoScaleAction();
  static void ToggleAutoRangeAction();
  static void ToggleInfoAction();
  static void ToggleGridAction();
*/

  if (strcmp(dostring, "autoscale") == 0) {
    i = 0;
    ChangeAutoScaleAction(XtParent(XtParent(w)), NULL, NULL, &i);
  } else if (strcmp(dostring, "autorange") == 0) {
    i = 0;
    ToggleAutoRangeAction(XtParent(XtParent(w)), NULL, NULL, &i);
  } else if (strcmp(dostring, "info") == 0) {
    i = 0;
    ToggleInfoAction(XtParent(XtParent(w)), NULL, NULL, &i);
  } else if (strcmp(dostring, "grid") == 0) {
    i = 0;
    ToggleGridAction(XtParent(XtParent(w)), NULL, NULL, &i);
  } else {
    i = 1;
    ChangeDisplayAction(XtParent(XtParent(w)), NULL, &dostring, &i);
  }
}

static String waveMenu[] = {
  "Toggle Info (space)",	"info",
  "Toggle Grid     (g)",	"grid",
  "AutoScale       (s)",	"autoscale",
  "AutoRange       (r)",	"autorange",
  "",				"",
  "Zoom            (z)",	"xzoom",
  "unzoom (1x)     (u)",	"unxzoom",
  "Unzoom (all)    (U)",	"unxzoomall",
  "",				"",
  "Pan Left",			"panleft",
  "Pan Right",			"panright",
  "Pan Home",			"panhome",
  "Pan End",			"panend",
  NULL
};

static Widget privMenu(Widget parent, String name, String menuData[])
{
  int i;
  Widget menu, menuEntry;
  char transbuf[1000];

  sprintf(transbuf,
	  "<Btn3Down>: XawPositionSimpleMenu(%s) MenuPopup(%s)", name, name);

  menu = XtVaCreatePopupShell(name, simpleMenuWidgetClass, parent, NULL);
  for (i = 0; menuData[i] != NULL; i += 2) {
    if (*menuData[i]) {
      menuEntry = XtVaCreateManagedWidget(menuData[i],
					  smeBSBObjectClass, menu, NULL);
      XtAddCallback(menuEntry, XtNcallback, privMenuCB, menuData[i + 1]);
    } else {
      menuEntry = XtVaCreateManagedWidget("menusep", smeLineObjectClass, menu,
					  XtNsensitive, False, NULL);
    }
  }
  XawSimpleMenuAddGlobalActions(XtWidgetToApplicationContext(parent));
  XtOverrideTranslations(parent,
			 XtParseTranslationTable(transbuf));
  return(menu);
}

static void Initialize(WaveWidget request, WaveWidget new)
{
  Font font;
  XGCValues values;
 
  values.background = new->core.background_pixel;
  values.foreground = new->wave.foreground_pixel;
  new->wave.gc = XtGetGC(new, GCBackground | GCForeground, &values);
  XSetLineAttributes(XtDisplay(new), new->wave.gc, 
		     0, LineSolid, CapRound, JoinRound);

  values.background = new->core.background_pixel;
  values.foreground = new->wave.gridcolor_pixel;
  new->wave.grid_gc = XtGetGC(new, GCBackground | GCForeground, &values);
  XSetLineAttributes(XtDisplay(new), new->wave.grid_gc, 
		     0, LineDoubleDash, CapRound, JoinRound);

  if ((font = XLoadFont(XtDisplay(new), new->wave.fontname)) != NULL) {
    XSetFont(XtDisplay(new), new->wave.grid_gc, font);
    new->wave.fontinfo = 
      XLoadQueryFont(XtDisplay(new), new->wave.fontname);
  } else {
    XtAppWarning(XtWidgetToApplicationContext(new),
		 "Can't load specified wave font.");
  }

  new->wave.xzoomstack = NULL;
  new->wave.waveform = NULL;
  new->wave.link_to = NULL;
  new->wave.link_hit = 0;
  new->wave.npoints = 0;
  privMenu(new, "menu", waveMenu);  
}

static int avail_height(WaveWidget w)
{
  if (w->wave.display_info) {
    return(w->core.height - w->wave.fontinfo->ascent - 3);
  } else {
    return(w->core.height);
  }
}


static void DrawCursors(WaveWidget w, int cursor, int begin, int end,
      int npoints)
{
  int z;

  if (!XtIsRealized(w)) {
    return;
  }

  if (begin < 0) {
    if (w->wave.wave_begin >= 0)
      begin = w->wave.wave_begin;
    else
      begin = 0;
    if (w->wave.wave_end >= 0 && w->wave.wave_end <= w->wave.npoints)
      end = w->wave.wave_end;
    else
      end = w->wave.npoints;
    npoints = end - begin;
  }

  if ((cursor == 0 || cursor == 1) &&
      (w->wave.cursor1 >= begin && w->wave.cursor1 < end)) {
    z = (w->wave.cursor1 - begin) * w->core.width / npoints;
    XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.gc,
		z, 0, z, avail_height(w));
  }
      
  if ((cursor == 0 || cursor == 2) &&
      (w->wave.cursor2 >= begin && w->wave.cursor2 < end)) {
    z = (w->wave.cursor2 - begin) * w->core.width / npoints;
    XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.grid_gc,
		z, 0, z, avail_height(w));
  }
}

static void DrawGrid(WaveWidget w)
{
  register int i, j, x, y;

  /* vertical lines */
  i = w->core.width / 10;
  for (x = i; x < w->core.width; x += i)
    XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
		x, 0, x, avail_height(w));
  
  /* horizontal lines */
  j = avail_height(w) / 2;
  i = j / 3;
  for (y = 0; y < 3; y++) {
    XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
		0, j + (y * i), w->core.width, j + (y * i));
    if (y) {
      XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
		  0, j - (y * i), w->core.width, j - (y * i));
    }
  }
}

static char *si_label(float value)
{
  float sign;
  static char buf[50];

  if (value >= 0) {
    sign = 1.0;
  } else if (value < 0) {
    sign = -1.0;
    value = -value;
  }

  if (value <= 1.0e-3 && value > 1.0e-6) {
    sprintf(buf, "%.1fm", sign * (value / 1.0e-3));
  } else if (value <= 1.0e-6) {
    sprintf(buf, "%.1fu", sign * (value / 1.0e-6));
  } else if (value >= 1.0e3 && value < 1.0e6) {
    sprintf(buf, "%.1fK", sign * (value / 1.0e3));
  } else if (value > 1.0e6) {
    sprintf(buf, "%.1fM", sign * (value / 1.0e6));
  } else {
    sprintf(buf, "%.1f", sign * value);
  }
  return(buf);
}

static void DrawYAxisLabels(WaveWidget w, int maxval, int minval)
{
	char *min,*max;

	min = si_label((float)minval);
	max = si_label((float)maxval);

  XPS_DrawString(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
	      0, 10, max, strlen(max));
  XPS_DrawString(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
	      0, avail_height(w), min, strlen(min));
}

static void Redisplay(WaveWidget w, XEvent *event, Region region)
{
  register int i, k, x, step, y, y2, range;
  register int max, min, ymin, dmax, dmin, last_min, last_max;
  register int begin, end, npoints;
  int range2, scale, offset;
  char lstr[100];
  XPoint *lines;
  XSegment *segs;
  int nlines, nsegs;

  if (w->wave.show_grid)
    DrawGrid(w);
  if (w->wave.display_info)
    *lstr = 0;
  if (w->wave.show_y_axis)
    DrawYAxisLabels(w, dmax, dmin);

  if (w->wave.waveform != NULL) {
    begin = w->wave.wave_begin;
    end = w->wave.wave_end;
    if (begin < 0)
      begin = 0;
    if (end < 0 || end > w->wave.npoints)
      end = w->wave.npoints;
    
    npoints = end - begin;
    offset = 0;

    range = w->wave.ymax - w->wave.ymin; /* user-set range */
    if (w->wave.auto_scale) {		/* 0==no scaling at all */
      if (w->wave.auto_scale == 1) { 	/* 1==auto_scale for whole waveform */
	dmax = w->wave.data_max;
	dmin = w->wave.data_min;
      } else {				/* 2==auto_scale for visible portion */
	dmax = dmin = w->wave.waveform[begin];
	for (i = begin + 1; i < end; i++) {
	  if (w->wave.waveform[i] > dmax)
	    dmax = w->wave.waveform[i];
	  else if (w->wave.waveform[i] < dmin)
	    dmin = w->wave.waveform[i];
	}
      }
      if (!w->wave.auto_range) {
	if (mag(dmax) > mag(dmin)) {
	  if (dmax > 0) {
	    min = -dmax;
	    max = dmax;
	  } else {
	    min = dmax;
	    max = -dmax;
	  }
	} else {
	  if (dmin > 0) {
	    min = -dmin;
	    max = dmin;
	  } else {
	    min = dmin;
	    max = -dmin;
	  }
	}
      } else {
	min = dmin;
	max = dmax;
	offset = (min + max) / 2;
      }
      range2 = max - min;
      dmax = max;		/* save for axis */
      dmin = min;		/* save for axis */

      if (range2 > range) {
	ymin = w->wave.ymin;
	if (w->wave.display_info)
	  sprintf(lstr + strlen(lstr), "*CLIP* ");
      } else if (range2 != 0) {
	scale = (int)((float)range / (float)range2);
	if (w->wave.display_info) {
	  if (offset) 
	    sprintf(lstr + strlen(lstr),
		    "x%d off=%d ", scale, offset);
	  else
	    sprintf(lstr + strlen(lstr), "x%d ", scale);
	}
	range = (int)(0.5 + (float)range / (float) scale);
	ymin = min - ((range - range2) / 2.0);
      } else {
	ymin = w->wave.ymin;
	if (w->wave.display_info)
	  sprintf(lstr + strlen(lstr), "RANGELESS ");
      }
    } else {
      ymin = w->wave.ymin;
      dmax = w->wave.ymax;	/* save for axis */
      dmin = w->wave.ymin;	/* save for axis */
    }

    if (w->wave.display_info) {
      if (w->wave.fc >= 0)
	sprintf(lstr + strlen(lstr), "%d-%dms ",
		begin * 1000 / w->wave.fc, end * 1000 / w->wave.fc);
      else
	sprintf(lstr + strlen(lstr), "%d-%dticks ", begin, end);
      if (w->wave.auto_scale == 0)
	strcat(lstr, "Noscale ");
      else if (w->wave.auto_scale == 1)
	strcat(lstr, "Scale ");
      else if (w->wave.auto_scale == 2)
	strcat(lstr, "ScaleVis ");
      if (w->wave.auto_range)
	strcat(lstr, "AutoRange ");
      if (w->wave.use_lines)
	strcat(lstr, "All ");
      else
	strcat(lstr, "Mm ");

      if (w->wave.cursor1 > 0) {
	if (w->wave.fc >= 0)
	  sprintf(lstr+strlen(lstr),"C1=%dms ",
		  w->wave.cursor1*1000/w->wave.fc);
	else
	  sprintf(lstr+strlen(lstr),"C1=:%dtk ",
		  w->wave.cursor1);
      }

      if (w->wave.cursor2 > 0) {
	if (w->wave.fc >= 0)
	  sprintf(lstr+strlen(lstr),"C2=%dms",
		  w->wave.cursor2*1000/w->wave.fc);
	else
	  sprintf(lstr+strlen(lstr),"C2=%dtk",
		  w->wave.cursor2);
      }


      i = avail_height(w) + 1;
      XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
		  			0, i, w->core.width, i);
      XPS_DrawString(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
		  			0 + 2, w->core.height - 1, lstr, strlen(lstr));
    }

    if (npoints < w->core.width || w->wave.use_lines) {
      nlines = 0;
      lines = (XPoint *) malloc(npoints * sizeof(XPoint));
      for (i = 0; i < npoints; i++) {
	x = i * w->core.width / npoints;
	y = avail_height(w) - 
	  (avail_height(w) * (w->wave.waveform[i + begin] - ymin) / range);
	if (y > avail_height(w))
	  y = avail_height(w);
	else if (y < 0)
	  y = 0;
	if (region == NULL || XPointInRegion(region, x, y)) {
	  lines[nlines].x = x;
	  lines[nlines].y = y;
	  nlines++;
	}
      }
      if (nlines > 0)
	XPS_DrawLines(XtDisplay(w), XtWindow(w), w->wave.gc,
		     lines, nlines, CoordModeOrigin);

      free(lines);
    } else {
      int a, b;
      nsegs = 0;
      segs = (XSegment *) malloc(w->core.width * sizeof(XSegment));
      step = npoints / w->core.width;

      for (i = 0; i < w->core.width; i++) {
	a = begin + npoints * i / w->core.width;
	b = begin + npoints * (i + 1) / w->core.width;
	max = min = w->wave.waveform[a];
	for (k = a + 1; k < b; k++) {
	  if (w->wave.waveform[k] > max)
	    max = w->wave.waveform[k];
	  else if (w->wave.waveform[k] < min)
	    min = w->wave.waveform[k];
	}
	if (i != 0) {
	  if (last_max < min)
	    min = last_max;
	  else if (last_min > max)
	    max = last_min;
	}
	last_min = min;
	last_max = max;
	
	y = avail_height(w) - (avail_height(w) * (max - ymin) / range);
	if (y > avail_height(w))
	  y = avail_height(w);
	else if (y < 0)
	  y = 0;
	
	y2 = avail_height(w) - (avail_height(w) * (min - ymin) / range);
	if (y2 > avail_height(w))
	  y2 = avail_height(w);
	else if (y2 < 0)
	  y2 = 0;

	segs[nsegs].x1 = segs[nsegs].x2 = i;
	segs[nsegs].y1 = y;
	segs[nsegs].y2 = y2;
	nsegs++;
      }
      XPS_DrawSegments(XtDisplay(w), XtWindow(w), w->wave.gc, segs, nsegs);
      free(segs);		/* this was missing! big memory leak */
    }


    DrawCursors(w, 0, begin, end, npoints);

    if (w->wave.show_h_cursor1) {
      y = avail_height(w) - 
	(avail_height(w) * (w->wave.hcursor1 - ymin) / range);
      if (y >= 0 && y < avail_height(w))
	XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.grid_gc,
		    0, y, w->core.width, y);
    }
    
    if (w->wave.show_h_cursor2) {
      y = avail_height(w) - 
	(avail_height(w) * (w->wave.hcursor2 - ymin) / range);
      if (y >= 0 && y < avail_height(w))
	XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.grid_gc,
		    0, y, w->core.width, y);
    }
  } else {
    /* I think this clause should JUST GO AWAY.. */
    if (w->wave.npoints > 0)
      DrawCursors(w, 0, 0, w->wave.npoints, w->wave.npoints);

    XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.gc,
		0, 0, w->core.width, avail_height(w));
    XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.gc,
		w->core.width, 0, 0, avail_height(w));

    if (w->wave.display_info) {
      i = avail_height(w) + 1;
      XPS_DrawLine(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
		  			0, i, w->core.width, i);
      XPS_DrawString(XtDisplay(w), XtWindow(w), w->wave.grid_gc, 
		  			0 + 2, w->core.height - 1, "No Data", 7);
    }
  }
}

static void forceupdate(WaveWidget w)
{
  if (XtIsRealized(w)) {
    XClearWindow(XtDisplay(w), XtWindow(w));
    Redisplay(w, NULL, NULL);
  }
}

static void Destroy(WaveWidget w)
{
  if (w->wave.waveform != NULL)
    free(w->wave.waveform);
  XFreeFont(XtDisplay(w), w->wave.fontinfo);
  XFreeGC(XtDisplay(w), w->wave.gc);
  XFreeGC(XtDisplay(w), w->wave.grid_gc);
}

static void Resize(WaveWidget w)
{
  Redisplay(w, NULL, NULL);
}

static Boolean SetValues(WaveWidget current, WaveWidget request, WaveWidget new)
{
  if ((current->wave.hcursor1 != new->wave.hcursor1) ||
      (current->wave.hcursor2 != new->wave.hcursor2) ||
      (current->wave.show_h_cursor1 != new->wave.show_h_cursor1) ||
      (current->wave.show_h_cursor2 != new->wave.show_h_cursor2))
    return(True);		/* do redisplay! */
  else {
    if (current->wave.cursor1 != new->wave.cursor1)
      return(True);
    if (current->wave.cursor2 != new->wave.cursor2)
      return(True);
    return(False);
  }
}

static void InputAction(WaveWidget w, XEvent *event, String *params,
      Cardinal *num_params)
/*
**   String *params;		ignored
**   Cardinal *num_params;	ignored
*/
{
  XtCallCallbacks(w, XtNcallback, (caddr_t) event);
}

static void ToggleLinesAction(WaveWidget w, XEvent *event, String *params,
      Cardinal *num_params)
/*
**   String *params;          ignored
**   Cardinal *num_params;    ignored
*/
{
  w->wave.use_lines = !w->wave.use_lines;
  forceupdate(w);
  if (w->wave.link_to != NULL) {
    if ((w->wave.link_to)->wave.link_hit == 0) {
      w->wave.link_hit = 1;
      ToggleLinesAction(w->wave.link_to, event, params, num_params);
      w->wave.link_hit = 0;
    }
  }
}

static void ChangeAutoScaleAction(WaveWidget w, XEvent *event, String *params,
      Cardinal *num_params)
/*
**   String *params;         ignored
**   Cardinal *num_params;   ignored
*/
{
  w->wave.auto_scale = (w->wave.auto_scale + 1) % 3;
  forceupdate(w);
  if (w->wave.link_to != NULL) {
    if ((w->wave.link_to)->wave.link_hit == 0) {
      w->wave.link_hit = 1;
      ChangeAutoScaleAction(w->wave.link_to, event, params, num_params);
      w->wave.link_hit = 0;
    }
  }
}

#define BEGIN	((w->wave.wave_begin < 0) ? 0 : w->wave.wave_begin)
#define END 	((w->wave.wave_end < 0) ? w->wave.npoints : w->wave.wave_end)

int XtWaveEventPosition(WaveWidget w, XEvent *event)
{
  return(BEGIN + ((END - BEGIN) * event->xbutton.x / w->core.width));
}
#undef BEGIN
#undef END


static void SetCursorAction(WaveWidget w, XEvent *event, String *params,
      Cardinal *num_params)
/*
     String *params;		ignored
     Cardinal *num_params;	ignored
*/
{
  int begin, end, x, c1, c2, c;

  if (w->wave.user_cursors) {
    if (w->wave.wave_begin < 0)
      begin = 0;
    else
      begin = w->wave.wave_begin;
    
    if (w->wave.wave_end < 0)
      end = w->wave.npoints;
    else
      end = w->wave.wave_end;

    if (*num_params == 2 && strcmp(params[0], "set")) {
      c = atoi(params[1]);
      x = begin + ((end - begin) * event->xbutton.x / w->core.width);
      c1 = w->wave.cursor1;
      c2 = w->wave.cursor2;
      
      if (c == 1) {
	c1 = x;
	if (c2 >= 0 && c2 < c1) {
	  c1 = c2;
	  c2 = x;
	}
      } else if (c == 2) {
	c2 = x;
	if (c1 >= 0 && c2 < c1) {
	  c2 = c1;
	  c1 = x;
	}
      } else {
	return;
      }
      if (c1 != w->wave.cursor1 || c2 != w->wave.cursor2) {
	w->wave.cursor1 = c1;
	w->wave.cursor2 = c2;
	XClearWindow(XtDisplay(w), XtWindow(w));
	Redisplay(w, NULL, NULL);
      }
    } else if (*num_params == 1 && strcmp(params[0], "clear") == 0) {
      w->wave.cursor1 = -1;
      w->wave.cursor2 = -1;
      XClearWindow(XtDisplay(w), XtWindow(w));
      Redisplay(w, NULL, NULL);
    }
    if (w->wave.link_to != NULL) {
      if ((w->wave.link_to)->wave.link_hit == 0) {
	w->wave.link_hit = 1;
	SetCursorAction(w->wave.link_to, event, params, num_params);
	w->wave.link_hit = 0;
      }
    }
  } else {
    XBell(XtDisplay(w), 0);
  }
}

static int xzoomPushPop(WaveWidget w, int push)
{
  WaveState *z;
  
  if (push) {
    z = (WaveState *)malloc(sizeof(WaveState));
    z->begin = w->wave.wave_begin;
    z->end = w->wave.wave_end;
    z->next = w->wave.xzoomstack;
    w->wave.xzoomstack = z;
    return(1);
  } else {
    if (w->wave.xzoomstack != NULL) {
      z = w->wave.xzoomstack;
      w->wave.wave_begin = z->begin;
      w->wave.wave_end = z->end;
      w->wave.xzoomstack = z->next;
      free(z);
      return(1);
    } else
      return(0);
  }
}

static void ChangeDisplayAction(WaveWidget w, XEvent *event, String *params,
      Cardinal *num_params)
/*
     String *params;		ignored
     Cardinal *num_params;	ignored
*/
{
  int begin, end, ctr, diff, cur;
  int propagate = 1;

  begin = (w->wave.wave_begin < 0) ? 0 : w->wave.wave_begin;
  end = (w->wave.wave_end < 0) ? w->wave.npoints : w->wave.wave_end;

  if (*num_params > 0) {
    if (strcmp(params[0], "xzoom") == 0) {
      /* xzoom:
       * if cursors are set, use cursor as zoom region,
       * overwise just zoom in on displayed region
       */
      cur = 0;
      if (w->wave.cursor1 > 0 && w->wave.cursor1 > begin) {
	begin = w->wave.cursor1;
	cur++;
      }
      if (w->wave.cursor2 > 0 && w->wave.cursor2 < end) {
	end = w->wave.cursor2;
	cur++;
      }
      if (cur == 2 && begin >= end) 	/* if cursors 1 and 2 and swapped.. */
	cur = 0;			/* act as if no cursorss.. */
      else if (cur > 0) {		/* if either one or two cursors, */
	if (begin > 1)			/* leave some room to see them */
	  begin--;
	if (end < w->wave.npoints)
	  end++;
      }
      xzoomPushPop(w, True);	/* push current view onto stack */
      if (cur == 0) {		/* setup new view w/o respect to cursors */
	diff = (end - begin) / 4;
	ctr = (end + begin) / 2;
	if (diff > 0) {
	  w->wave.wave_begin = ctr - diff;
	  w->wave.wave_end = ctr + diff;
	}
      } else {			/* setup new view w/ respect to cursors */
	w->wave.wave_begin = begin;
	w->wave.wave_end = end;
      }
      forceupdate(w);
    } else if (strcmp(params[0], "unxzoom") == 0) {
      if (xzoomPushPop(w, False)) {
	forceupdate(w);
      } else {
	XBell(XtDisplay(w), 0);
	propagate = 0;
      }
    } else if (strcmp(params[0], "unxzoomall") == 0) {
      while (xzoomPushPop(w, False))
	;
      forceupdate(w);
    } else if (strcmp(params[0], "panleft") == 0) {
      diff = end - begin;
      if ((begin = begin - diff) < 0)
	begin = 0;
      end = begin + diff;
      w->wave.wave_begin = begin;
      w->wave.wave_end = end;
      forceupdate(w);
    } else if (strcmp(params[0], "panright") == 0) {
      diff = end - begin;
      if ((end = end + diff) > w->wave.npoints)
	end = w->wave.npoints;
      begin = end - diff;
      w->wave.wave_begin = begin;
      w->wave.wave_end = end;
      forceupdate(w);
    } else if (strcmp(params[0], "panhome") == 0) {
      diff = end - begin;
      begin = 0;
      if ((end = begin + diff) > w->wave.npoints)
	end = w->wave.npoints;
      w->wave.wave_begin = begin;
      w->wave.wave_end = end;
      forceupdate(w);
    } else if (strcmp(params[0], "panend") == 0) {
      diff = end - begin;
      end = w->wave.npoints;
      if ((begin = end - diff) < 0)
	begin = 0;
      w->wave.wave_begin = begin;
      w->wave.wave_end = end;
      forceupdate(w);
    } else if (strcmp(params[0], "scrollleft") == 0) {
      diff = end - begin;
      if ((begin = begin - 10) < 0)
	begin = 0;
      end = begin + diff;
      w->wave.wave_begin = begin;
      w->wave.wave_end = end;
      forceupdate(w);
    } else if (strcmp(params[0], "scrollright") == 0) {
      diff = end - begin;
      if ((end = end + 10) > w->wave.npoints)
	end = w->wave.npoints;
      begin = end - diff;
      w->wave.wave_begin = begin;
      w->wave.wave_end = end;
      forceupdate(w);
    } else {
      XBell(XtDisplay(w), 0);
      propagate = 0;
    }
  } else {
    XBell(XtDisplay(w), 0);
    propagate = 0;
  }
  
    
  if (propagate && w->wave.link_to != NULL) {
    if ((w->wave.link_to)->wave.link_hit == 0) {
      w->wave.link_hit = 1;
      ChangeDisplayAction(w->wave.link_to, event, params, num_params);
      w->wave.link_hit = 0;
    }
  }
}

static void ToggleAutoRangeAction(WaveWidget w, XEvent *event, String *params,
     Cardinal *num_params)
/*
     String *params;		ignored
     Cardinal *num_params;	ignored
*/
{
  w->wave.auto_range = !w->wave.auto_range;
  forceupdate(w);
  if (w->wave.link_to != NULL) {
    if ((w->wave.link_to)->wave.link_hit == 0) {
      w->wave.link_hit = 1;
      ToggleAutoRangeAction(w->wave.link_to, event, params, num_params);
      w->wave.link_hit = 0;
    }
  }
}

void XtWavePS(WaveWidget w, char *destination, char *titlestring)
{
  if (destination != NULL) {
    XPS_SetFileOutput(1);	/* 1=file, 0=printer */
    XPS_SetFilename(destination);
  } else {
    XPS_SetFileOutput(0);	/* 1=file, 0=printer */
  }

  XPS_PreparePS(XtDisplay(w), XtWindow(w), (double) 0.80, 0, titlestring, 0);
  Redisplay(w, NULL, NULL);
  XPS_FinishPS();
}

static void ToggleInfoAction(WaveWidget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  w->wave.display_info = !w->wave.display_info;
  forceupdate(w);
  if (w->wave.link_to != NULL) {
    if ((w->wave.link_to)->wave.link_hit == 0) {
      w->wave.link_hit = 1;
      ToggleInfoAction(w->wave.link_to, event, params, num_params);
      w->wave.link_hit = 0;
    }
  }
}

static void ToggleGridAction(WaveWidget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  w->wave.show_grid = !w->wave.show_grid;
  forceupdate(w);
  if (w->wave.link_to != NULL) {
    if ((w->wave.link_to)->wave.link_hit == 0) {
      w->wave.link_hit = 1;
      ToggleGridAction(w->wave.link_to, event, params, num_params);
      w->wave.link_hit = 0;
    }
  }
}

static XtActionsRec actions[] = {
  { "wave-input",	InputAction 		},
  { "wave-autoscale",	ChangeAutoScaleAction	},
  { "wave-autorange",	ToggleAutoRangeAction 	},
  { "wave-change",	ChangeDisplayAction 	},
  { "wave-setcursor",	SetCursorAction 	},
  { "wave-toggleinfo",	ToggleInfoAction	},
  { "wave-togglegrid",	ToggleGridAction 	},
  { "wave-togglelines",	ToggleLinesAction 	},
  { "wave-ps",	  	GeneratePSAction	},
  { "wave-fig",	  	GenerateFIGAction	},
};

static char default_translations[] = "\
     <KeyPress>space:	wave-toggleinfo()	\n\
         <KeyPress>g:	wave-togglegrid()	\n\
         <KeyPress>l:	wave-togglelines()	\n\
\
         <KeyPress>s:	wave-autoscale()	\n\
         <KeyPress>r:	wave-autorange()	\n\
\
         <KeyPress>z:	wave-change(xzoom)	\n\
     None<KeyPress>u:	wave-change(unxzoom)	\n\
    Shift<KeyPress>U:	wave-change(unzxoomall)	\n\
\
  Meta<KeyPress>Left:   wave-change(scrollleft)	\n\
 Meta<KeyPress>Right:   wave-change(scrollright)\n\
  None<KeyPress>Left:   wave-change(panleft)	\n\
 None<KeyPress>Right:   wave-change(panright)	\n\
      <KeyPress>Home:   wave-change(panhome)	\n\
       <KeyPress>End:   wave-change(panend)	\n\
\
      Meta<Btn1Down>:	wave-setcursor(set,1)	\n\
      Meta<Btn2Down>:	wave-setcursor(set,2)	\n\
      Meta<Btn3Down>:	wave-setcursor(clear)	\n\
\
          <KeyPress>:	wave-input()\n\
     Shift<KeyPress>:	wave-input()\n\
      None<Btn1Down>:	wave-input()\n\
      None<Btn2Down>:	wave-input()\n\
\
          <Key>Print:	wave-ps()\n\
      Meta<Key>Print:	wave-ps(file)\n\
     Shift<Key>Print:	wave-fig()\n\
          Meta<Key>C:	wave-fig(file)\n";


WaveClassRec waveClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &widgetClassRec,
    /* class_name		*/	"Wave",
    /* widget_size		*/	sizeof(WaveRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	False,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	True,
    /* compress_exposure	*/	True,
    /* compress_enterleave	*/	True,
    /* visible_interest		*/	False,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	default_translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* wave fields */
    /* empty			*/	0
  }
};

WidgetClass waveWidgetClass = (WidgetClass)&waveClassRec;

void XtWaveSetWaveform(WaveWidget w, xword *data, int npoints, int skip,
      int ymin, int ymax)
{
  XtWaveSetWaveformOffset(w, data, npoints, skip, ymin, ymax, 0);
}

void XtWaveSetWaveformOffset(WaveWidget w, xword *data, int npoints, int skip,
      int ymin, int ymax, int offset)
{
  register int i, j;

  if (data == NULL) {
    if (w->wave.waveform) {
      free(w->wave.waveform);
      w->wave.waveform = NULL;
    }
    w->wave.npoints = 0;
  } else {
    if (w->wave.waveform != NULL && w->wave.npoints != npoints) {
      free(w->wave.waveform);
      w->wave.waveform = NULL;
    }
    if (w->wave.waveform == NULL) {
      w->wave.waveform = (xword *)malloc(npoints * sizeof(xword));
      w->wave.npoints = npoints;
    }
    w->wave.data_min = w->wave.data_max = w->wave.waveform[0] = data[0];
    for (j = skip, i = 1; i < npoints; i++, j += skip) {
      w->wave.waveform[i] = data[j] - offset;
      if (w->wave.waveform[i] > w->wave.data_max)
	w->wave.data_max = w->wave.waveform[i];
      else if (w->wave.waveform[i] < w->wave.data_min)
	w->wave.data_min = w->wave.waveform[i];
    }
  }
  w->wave.ymin = ymin;
  w->wave.ymax = ymax;

  forceupdate(w);
}

static void UnlinkCB(WaveWidget w2, WaveWidget w, caddr_t call_data)
{
  w->wave.link_to = w2->wave.link_to;
}

void XtWaveLink(WaveWidget w1, WaveWidget w2, int bidir)
{
  w1->wave.link_to = w2;
  XtAddCallback(w2, XtNdestroyCallback, UnlinkCB, w1);
  if (bidir) {
    w2->wave.link_to = w1;
    XtAddCallback(w1, XtNdestroyCallback, UnlinkCB, w2);
  }
}


static char *getafilename(char *startwith, char *prompt)
{
  char *name, *fileBox();

  if ((name = fileBox(startwith, "a", prompt)) != NULL) {
    if (ishiddenfile(name) &&
	pop_box("Write to hidden file?", "No", "Yes", NULL) == 1) {
      free(name);
      return(NULL);
    } else {
      return(name);
    }
  } else {
    return(NULL);
  }
}

static void GeneratePSAction(WaveWidget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  char *name;

  if (*num_params > 0 && strcasecmp(params[0], "file") == 0) {
    if ((name = getafilename(".ps", "Save PostScript to:")) != NULL) {
      XtWavePS(w, name, NULL);
      free(name);
    }
  } else {
    XtWavePS(w, NULL, name);
  }
}

static void GenerateFIGAction(Widget w, XEvent *event, String *params,
      Cardinal *num_params)
{
  char *name;
  WidgetClass class;

  class = XtClass(w);
  if (*num_params > 0 && strcasecmp(params[0], "file") == 0) {
    if ((name = getafilename(".fig", "Save Fig to:")) == NULL) {
      return;
    } else {
      XPS_SetFilename(name);
    }
  } else {
    unlink(findhome("~/.xfig"));
    XPS_SetFilename(findhome("~/.xfig"));
  }
  XtUnmapWidget(w);
  XPS_PrepareFIG(XtDisplay(w), XtWindow(w));
  XtMapWidget(w);
  dloop_empty();
  XPS_FinishFIG();
  free(name);
  XtUnmapWidget(w);
  XtMapWidget(w);
}
