function [depints,driven,spont]=stimpower(filename, start_ms, end_ms, fc)
% USAGE: STIMPOWER('FILENAME', START_MS, END_MS, FC);
%                    -or-
%        [DEPINTS,DRIVEN] = STIMPOWER('FILENAME', START_MS, END_MS, FC);
%                    -or-
%        [DEPINTS,DRIVEN,SPONT] = STIMPOWER('FILENAME', START_MS, END_MS, FC);
%
%  STIMPOWER plots the value of the FFT of an averaged analog trace 
%  (averaged across trials to eliminate spikes) at the stimulus 
%  frequency across a range of ITDs.
%
%  For the red line and points, this value is calculated for the region 
%  of the trace which lies between START_MS and END_MS milliseconds.
%
%  For the blue line and points, this value is calculated for the region 
%  of the trace which lies between 1 and 100 milliseconds (spontaneous
%  response).
%
%  Inputs: 
%
%    FILENAME:     The name of an xdphys file containing analog data.
%
%    START_MS:     Start of the region to be inspected in milliseconds.
%
%    END_MS:       Start of the region to be inspected in milliseconds
% 
%    FC:       Sampling frequency for the traces, in kHz.
%
%  Outputs: 
%
%    DEPINTS:     An ordered list of ITDs 
%
%    DRIVEN:      The associated FFT values for the driven region.
%
%    SPONT:       The associated FFT values for the spontaneous region.
%
%  NOTE: This function will probably only work for new xdphys files, and
%        only if the stimulus is a tone.
%  


start_index = start_ms*fc;
end_index = end_ms*fc;
resting_start = 1;
resting_end = 100*fc;

stim_freq = xdview_stim(filename);

data = readana(filename);

for i = 1:length(data),
	traces = data(i).traces;

	stim = traces(start_index:end_index,:)/10;
	rest = traces(resting_start:resting_end,:)/10;

	power(i)=stim_power(stim,fc,stim_freq);
	rest_power(i)=stim_power(rest,fc,stim_freq);

	depints(i)=data(i).depint;
end

clf;
plot(depints,power,'r-o',depints,rest_power,'b--o');
%errorbar(depints,mean(rest_power'), std(rest_power'),'b--o');
xlabel('ITD (microseconds)');
ylabel('FFT value');
title(sprintf('%s: FFT value at %dHz vs ITD', filename, stim_freq));
legend('driven','spont');

if (nargout==3),
	depvars=depints;
	driven=power;
	spont=rest_power;
elseif (nargout == 2),
	depvars=depints;
	driven=power;
end


%===================================================
function power = stim_power(traces, fc, stim_freq)
% return the power at frequency of stim_freq

traces_fft=abs(fft(mean(traces')));
%traces_fft=abs(fft(traces'));

fc_half=(fc*1000)/2;
halfway=round(length(traces_fft)/2);
sf_index=round((stim_freq*halfway)/fc_half);

power=traces_fft(sf_index);

%===================================================
function stim = xdview_stim(filename)

[s,w] = unix(['xdview -params ' filename]);

t = findstr(w,'stim=');
tt = findstr(w(t(1)+5:end),char(10));
stimspec = w(t(1)+5:t(1)+5+tt(1));

% Just assume it's a tone for now
stim = sscanf(stimspec,'%f');
