function data=readana(filename, depvar)
% USAGE: DATA = READANA('FILENAME');
%        DATA = READANA('FILENAME', DEPVAR);
%
%   READANA reads analog traces from an xdphys data file.  It can read
%   both the old style .ana files, and the newer "gen" style files. 
%
%   DATA is an array of structs, where N is the number of different
%   depvars.  Each struct has the following structure:
%
%        data.depint:    depvar 
%        data.traces:    MxN matrix of analog traces for data.depint 
%   
%   Each trace in data.traces M=(samp_freq * epoch/1000) samples long,
%	and is in mV.
%
%   If the parameter DEPVAR is supplied, READANA only reads those traces
%   whose depvar equals DEPVAR (and DATA is 1x1).  If DEPVAR is not supplied, 
%	all traces in the file are read (and DATA is 1xN, N=(# depvars)).
%
%	DATA is sorted in order of increasing depvar.
%
%   NEW FILES:
%   ---------
%
%       If you used the new xdphys with the gen module to save your data,
%       your analog data is stored in the .gen/.itd/.abi/etc. file.  Simply
%       use READANA on that file to extract the analog data.
%
%   OLD (".ana") FILES: 
%   ------------------
%
%       If you are trying to read the OLD style files, you must uncompress 
%       the analog datafile before using READANA: if it has .Z or .gz at the 
%       end of the filename, it is compressed.
%
%       Outside of matlab, run
%
%            uncompress <datafile>  (for .Z files)
%	         gunzip <datafile>  (for .gz files)
%
%       and then use READANA in matlab to read the resulting file.
%
% SEE ALSO: XDVIEW_CURVE, XDVIEW_GETCAL

% NOTES:
%
% Two kinds of analog files: pre-gen module and post-gen module.
%
%  An old style header struct looks like, for dowl, and pre-gen module xdphys:
%
%   int        magic   ( = 0x5731 )
%   int        fc
%   int        nsamps
%   int        nchans
%   char[100]  comment
%   int        bits
%   char[20]   program
%   char[20]   arch
%   int        tomv
%   int        tomv_div
%   int        offset
%   int        offset_div
%   char[336]  padding
%
%  It is 512 bytes long, and is defined in xdphys/src/waveio.h
%  xdphys is probably compiled to align structs along 4 byte boundries,
%  so types with size < 4 get converted to 4 bytes in the header. This 
%  header preceeds each trace.  Old style ana files are always written in
%  big endian format.
%
%  A post-gen module xdphys header looks like:
%
%	int magic                ( = 0x5732 )
%	char adFc[15]            ( = "adFc=%d" in Hz)
%	char epoch[15]           ( = "epoch=%d in ms)
%	char decimate[15]        ( = "ana-decimate=%d", boolean (1/0))
%	char tomv[30]            ( = "ana-tomv=%f"  in mV/tick)
%	char offset[30]          ( = "ana-offset=%f" in mV)
%	char nrasters[15]        ( = "nrasters=%d" )
%
%  This header appears at the beginning of the file only.  New style ana files
%  are written in whatever was the default endianness for the machine on
%  which it was created.
%

if(nargin<2)
  depvar = 'all';
end

[t, fid]=file_type(filename,depvar);

switch t
	case 0,
		[d, depints]=read_old_style(fid, depvar);
	case 1,
		[d, depints]=read_new_style(fid, depvar);
end
data=make_struct(d,depints);


% ---------------------------------------------------
function data=make_struct(d,depints)

% make_struct turns the matrix D and the vector DEPINTS into a vector
%   of structs DATA.  Each struct in DATA has the following form:
%
%        data.depint:    depvar 
%        data.traces:    MxN matrix of analog traces for data.depint 
%   
% make_struct assumes that d and depints are sorted.

i=0;
rep=1;
last_depint=-9999;
for j=1:length(depints),
	if (depints(j) ~= last_depint)
		i=i+1;
		rep=1;
		data(i).depint=depints(j);
	end
	data(i).traces(:,rep)=d(:,j);
	rep=rep+1;
	last_depint=depints(j);
end
	
% ---------------------------------------------------
function [data, depints]=read_old_style(fid, depvar);

% read_old_style expects that the first magic has been read when it
% begins.

count=1;
trace_num=1;
while (count == 1),
	fc = fread(fid,1,'int');
	nsamps = fread(fid,1,'int');
	nchans = fread(fid,1,'int');
	temp=sscanf(setstr(fread(fid,100,'char')'),'depvar=%d');
	if (strcmp(depvar,'all') | (depvar == temp))
		dp(trace_num)=temp;
		pad = fread(fid,44,'char');
		to_mv = fread(fid,1,'int');
		to_mv_div = fread(fid,1,'int');
		offset = fread(fid,1,'int');
		offset_div = fread(fid,1,'int');
		pad = fread(fid,336,'char')';

		trace=fread(fid,nsamps*nchans,'short');

		data(:,trace_num) = (trace - (offset/offset_div))*(to_mv/to_mv_div);
		disp(sprintf('Read trace for depvar=%d', dp(trace_num)));
		trace_num = trace_num + 1;
	else
		pad=fread(fid,198+(nsamps*nchans),'short');
	end;
	[magic, count] = fread(fid,1,'int');
end;

% now sort the data and depints array
[depints, i]=sort(dp);
data=data(:,i);

% ---------------------------------------------------
function [data, depints]=read_new_style(fid, depvar);

% read_new_style expects that magic has been read when it
% begins.

head.adFc=sscanf(setstr(fread(fid,15,'char')'),'adFc=%d');
head.epoch=sscanf(setstr(fread(fid,15,'char')'),'epoch=%d');
head.decimate=sscanf(setstr(fread(fid,15,'char')'),'ana-decimate=%d');
head.tomv=sscanf(setstr(fread(fid,30,'char')'),'ana-tomv=%f');
head.offset=sscanf(setstr(fread(fid,30,'char')'),'ana-offset=%f');
head.nrasters=sscanf(setstr(fread(fid,15,'char')'),'nrasters=%d');

disp(sprintf('Reading %d traces.', head.nrasters));

nsamps=((head.adFc*head.epoch)/1000)/head.decimate;

trace_num=1;
max=head.nrasters;
for n = 1:max,
	depint=sscanf(setstr(fread(fid,30,'char')'),'depvar=%d');
	trace=fread(fid,nsamps,'short');
	if (strcmp(depvar,'all') | (depint == depvar))
		depints(trace_num)=depint;
		data(:,trace_num) = (trace*head.tomv)+head.offset;
		disp(sprintf('Read trace for depvar=%d', depints(trace_num)));
		trace_num=trace_num+1;
	end
end


% ---------------------------------------------------
function [t, fid]=file_type(filename,depvar)
%
%  t=0   if file is old style file
%  t=1   if file is new style file
%  t=2   if file is new style file, but we created it via xdview


% First, try to open it as an old style ana file
% Old style ana files were in big-endian format.

[magic, fid]=get_magic(filename, 'b');

disp(' ');
if (bitand(magic,hex2dec('ffff0000')) == hex2dec('57310000')),
	% this is an old style file
	disp([filename ' is an old style file.']);
	t=0;
else
	if (magic == hex2dec('5732')),
		% this is an new style file, written on a big endian machine
		disp([filename ' is output from "xdview -ana", big endian.']);
		t=1;
	else
		% try to reopen it as little endian and check the magic
		fclose(fid);
		[magic, fid]=get_magic(filename, 'l');
		if (magic == hex2dec('5732')),
			disp([filename ' is output from "xdview -ana", little endian.']);
			t=1;
		end
	end
end
disp(' ');


% ---------------------------------------------------
function [magic, fid]=get_magic(filename, opt)

fid=fopen(filename, 'r', opt);
if fid == (-1),
	error(['Unable to open file ', filename]);
end;

[magic,count] = fread(fid,1,'int');

