function [p,sig,tt,wr]=itdscatter(pre, post, alpha);
% USAGE: [P,SIG,TT,WR]=ITDSCATTER(FILENAME1, FILENAME2, ALPHA)
%
%   ITDSCATTER extracts the spike rates for each trial over a range of depvars 
%   from two xdphys data files, FILENAME1 and FILENAME2, and plots these spike 
%   rates (trial by trial) against each other.  
%
%   The blue line is the average of NREPS lines fit to the NREPS repetitions
%   of (nspikes_pre,nspikes_pre) by the least squares method.  Likewise, the
%   pink line is the average of nreps lines fit to the nreps repetitions of
%   (nspikes_pre,nspikes_post) by the least squares method.
%
%   "b" is the y-intercept of the pink line, with a variance estimate.
%
%   "m" is the slope of the pink line, with a variance estimate.
%   
%   PARAMETERS:
%
%       FILENAME1, FILENAME2 are xdphys data files (*.itd, *.abi, etc.)
%
%       (optional) ALPHA is the desired level of significance for the 
%         t-tests and the Wilcoxon rank test, and must be between 0 and 1 
%         (default = 0.05).
%
%   RETURN VALUES:
% 
%       P contains the parameters of the equation of the pink line:
%
%            Y = P(1)*X + P(2)
%
%       SIG contains the variance estimates for the parameters in P.
%
%       TT contains the t-test results (see below) for the parameters in P.
%
%       WR contains the Wilcoxon rank test results (see below) for the
%          parameters in P.
%
%   VALIDITY TESTING:
%   ----------------
%
%     ITDSCATTER performs both t-tests and Wilcoxon ranks test on the 
%     line coordinates.  
%
%     T-TESTS:
%
%       For the y-intercept, the null hypothesis is that the y-intercept of 
%       the pink line is from a distribution whose is mean equal to that of 
%       the distribution of the y-intercept of the blue line.
%
%       "tt_b" shows the probability of observing the y-intercept of the
%       pink line by chance given that the null hypothesis is true. 
%
%       In the second, the null hypothesis is that the slope of the
%       pink line is from a distribution whose mean is equal to that of the
%       distribution of the slope of the blue line.
%
%       "tt_m" shows the probability of observing the slope of the pink line 
%       by chance given that the null hypothesis is true. 
%   
%       Small values of "tt_b" and "tt_m" cast doubt on the validity 
%       of the null hypotheses.
%
%     WILCOXON RANK TESTS:
% 
%       The Wilcoxon rank test is a non-parametric test which determines
%       the probability that two sets of observations are from the same
%       distribution.  
%
%       "wr_b" shows the probability that the y-intercept of the pink
%       line is from the same distribution as that of the blue line.
%
%       "wr_m" shows the probability that the y-intercept of the pink
%       line is from the same distribution as that of the blue line.
%
%       Values of "wr_b" and "wr_m" close to zero suggest that the 
%       distributions are not the same.
%
%   Example:
%
%      [p,sig,tt,wr] = itdscatter('123.01.01.itd','123.01.08.itd');
%


if (nargin < 3),
	alpha=0.05;
end

[pre_nspikes, pre_nreps]=get_nspikes(pre);
[post_nspikes, post_nreps]=get_nspikes(post);

nreps = min(pre_nreps,post_nreps);
if (pre_nreps ~= post_nreps),
	disp('NREPS for the two files are not the same.');
	disp(sprintf('Using min(pre_nreps,post_nreps) = %d\n',nreps));
end

% first find the slope/intercepts for pre vs. pre
for i=1:nreps,
	p_pre(:,i)=polyfit(pre_nspikes(:,i),pre_nspikes(:,i),1)';
end;

p_pre_means=mean(p_pre');
p_pre_stds=std(p_pre');

% now find the slope/intercepts for pre vs. post
for i=1:nreps,
	p_post(:,i)=polyfit(pre_nspikes(:,i),post_nspikes(:,i),1)';
end;
	
p_post_means=mean(p_post');
p_post_stds=std(p_post');
	
[hb,tt(1,1)]=ttest2(p_pre(1,:),p_post(1,:),alpha);
[ha,tt(1,2)]=ttest2(p_pre(2,:),p_post(2,:),alpha);
wr(1,1)=ranksum(p_pre(1,:),p_post(1,:),alpha);
wr(1,2)=ranksum(p_pre(2,:),p_post(2,:),alpha);

x=[min(min(pre_nspikes)) max(max(pre_nspikes))];
y=polyval(p_pre_means,x);
y_post=polyval(p_post_means,x);

% Plot the graph
hold off
xlabel(sprintf('impulses/sec (%s)',pre));
ylabel(sprintf('impulses/sec (%s)',post));
hold on
for i=1:nreps,
	plot(pre_nspikes(:,i),pre_nspikes(:,i),'bo');
	plot(pre_nspikes(:,i),post_nspikes(:,i),'md');
end	
plot(x,y,'b-');
plot(x,y_post,'m-');
v=axis;
v=axis;
text_x=v(2)-((v(2)-v(1))*.3);
text_y=v(3)+((v(4)-v(3))*.2);
text(text_x,text_y,sprintf('tt_m=%.4f\ntt_b=%.4f\nwr_m=%.4f\nwr_b=%.4f\nm=%7.4f +/-%.3f\nb=%7.4f +/-%.3f',tt(1,1),tt(1,2),wr(1,1),wr(1,2),p_post_means(1),p_post_stds(1),p_post_means(2),p_post_stds(2)));

p=p_post_means';
sig=p_post_stds';

% ---------------------------------------------
function [y,nreps]=get_nspikes(filename)

% first, figure out how many reps we have (ugly, I know)
[s,w] = unix(['xdview -curve -text ' filename]);
t = findstr(w,'Reps=');
tt = findstr(w(t(1)+5:end),char(10));
nreps = sscanf(w(t(1)+5:t(1)+5+tt(1)),'%f');

[s,w] = unix(['xdview -nspikes ' filename]);
y = sscanf(w,'%f');
y = reshape(y,nreps,length(y)/nreps)';
% convert to impulses/sec (we're in impulses/100ms now)
y = y * 10;
