#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "synth.h"
#include "misc.h"
#include "calib.h"

/* This program is designed to test the synthesis routines of libsynth */
/* It does NOT test the calibration code */

#define ATTMAXATTEN 90.0
#define DEBUG  if (ctx.debug) fprintf
#define RFILENAME  "right.out"
#define LFILENAME  "left.out"
#define NCHANS  2

extern int syn_spont;
static float synl_get_bb_low_freq(void);
static float synl_get_bb_high_freq(void);
static int synl_get_dafc(void);
static int synl_get_tone_rad_vary(void);
static int synl_get_stack_rms(void);
void message(char *fmt, ...);
static void parse_cmdline(int ac, char **av);
void init_ctx(void);
void syn_init(void);
double getRad(void);
void deinterlace(xword *outbuf, xword **single, int nsamples, int offset);
void write_outfiles(xword *outbuf, int nsamples);

struct {
	int debug;
	int fc;
	int noise_low;
	int noise_high;
	int rad_vary;
	int stack_rms;
	int use_cache;
	char *stimspec;
	int delay;
	int dur;
	int epoch;
	int itd;
	int bc;
	int rise;
	int fall;
	int side;                  
	int	dump_waveforms;        /* write the synthesized waveforms to files */
} ctx;


int main(int argc, char **argv) 
{
	xword *outbuf = NULL;
	int nsamples;
	syn_spec ss;

	init_ctx();
	parse_cmdline(argc, argv);

	syn_init();

	nsamples=(int) ((double)ctx.epoch)/1000.0*((double)ctx.fc);
	assert((outbuf=calloc(nsamples*NCHANS,sizeof(xword)))!=NULL);
	syn_spec_parse(ctx.stimspec, 0, &ss);
	synthesize(&ss, NULL, outbuf, nsamples, NCHANS, ctx.delay,
		ctx.dur, ctx.epoch, ctx.itd, ctx.bc, ctx.rise, ctx.fall, ctx.side);

	if (ctx.dump_waveforms)
		write_outfiles(outbuf,nsamples);

	return(0);
}

static void parse_cmdline(int ac, char **av)
{
	char		c;
	static struct option longopts[] = {
		{"fc", 1, NULL, 'a'},
		{"debug", 0, NULL, 'A'},
		{"stimspec", 1, NULL, 'b'},
		{"noise_low", 1, NULL, 'B'},
		{"noise_high", 1, NULL, 'c'},
		{"rad_vary", 0, NULL, 'C'},
		{"stack_rms", 0, NULL, 'd'},
		{"use_cache", 0, NULL, 'D'},
		{"delay", 1, NULL, 'e'},
		{"dur", 1, NULL, 'E'},
		{"epoch", 1, NULL, 'f'},
		{"itd", 1, NULL, 'F'},
		{"bc", 1, NULL, 'g'},
		{"rise", 1, NULL, 'G'},
		{"fall", 1, NULL, 'h'},
		{"side", 1, NULL, 'H'},
		{"dump", 0, NULL, 'i'},
		{0, 0, 0, 0}};

	while (1) {

		if ((c = getopt_long_only(ac, av, "a:Ab:B:c:CdDe:E:f:F:g:G:h:H:i", 
					longopts, NULL)) == -1)
			break;

		switch (c) {
			case 'a':
				sscanf(optarg,"%d",&ctx.fc);
				DEBUG(stderr,"acknowledged -fc: fc=%d\n",ctx.fc);
				break;
			case 'A':
				ctx.debug=1;
				DEBUG(stderr,"acknowledged -debug: debug=%d\n",ctx.debug);
				break;
			case 'b':
				assert((ctx.stimspec=calloc(strlen(optarg)+1,sizeof(char)))!=NULL);
				strncpy(ctx.stimspec,optarg,strlen(optarg));
				DEBUG(stderr,"acknowledged -stimspec: stimspec=%s\n",ctx.stimspec);
				break;
			case 'B':
				sscanf(optarg,"%d",&ctx.noise_low);
				DEBUG(stderr,"acknowledged -noise_low: noise_low=%d\n",ctx.noise_low);
				break;
			case 'c':
				sscanf(optarg,"%d",&ctx.noise_high);
				DEBUG(stderr,"acknowledged -noise_high: noise_high=%d\n",ctx.noise_high);
				break;
			case 'C':
				ctx.rad_vary=1;
				DEBUG(stderr,"acknowledged -rad_vary: rad_vary=%d\n",ctx.rad_vary);
				break;
			case 'd':
				ctx.stack_rms=1;
				DEBUG(stderr,"acknowledged -stack_rms: stack_rms=%d\n",ctx.stack_rms);
				break;
			case 'D':
				ctx.use_cache=1;
				DEBUG(stderr,"acknowledged -use_cache: use_cache=%d\n",ctx.use_cache);
				break;
			case 'e':
				sscanf(optarg,"%d",&ctx.delay);
				DEBUG(stderr,"acknowledged -delay: delay=%d\n",ctx.delay);
				break;
			case 'E':
				sscanf(optarg,"%d",&ctx.dur);
				DEBUG(stderr,"acknowledged -dur: dur=%d\n",ctx.dur);
				break;
			case 'f':
				sscanf(optarg,"%d",&ctx.epoch);
				DEBUG(stderr,"acknowledged -epoch: epoch=%d\n",ctx.epoch);
				break;
			case 'F':
				sscanf(optarg,"%d",&ctx.itd);
				DEBUG(stderr,"acknowledged -itd: itd=%d\n",ctx.itd);
				break;
			case 'g':
				sscanf(optarg,"%d",&ctx.bc);
				DEBUG(stderr,"acknowledged -bc: itd=%d\n",ctx.bc);
				break;
			case 'G':
				sscanf(optarg,"%d",&ctx.rise);
				DEBUG(stderr,"acknowledged -rise: rise=%d\n",ctx.rise);
				break;
			case 'h':
				sscanf(optarg,"%d",&ctx.fall);
				DEBUG(stderr,"acknowledged -fall: fall=%d\n",ctx.fall);
				break;
			case 'H':
				switch(optarg[1]) {
					case 'b':
					case 'B':
						ctx.side=SYN_BOTH;
						break;
					case 'l':
					case 'L':
						ctx.side=SYN_LEFT;
						break;
					case 'r':
					case 'R':
						ctx.side=SYN_RIGHT;
						break;
					default:
						fprintf(stderr, "\"%s\" is not valid for -side\n",
							optarg);
						exit(1);
				}
				DEBUG(stderr,"acknowledged -side: side=%d\n",ctx.side);
				break;
			case 'i':
				ctx.dump_waveforms=1;
				DEBUG(stderr,"acknowledged -rad_vary: rad_vary=%d\n",ctx.rad_vary);
				break;
			case '?':
				fprintf(stderr, "flag unrecognised\n");
				exit(1);
				break;
		}
	}

}

static float synl_get_bb_low_freq(void)
{
	return((float)ctx.noise_low);
}

static float synl_get_bb_high_freq(void)
{
	return((float)ctx.noise_high);
}

static int synl_get_dafc(void)
{
	return(ctx.fc);
}

static int synl_get_tone_rad_vary(void)
{
	return(ctx.rad_vary);
}

static int synl_get_stack_rms(void)
{
	return(ctx.stack_rms);
}

void message(char *fmt, ...)
{
  char buf[1000];
  va_list args;

  va_start(args,fmt);
  vsprintf(buf, fmt, args);
  va_end(args);

  fprintf(stderr,"%s",buf);
}

void write_outfiles(xword *outbuf, int nsamples)
{
	int r_fd;
	int l_fd;
	xword *right = NULL;
	xword *left = NULL;

	deinterlace(outbuf,&left,nsamples,0);
	deinterlace(outbuf,&right,nsamples,1);
	
	if ((r_fd = creat(RFILENAME,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))==-1) {
		perror("opening right outfile");
		exit(1);
	} else {
		if (write(r_fd,right,nsamples*sizeof(xword))!=(nsamples*sizeof(xword))) {
			perror("writing to right outfile");
			exit(1);
		}
		if (close(r_fd) == -1) {
			perror("closing right outfile");
			exit(1);
		}
	}

	if ((l_fd = creat(LFILENAME,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))==-1) {
		perror("opening left outfile");
		exit(1);
	} else {
		if (write(l_fd,left,nsamples*sizeof(xword))!=(nsamples*sizeof(xword))) {
			perror("writing to left outfile");
			exit(1);
		}
		if (close(l_fd) == -1) {
			perror("closing left outfile");
			exit(1);
		}
	}

	free(left);
	free(right);
}

void deinterlace(xword *outbuf, xword **single, int nsamples, int offset)
{
	int i,j;

	assert(outbuf != NULL);
	assert((*single=calloc(nsamples,sizeof(xword)))!=NULL);

	for (i=0,j=offset; i<nsamples; i++,j+=NCHANS) 
		(*single)[i]=outbuf[j];

}

void syn_init(void)
{
	syn_alert = message;
	syn_notify = message;
	syn_get_rad = getRad;

	syn_tone_rad_vary = synl_get_tone_rad_vary;
	syn_stack_rms = synl_get_stack_rms;

	syn_bb_low_freq = synl_get_bb_low_freq;
	syn_bb_high_freq = synl_get_bb_high_freq;

	syn_dafc = synl_get_dafc;

	cal_max_atten = ATTMAXATTEN;
	syn_spont = 0;

	syn_use_cache = ctx.use_cache;
	syn_calibrate = 0;
}

double getRad(void)
{
  return(0.0);
}

void init_ctx(void)
{
	ctx.debug = 0;
	ctx.fc = 48000;
	ctx.stimspec = "tone=1000";
	ctx.noise_low = 400;
	ctx.noise_high = 13000;
	ctx.rad_vary = 0;
	ctx.stack_rms = 1;
	ctx.use_cache = 0;
	ctx.delay = 100;
	ctx.dur = 100;
	ctx.epoch = 400;
	ctx.itd = 0;
	ctx.bc = 100;
	ctx.rise = 5;
	ctx.fall = 5;
	ctx.side = SYN_BOTH;
}
