/******************************************************************
**  RCSID: $Id: syn-bind.c,v 1.5 1998/11/29 18:45:25 cmalek Exp $
** Program: xdphys
**  Module: libsynth.a
**  Author: mazer, cmalek
** Descrip: tcl interface functions for standalone synth library
**
*******************************************************************/

#ifndef _IS_STANDALONE_
#define _IS_STANDALONE_
#endif /*_IS_STANDALONE_*/
#ifndef _SYN_STANDALONE_
#define _SYN_STANDALONE_
#endif /* _SYN_STANDALONE_ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <string.h>
#include <tcl.h>
#include "att.h"
#include "iserver.h"
#include "synth.h"
#include "earcal.h"
#include "pm_cal.h"
#include "waveio.h"

/*-----------------------------------------------------------------------
  Typedefs
  ----------------------------------------------------------------------- */
typedef struct syn_data_struct {
	int		ldb;
	float	latten;
	int		rdb;
	float	ratten;
	int		dur;		
	int 	rise;
	int		fall;
	syn_spec 	ss;
	int		delay;
	int 	epoch;
	int 	itd;
	int 	side;
	char	 spec[255];
	int	print;
	char	*file;
	int		ignore;
} SynData;


/*-----------------------------------------------------------------------
  Local Prototypes
  ----------------------------------------------------------------------- */
static int Syn_Synthesize(ClientData, Tcl_Interp	*, int, char **);
static int Syn_TwoSound(ClientData, Tcl_Interp	*, int, char **);
static int Syn_SynSet(ClientData, Tcl_Interp	*, int, char **);
static int Syn_Dump(ClientData, Tcl_Interp *, int, char **);
static void init_syn_data(SynData *flags);
static int parse_syn_data(Tcl_Interp *, SynData *, int, char **);
static int parse_twosound_data(Tcl_Interp *, SynData *, SynData *, int, 
	char **);
static int check_syn_data(Tcl_Interp *, SynData *, char *);
static void print_syn_data(SynData *, char *i);
static void print_syn_spec_types(Tcl_Interp *) ;
static void adjust_db(SynData *flags, float, float);
static void compute_iid_abi(SynData *, SynData *, float *, float *);
static void synthesize_usage(Tcl_Interp *);
static void twosound_usage(Tcl_Interp *);
static void syn_set_usage(Tcl_Interp *);
static void syn_dump_usage(Tcl_Interp *);

/*-----------------------------------------------------------------------
  Static Variables
  ----------------------------------------------------------------------- */
static char last_calfile[80];    /* Name of the last calibration file loaded */

/* ---------------------------------------------------------------------
          					GLOBAL FUNCTIONS
   ---------------------------------------------------------------------
   Syn_Init
   -------------------------------------------------------------------- */
int Syn_Init(Tcl_Interp *interp)
{
	Tcl_CreateCommand(interp, "synthesize", Syn_Synthesize, (ClientData) NULL,
						(Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "twosound", Syn_TwoSound, (ClientData) NULL,
						(Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "syn_set", Syn_SynSet, (ClientData) NULL,
						(Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "syn_dump", Syn_Dump, (ClientData) NULL,
						(Tcl_CmdDeleteProc *) NULL);

	pm_cal_init();

	return(TCL_OK);
}


/* ---------------------------------------------------------------------
          					GLOBAL FUNCTIONS
   ---------------------------------------------------------------------
   Syn_Synthesize
   -------------------------------------------------------------------- */
static int Syn_Synthesize(ClientData z, Tcl_Interp *interp,	int ac,	char **av)
{
	float max_lspl, max_rspl;
	char buf[30];
	SynData	flags;
	int reload = EARCAL_RELOAD_NO;

	if (!iserver_started()) {
		Tcl_AppendResult(interp, "synthesize: iserver not started!", 
			(char *) NULL);
		return(TCL_ERROR);
	}
		
	if ((ac > 1) && Tcl_StringMatch(av[1], "-?")) {
		synthesize_usage(interp);
		return(TCL_ERROR);
	} else {
		init_syn_data(&flags);
		if (parse_syn_data(interp, &flags, ac, av) != TCL_OK)
			return(TCL_ERROR);
		if (check_syn_data(interp, &flags, "synthesize") != TCL_OK)
			return(TCL_ERROR);
		if (strcmp(last_calfile, flags.file) != 0) {
			reload = EARCAL_RELOAD_YES;
			strcpy(last_calfile, flags.file);
		} 
		if (EarCal_LoadData(flags.file, reload)) {
			syn_calibrate = 1;

			syn_spec_parse(flags.spec, 0, &flags.ss);
			if (flags.ss.class == SC_INVALID) {
				Tcl_AppendResult(interp, "synthesize: invalid stimspec!", 
					(char *) NULL);
				return(TCL_ERROR);
			}
			synthesize(&flags.ss, NULL, flags.delay, flags.dur, 
				flags.epoch, flags.itd, flags.rise, flags.fall, 
				flags.side);
			EarCal_FigureSPL(&flags.ss,0,0,&max_lspl,&max_rspl);
			adjust_db(&flags, max_lspl, max_rspl);

			if (!EarCal_SafeFigureAtten(&flags.ss, flags.ldb, flags.rdb, 
				  &flags.latten, &flags.ratten)) {
				if (!flags.ignore) {
					Tcl_AppendResult(interp, 
						"synthesize: error getting attenuator settings\n", 
						(char *) NULL);
					return(TCL_ERROR);
				}
			}
			if (flags.print)
				print_syn_data(&flags, "synthesize: ");
			Tcl_ResetResult(interp);
			if (flags.side == IS_BOTH) {
				sprintf(buf, "%.1f", flags.latten);
				Tcl_AppendElement(interp, buf);
				sprintf(buf, "%.1f", flags.ratten);
				Tcl_AppendElement(interp, buf);
			} else if (flags.side == IS_LEFT) {
				sprintf(buf, "%.1f", flags.latten);
				Tcl_AppendElement(interp, buf);
				sprintf(buf, "%.1f", attMaxAtten);
				Tcl_AppendElement(interp, buf);
			} else if (flags.side == IS_RIGHT) {
				sprintf(buf, "%.1f", attMaxAtten);
				Tcl_AppendElement(interp, buf);
				sprintf(buf, "%.1f", flags.ratten);
				Tcl_AppendElement(interp, buf);
			}
		}
	}

	return(TCL_OK);
}

/* ---------------------------------------------------------------------
   Syn_TwoSound
   -------------------------------------------------------------------- */
static int Syn_TwoSound(ClientData z, Tcl_Interp *interp,	int ac,	char **av)
{
	float del_db_iid, del_db_abi;
	SynData	flags1, flags2;
	syn_spec ss_two;
	char buf[30];
	int reload = EARCAL_RELOAD_NO;

	if (!iserver_started()) {
		Tcl_AppendResult(interp, "twosound: iserver not started!", 
			(char *) NULL);
		return(TCL_ERROR);
	}
		
	if ((ac > 1) && Tcl_StringMatch(av[1], "-?")) {
		twosound_usage(interp);
		return(TCL_ERROR);
	} else {
		init_syn_data(&flags1);
		init_syn_data(&flags2);
		if (parse_twosound_data(interp, &flags1, &flags2, ac, av) != TCL_OK)
			return(TCL_ERROR);
		if (check_syn_data(interp, &flags1, "twosound, stim #1") != TCL_OK)
			return(TCL_ERROR);
		if (check_syn_data(interp, &flags2, "twosound, stim #2") != TCL_OK)
			return(TCL_ERROR);
		if (strcmp(last_calfile, flags1.file) != 0) {
			reload = EARCAL_RELOAD_YES;
			strcpy(last_calfile, flags1.file);
		} 
		if (EarCal_LoadData(flags1.file, reload)) {
			syn_calibrate = 1;

			syn_spec_parse(flags1.spec, 0, &flags1.ss);
			if (flags1.ss.class == SC_INVALID) {
				Tcl_AppendResult(interp, "twosound: invalid stimspec #1!", 
					(char *) NULL);
				return(TCL_ERROR);
			} else 
			syn_spec_parse(flags2.spec, 0, &flags2.ss);
			if (flags2.ss.class == SC_INVALID) {
				Tcl_AppendResult(interp, "twosound: invalid stimspec #2!", 
					(char *) NULL);
				return(TCL_ERROR);
			}
			compute_iid_abi(&flags1, &flags2, &del_db_iid, &del_db_abi);
			two_sound_synthesize(&ss_two, NULL, 1, flags1.epoch, del_db_iid,
				del_db_abi, &flags1.ss,  flags1.delay, flags1.dur, flags1.itd, 
				flags1.rise, flags1.fall, flags1.side, &flags2.ss, 
				flags2.delay, flags2.dur, flags2.itd,
				flags2.rise, flags2.fall, flags1.side);

		  if(!EarCal_SafeFigureAtten(&ss_two, flags1.ldb, flags1.rdb, 
			  &flags1.latten, &flags1.ratten)) {
			if (!flags1.ignore) {
				Tcl_AppendResult(interp, 
					"synthesize: error computing attenuator settings\n", 
					(char *) NULL);
				return(TCL_ERROR);
			}
			}
			if (flags1.print)
				print_syn_data(&flags1, "twosound, stim #1");
			if (flags2.print)
				print_syn_data(&flags2, "twosound, stim #2");
			Tcl_ResetResult(interp);
			if (flags1.side == IS_BOTH) {
				sprintf(buf, "%.1f", flags1.latten);
				Tcl_AppendElement(interp, buf);
				sprintf(buf, "%.1f", flags1.ratten);
				Tcl_AppendElement(interp, buf);
			} else if (flags1.side == IS_LEFT) {
				sprintf(buf, "%.1f", flags1.latten);
				Tcl_AppendElement(interp, buf);
				sprintf(buf, "%.1f", attMaxAtten);
				Tcl_AppendElement(interp, buf);
			} else if (flags1.side == IS_RIGHT) {
				sprintf(buf, "%.1f", attMaxAtten);
				Tcl_AppendElement(interp, buf);
				sprintf(buf, "%.1f", flags1.ratten);
				Tcl_AppendElement(interp, buf);
			}
		}
	}

	return(TCL_OK);
}

/*  ---------------------------------------------------------------------
   Syn_Synthesize
   -------------------------------------------------------------------- */
static int Syn_SynSet(ClientData z, Tcl_Interp *interp,	int ac,	char **av)
{
	int i = 1;
	double f;
	char buf[80];

	if ((ac < 3) || Tcl_StringMatch(av[1], "-?")) {
		syn_set_usage(interp);
		return(TCL_ERROR);
	} else {

		while (i < ac) {
			if (i == ac-1) {
				sprintf(buf, "syn_set: %s flag needs a value", av[i]);
				Tcl_AppendResult(interp, buf, (char *) NULL);
				return(TCL_ERROR);
			}
			if (Tcl_StringMatch(av[i], "-bblow")) {
				if (Tcl_GetDouble(interp, av[++i], &f) != TCL_OK) {
					Tcl_AppendResult(interp, "synthesize: -bblow value invalid", 
						(char *) NULL);
					return(TCL_ERROR);
				} else {
					set_bb_low_freq(f);
				}
			} else if (Tcl_StringMatch(av[i], "-bbhigh")) {
				if (Tcl_GetDouble(interp, av[++i], &f) != TCL_OK) {
					Tcl_AppendResult(interp, "synthesize: -bbhigh value invalid", 
						(char *) NULL);
					return(TCL_ERROR);
				} else {
					set_bb_high_freq(f);
				}
			} else if (Tcl_StringMatch(av[i], "-hplow")) {
				if (Tcl_GetDouble(interp, av[++i], &f) != TCL_OK) {
					Tcl_AppendResult(interp, "synthesize: -hplow value invalid", 
						(char *) NULL);
					return(TCL_ERROR);
				} else {
					set_hp_low_freq(f);
				}
			} else if (Tcl_StringMatch(av[i], "-hphigh")) {
				if (Tcl_GetDouble(interp, av[++i], &f) != TCL_OK) {
					Tcl_AppendResult(interp, "synthesize: -hphigh value invalid", 
						(char *) NULL);
					return(TCL_ERROR);
				} else {
					set_hp_high_freq(f);
				}
			}
			i++;
		}
	}
	return(TCL_OK);
}

/* ---------------------------------------------------------------------
   Syn_Dump
   -------------------------------------------------------------------- */
static int Syn_Dump(ClientData z, Tcl_Interp *interp, int ac, char **av)
{
	char filename[80];
	xword *ibuf;
	int paircount;

	if ((ac < 3) || Tcl_StringMatch(av[1], "-?")) {
		syn_dump_usage(interp);
		return(TCL_ERROR);
	} else {

		strcpy(filename, av[1]);
		if (Tcl_StringMatch(av[2], "-ad")) {
			if ((ibuf = is_getADbuffer(&paircount)) != NULL) {
				savewave(filename, ibuf, is_adFc, paircount, is_getADnchans(), 
					"libsynth waveform", "libsynth", 0.0, 0.0, NULL, 0);
			}
		} else {
			if (Tcl_StringMatch(av[2], "-da")) {
				if ((ibuf = is_getDAbuffer(&paircount)) != NULL) {
					savewave(filename, ibuf, is_daFc, paircount, 
						is_getDAnchans(), "libsynth waveform", "libsynth", 
						0.0, 0.0, NULL, 0);
				}
			} else {
				Tcl_SetResult(interp, "syn_dump: must choose -da or -ad", NULL);
				return(TCL_ERROR);
			}
		}

	}
	return(TCL_OK);
}

/* ------------------------------------------------------------------------
	adjust_db
   ------------------------------------------------------------------------ */
static void adjust_db(SynData *flags, float max_lspl, float max_rspl)
{
	int err = 0;

	if (flags->ldb < (max_lspl-attMaxAtten-20.0)) {
		err |= EARCAL_LEFT_TOO_SOFT;
		fprintf(stderr, "synth: LEFT channel TOO SOFT!\n");
		fprintf(stderr, "synth: raised intensity from %d to %0.1f\n",
			flags->ldb, max_lspl-attMaxAtten-20.0);
		flags->ldb = max_lspl-attMaxAtten-20.0;
	}
	if (flags->ldb > max_lspl) {
		err |= EARCAL_LEFT_TOO_LOUD;
		fprintf(stderr, "synth: LEFT channel TOO LOUD!\n");
		fprintf(stderr, "synth: lowered intensity from %d dB to %0.0f dB \n",
			flags->ldb, max_lspl);
		flags->ldb = max_lspl;
	}
	if (flags->rdb < (max_rspl-attMaxAtten-20.0)) {
		err |= EARCAL_RIGHT_TOO_SOFT;
		fprintf(stderr, "synth: RIGHT channel TOO SOFT!\n");
		fprintf(stderr, "synth: raised intensity from %d to %0.0f\n",
			flags->rdb, max_rspl-attMaxAtten-20.0);
		flags->rdb = max_rspl-attMaxAtten-20.0;
	}
	if (flags->rdb > max_rspl) {
		err |= EARCAL_RIGHT_TOO_LOUD;
		fprintf(stderr, "synth: RIGHT channel TOO LOUD!\n");
		fprintf(stderr, "synth: lowered intensity from %d dB to %0.0f dB\n",
			flags->rdb, max_rspl);
		flags->rdb = max_rspl;
	}
}

/* ---------------------------------------------------------------------
   init_syn_data
   -------------------------------------------------------------------- */
static void init_syn_data(SynData *flags)
{
	flags->ldb = 0;     /* dBspl */
	flags->latten = 0;
	flags->rdb = 0;     /* dBspl */
	flags->ratten = 0;
	flags->dur=100;		 /* ms */
	flags->rise=10;      /* ms */
	flags->fall=10;      /* ms */
	flags->delay=0;      /* ms */
	flags->epoch=is_getDAmslength();  /* ms */
	flags->itd=0;        /* us */
	flags->side=IS_BOTH;
	strcpy(flags->spec, "tone=4000");
	flags->print=0;
	flags->ignore=0;
	flags->file=NULL;
}

/* ---------------------------------------------------------------------
   synthesize_usage
   -------------------------------------------------------------------- */
static void synthesize_usage(Tcl_Interp *interp) 
{
	Tcl_AppendResult(interp, "usage: synthesize <flags>\n\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -ldb <db>     left absolute intensity\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -rdb <dB>     right absolute intensity\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -delay <ms>   delay before onset of stimulus\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -dur <ms>     duration of stimulus\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -rise <ms>    rise time of stimulus envelope\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -fall <ms>    fall time of stimulus envelope\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -itd <us>     itd b/t left and right channel\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -side         left, right or both\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -spec <spec>  syn spec for synthesize()\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -file <fname>  use earcal file <fname>\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -types  	  	 print synth sound types\n", (char *) NULL);

}

/* ---------------------------------------------------------------------
   twosound_usage
   -------------------------------------------------------------------- */
static void twosound_usage(Tcl_Interp *interp) 
{
	Tcl_AppendResult(interp, "usage: twosound <flags>\n\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -ldb1 <dB>     left absolute intensity, #1\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -ldb2 <dB>     left absolute intensity, #2\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -rdb1 <dB>     right absolute intensity, #1\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -rdb2 <dB>     right absolute intensity, #2\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -delay1 <ms>   delay before onset of stimulus, #1\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -delay2 <ms>   delay before onset of stimulus, #2\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -dur1 <ms>     duration of stimulus, #1\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -dur2 <ms>     duration of stimulus, #2\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -rise1 <ms>    rise time of stimulus envelope, #1\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -rise2 <ms>    rise time of stimulus envelope, #2\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -fall1 <ms>    fall time of stimulus envelope, #1\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -fall2 <ms>    fall time of stimulus envelope, #2\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -itd1 <us>     itd b/t left and right channel, #1\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -itd2 <us>     itd b/t left and right channel, #2\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -side         left, right or both\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -spec1 <spec>  syn spec for twosound(), #1\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -spec2 <spec>  syn spec for twosound(), #2\n", 
		(char *) NULL);
	Tcl_AppendResult(interp, 
		"      -file <fname>  use earcal file <fname>\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -types  	  	 print synth sound types\n", (char *) NULL);


}

/* ---------------------------------------------------------------------
   syn_set_usage
   -------------------------------------------------------------------- */
static void syn_set_usage(Tcl_Interp *interp) 
{
	Tcl_AppendResult(interp, "usage: syn_set <flags>\n\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -bblow <Hz>     broadband noise low freq\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -bbhigh <Hz>    broadband noise high freq\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -hplow <Hz>     highpass noise low freq\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -hphigh <Hz>     highpass noise high freq\n", (char *) NULL);
}

/* ---------------------------------------------------------------------
   syn_dump_usage
   -------------------------------------------------------------------- */
static void syn_dump_usage(Tcl_Interp *interp) 
{
	Tcl_AppendResult(interp, "usage: syn_dump <filename> -da | -ad\n\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -da <Hz>     write DA buffers to <filename> in xdphys analog file format\n", (char *) NULL);
	Tcl_AppendResult(interp, 
		"      -ad <Hz>     write AD buffers to <filename> in xdphys analog file format\n", (char *) NULL);
}

/* ---------------------------------------------------------------------
   print_syn_data
   -------------------------------------------------------------------- */
static void print_syn_data(SynData *flags, char *id)
{
	fprintf(stderr, "%s: ldb=%d\n", id, flags->ldb);
	fprintf(stderr, "%s: latten=%f\n", id, flags->latten);
	fprintf(stderr, "%s: rdb=%d\n", id, flags->rdb);
	fprintf(stderr, "%s: ratten=%f\n", id, flags->ratten);
	fprintf(stderr, "%s: dur=%d\n", id, flags->dur);
	fprintf(stderr, "%s: delay=%d\n", id, flags->delay);
	fprintf(stderr, "%s: rise=%d\n", id, flags->rise);
	fprintf(stderr, "%s: fall=%d\n", id, flags->fall);
	fprintf(stderr, "%s: spec=%s\n", id, flags->spec);
	fprintf(stderr, "%s: file=%s\n", id, flags->file);
}

/* ---------------------------------------------------------------------
   parse_syn_data
   -------------------------------------------------------------------- */
static int parse_syn_data(Tcl_Interp *interp, SynData *flags, int ac, 
	char **av)
{
	int	i = 1;
	char buf[80];

	while (i < ac) {

		if (Tcl_StringMatch(av[i], "-types")) {
			print_syn_spec_types(interp);
			return(TCL_ERROR);
		} else if (Tcl_StringMatch(av[i], "-print")) {
			flags->print = 1;
			i++;
			continue;
		} else if (Tcl_StringMatch(av[i], "-ignore")) {
			flags->ignore = 1;
			i++;
			continue;
		}

		if (i == ac-1) {
			sprintf(buf, "synthesize: %s flag needs a value", av[i]);
			Tcl_AppendResult(interp, buf, (char *) NULL);
			return(TCL_ERROR);
		}
		if (Tcl_StringMatch(av[i], "-file")) {
			assert((flags->file = (char *) calloc(strlen(av[++i]), 
				sizeof(char))) != NULL);
			strcpy(flags->file, av[i]);
		} else if (Tcl_StringMatch(av[i], "-ldb")) {
			if (Tcl_GetInt(interp, av[++i], &(flags->ldb)) != TCL_OK) {
				Tcl_AppendResult(interp, "synthesize: -ldb value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-rdb")) {
			if (Tcl_GetInt(interp, av[++i], &(flags->rdb)) != TCL_OK) {
				Tcl_AppendResult(interp, "synthesize: -rdb value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-delay")) {
			if (Tcl_GetInt(interp, av[++i], &(flags->delay)) != TCL_OK) {
				Tcl_AppendResult(interp, "synthesize: -delay value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-dur")) {
			if (Tcl_GetInt(interp, av[++i], &(flags->dur)) != TCL_OK) {
				Tcl_AppendResult(interp, "synthesize: -dur value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-fall")) {
			if (Tcl_GetInt(interp, av[++i], &(flags->fall)) != TCL_OK) {
				Tcl_AppendResult(interp, "synthesize: -fall value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-rise")) {
			if (Tcl_GetInt(interp, av[++i], &(flags->rise)) != TCL_OK) {
				Tcl_AppendResult(interp, "synthesize: -rise value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-itd")) {
			if (Tcl_GetInt(interp, av[++i], &(flags->itd)) != TCL_OK) {
				Tcl_AppendResult(interp, "synthesize: -itd value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-spec")) {
			strncpy(flags->spec, av[++i], 255);
		} else if (Tcl_StringMatch(av[i], "-side")) {
			if (Tcl_StringMatch(av[++i], "l*")) {
				flags->side = IS_LEFT;
			} else if (Tcl_StringMatch(av[i], "r*")) {
				flags->side = IS_RIGHT;
			} else if (Tcl_StringMatch(av[i], "b*")) {
				flags->side = IS_BOTH;
			} else {
				Tcl_AppendResult(interp, 
					"synthesize: invalid value for -side", (char *) NULL);
				return(TCL_ERROR);
			}
		}

		i++;
	}

	return(TCL_OK);
}

/* ---------------------------------------------------------------------
   check_syn_data
   -------------------------------------------------------------------- */
static int check_syn_data(Tcl_Interp *interp, SynData *flags, char *id)
{
	char buf[80];

	if (flags->dur > flags->epoch) {
		sprintf(buf, "%s: stim duration (%d) > epoch (%d)",
			id, flags->dur, flags->epoch);
		Tcl_AppendResult(interp, buf, (char *) NULL);
		return(TCL_ERROR);
	}

	if ((flags->dur+flags->delay) > flags->epoch) {
		sprintf(buf, "synthesize: stim delay (%d) + dur (%d) > epoch (%d)",
			flags->delay, flags->dur, flags->epoch);
		Tcl_AppendResult(interp, buf, (char *) NULL);
		return(TCL_ERROR);
	}

	return(TCL_OK);
}

/* ---------------------------------------------------------------------
   print_syn_spec_types
   -------------------------------------------------------------------- */
static void print_syn_spec_types(Tcl_Interp * interp) 
{
	Tcl_AppendResult(interp, "synthesize sound specs (for -spec switch):\n\n",
		"     bb................. Broadband Noise\n",
		"     hp................. high pass noise (4-13k)\n",
		"     tone=NNN .......... pure tone of NNN hz\n",
		"     bp=lo-hi .......... band pass from f1-f2\n",
		"     CF=<cf>,BW=<w> .... bandpass from cf-(bw/2) to cf+(bw/2)\n",
		"     stack=<range>...... pure tone list (tone stack)\n",
		"     fmsweep=from-to:n.. FM Sweep (n == 1, if not specified..)\n",
		(char *) NULL);
}

/* ---------------------------------------------------------------------
   parse_twosound_data
   -------------------------------------------------------------------- */
static int parse_twosound_data(Tcl_Interp *interp, SynData *flags1, 
	SynData *flags2, int ac, char **av)
{
	int	i = 1;
	char buf[80];

	while (i < ac) {

		if (Tcl_StringMatch(av[i], "-types")) {
			print_syn_spec_types(interp);
			return(TCL_ERROR);
		} else if (Tcl_StringMatch(av[i], "-print")) {
			flags1->print = 1;
			flags2->print = 1;
			i++;
			continue;
		} else if (Tcl_StringMatch(av[i], "-ignore")) {
			flags1->ignore = 1;
			i++;
			continue;
		}

		if (i == ac-1) {
			sprintf(buf, "twosound: %s flag needs a value", av[i]);
			Tcl_AppendResult(interp, buf, (char *) NULL);
			return(TCL_ERROR);
		}
		if (Tcl_StringMatch(av[i], "-file")) {
			assert((flags1->file = (char *) calloc(strlen(av[++i]), 
				sizeof(char))) != NULL);
			strcpy(flags1->file, av[i]);
			fprintf(stderr, "twosound: using %s\n", flags1->file);
		} else if (Tcl_StringMatch(av[i], "-ldb1")) {
			if (Tcl_GetInt(interp, av[++i], &(flags1->ldb)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -ldb1 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-rdb1")) {
			if (Tcl_GetInt(interp, av[++i], &(flags1->rdb)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -rdb1 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-delay1")) {
			if (Tcl_GetInt(interp, av[++i], &(flags1->delay)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -delay1 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-dur1")) {
			if (Tcl_GetInt(interp, av[++i], &(flags1->dur)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -dur1 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-fall1")) {
			if (Tcl_GetInt(interp, av[++i], &(flags1->fall)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -fall1 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-rise1")) {
			if (Tcl_GetInt(interp, av[++i], &(flags1->rise)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -rise1 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-itd1")) {
			if (Tcl_GetInt(interp, av[++i], &(flags1->itd)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -itd1 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-spec1")) {
			strncpy(flags1->spec, av[++i], 255);
		} else if (Tcl_StringMatch(av[i], "-ldb2")) {
			if (Tcl_GetInt(interp, av[++i], &(flags2->ldb)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -ldb2 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-rdb2")) {
			if (Tcl_GetInt(interp, av[++i], &(flags2->rdb)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -rdb2 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-delay2")) {
			if (Tcl_GetInt(interp, av[++i], &(flags2->delay)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -delay2 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-dur2")) {
			if (Tcl_GetInt(interp, av[++i], &(flags2->dur)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -dur2 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-fall2")) {
			if (Tcl_GetInt(interp, av[++i], &(flags2->fall)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -fall2 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-rise2")) {
			if (Tcl_GetInt(interp, av[++i], &(flags2->rise)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -rise2 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-itd2")) {
			if (Tcl_GetInt(interp, av[++i], &(flags2->itd)) != TCL_OK) {
				Tcl_AppendResult(interp, "twosound: -itd2 value invalid", 
					(char *) NULL);
				return(TCL_ERROR);
			}
		} else if (Tcl_StringMatch(av[i], "-spec2")) {
			strncpy(flags2->spec, av[++i], 255);
		} else if (Tcl_StringMatch(av[i], "-side")) {
			if (Tcl_StringMatch(av[++i], "l*")) {
				flags1->side = IS_LEFT;
			} else if (Tcl_StringMatch(av[i], "r*")) {
				flags1->side = IS_RIGHT;
			} else if (Tcl_StringMatch(av[i], "b*")) {
				flags1->side = IS_BOTH;
			} else {
				Tcl_AppendResult(interp, 
					"twosound: invalid value for -side", (char *) NULL);
				return(TCL_ERROR);
			}
		}

		i++;
	}

	return(TCL_OK);
}

/* ------------------------------------------------------------------------
	compute_iid_abi
   ------------------------------------------------------------------------ */
static void compute_iid_abi(SynData *flags1, SynData *flags2, 
	float *del_db_iid, float* del_db_abi)
{
	if (flags1->side == IS_BOTH) {
		*del_db_iid = (float) ((flags2->rdb - flags2->ldb) - (flags1->rdb - 
			flags1->ldb));
		*del_db_abi = ((float) flags2->rdb + (float) flags2->ldb)/2.0 - 
			((float) flags1->rdb + (float) flags1->ldb)/2.0;
	} else if (flags1->side == IS_LEFT) {
		*del_db_iid = 0.0;
		*del_db_abi = (float) (flags2->ldb-flags1->ldb);
	} else if (flags1->side == IS_RIGHT) {
		*del_db_iid = 0.0;
		*del_db_abi = (float) (flags2->rdb-flags1->rdb);
	}
}

