/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: stimarray.c,v 1.8 2000/11/27 04:34:57 cmalek Exp $
** Program: xdphys
**  Module: StimArray.c
**  Author: jamie/BA
** Descrip: functions to deal with stimulus arrays
**
** Revision History (most recent last)
**
** Tue Aug  6 11:20:44 PDT 1996
**   spliced out of misc.c; BA
**
** 97.2 bjarthur
**   added StimArray_GenDisjoint2() which is similar to
**   StimArray_GenDisjoint() but is used in modules whose independent
**   variable is monaural, but varied binaurally (eg mono, fiid)
**
** 97.4 bjarthur
**   StimArray_Gen() deleted and mod_itd.c rewritten a bit to use
**     StimArray_GenDisjoint()
**   StimArray_GenDisjoint() renamed to StimArray_Gen()
**   StimArray_GenDisjoint(), _RangeToFloats(), and _RangeToInts()
**     all did basically the same thing.  So I wrote StimArray_Parse()
**     to do the commonalities, and then modified _GenDisjoint() to call
**     it.  _Parse() is also called by synth.c, which used to call
**     _RangeToFloats()
**   _RangeToFloats() and _RangeToInts() deleted
**
** 97.12 bjarthur
**  modified StimArray_Parse to much more rigorously check syntax.  if there
**  is any portion which doesn't fit, then it sends an alert message
**
** 98.02 bjarthur
**  added ()+x, ()-x syntax to StimArray_Parse.  it shifts whatever is in
**  the parens by x.
*******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#ifndef __linux__
#include <malloc.h>
#endif /* __linux__ */

#include "stimarray.h"
#include "misc.h"

/*
#include "mathfns.h"
#include "synthlib.h"
*/

extern int (*syn_dafc)(void);
int syn_spont = -6666;

static int CheckRange(int, int, int);

static int rnd2int(float f)
{
  if (f < 0)
    return((int)(f - 0.5));
  else
    return((int)(f + 0.5));
}

#define SWAP(a,b,type) { type tmp; tmp=a; a=b; b=tmp; }

static void randomize_array(int *data, int ndata)
{
  int i, j;
#ifndef __linux__
  extern time_t time();
#endif

  /* initialize random number generator */
  srand(time((long *)0) * rand());

  for(i=0; i<(ndata-1); i++) {
    j = 1+(rand()%(ndata-1-i));  /* used to be random() */
    SWAP(data[i],data[i+j],int); }
}


/* given two ranges, determine whether they have the exact same stimuli */
int StimArray_Equivalent(char *old, char *new)
{
  int old_nsamps, new_nsamps;
  int *old_stimarray, *new_stimarray;
  int i,j,ret_val,found;

  old_stimarray = StimArray_Gen(&old_nsamps,0,old,1,1,0);
  new_stimarray = StimArray_Gen(&new_nsamps,0,new,1,1,0);

  if(old_nsamps != new_nsamps)
    return(0);

  for(i=0,ret_val=1; (i<old_nsamps) && ret_val; i++) {
    for(j=0,found=0; j<new_nsamps; j++)
      if(old_stimarray[i]==new_stimarray[j]) {
        found=1;
        break; }
    ret_val &= found; }

  return(ret_val);
}

/* given two ranges, determine whether they have any common stimuli */
int StimArray_Overlap(char *old, char *new)
{
  int old_nsamps, new_nsamps;
  int *old_stimarray, *new_stimarray;
  int i,j;

  old_stimarray = StimArray_Gen(&old_nsamps,0,old,1,1,0);
  new_stimarray = StimArray_Gen(&new_nsamps,0,new,1,1,0);

  for(i=0; i<old_nsamps; i++) {
    for(j=0; j<new_nsamps; j++) {
      if(old_stimarray[i]==new_stimarray[j])
        return(1); } }

  return(0);
}

static int get_shift(char *desc, char **q)
{
	char *p;
	char remain[128];
	char *tmp = NULL;
	int shift;

	if (desc[0] == '(') {
		assert((tmp =(char *) calloc(strlen(desc) + 1, sizeof(char))) != NULL);
		strcpy(tmp, desc + 1);
		if ((p = strstr(tmp, ")-")) != NULL) {
			*p = '\0';
			strcpy(*q, tmp);
			if (sscanf(p + 2, "%d%s", &shift, remain) == 2) {
				syn_alert("parse error.");
				return (0);
			}
			shift = -shift;
		} else if ((p = strstr(tmp, ")+")) != NULL) {
			*p = '\0';
			strcpy(*q, tmp);
			if (sscanf(p + 2, "%d%s", &shift, remain) == 2) {
				syn_alert("parse error.");
				return (0);
			}
		} else {
			strcpy(*q, desc);
			shift = 0;
		}
	} else {
		strcpy(*q, desc);
		shift = 0;
	}
	
	free(tmp);

	return (shift);
}

/* given two ranges, determine whether they are shifted the same */
int StimArray_SameShift(char *old, char *new)
{
	char *q;

	assert((q=(char *)calloc(128,sizeof(char)))!=NULL);

	if(get_shift(old,&q) == get_shift(new,&q)) {
		free(q);
		return(1); }

	free(q);
  return(0);
}

char *StimArray_Combine(char *one, char *two)
{
	static char ret_val[256];
  char *q1, *q2;
	int shift;

	assert((q1=(char *)calloc(128,sizeof(char)))!=NULL);
	assert((q2=(char *)calloc(128,sizeof(char)))!=NULL);

	assert((shift=get_shift(one,&q1)) == get_shift(two,&q2));

	if(shift==0)
	  sprintf(ret_val,"%s,%s",q1,q2);
	else
	  sprintf(ret_val,"(%s,%s)%+d",q1,q2,shift);
  
	free(q1); free(q2);
	return(ret_val);
}

/* genDisjointStimArray() --
** like genStimArray(), but takes a string description of the
** search space that can be disjoint. For example, you could
** specify "-120:-80:10, -20, 0, 25, 200:300:30" to generate
** two ranges and three specific datapoints. The rand and reps
** parameters are the same as in genStimArray() and SPONT
** trial are automatically included.
**
** one new twist, a range such as a:b:/10 will be read as go
** from a to b in 10 steps..
**
** a 2nd new twist, a range such as a:b:@10 will be read as go
** from a to b in 10 steps spaced exponentially...
**
** no duplicates allowed if nodups is specified (NULL is
** returned to signal an error).
*/
int *StimArray_Gen(int *nsamp, int rand, char *descr, int reps,
      int nodups, int spont)
{
	int i, j;
	extern char *strtok();
	int n, n0, *sarray, *temp2;

	sarray = NULL;
	if (!StimArray_Parse(descr, &sarray, NULL, &n))
		return (NULL);

	if (spont) {
		if (n++ == 0)
			sarray = (int *) malloc(n * sizeof(int));
		else
			sarray = (int *) realloc(sarray, n * sizeof(int));
		sarray[n - 1] = syn_spont;
	}

	if (nodups) {
		for (i = 0; i < (n - 1); i++) {
			for (j = i + 1; j < n; j++) {
				if (sarray[i] == sarray[j]) {
					syn_alert("Duplicates in stimulus array");
					free(sarray);
					return (NULL);
				}
			}
		}
	}

	n0 = n;
	n = 0;
	temp2 = (int *) malloc(n0 * reps * sizeof(int));
	for (i = 0; i < reps; i++) {
		if (rand)
			randomize_array(sarray, n0);
		for (j = 0; j < n0; j++)
			temp2[n++] = sarray[j];
	}

	free(sarray);
	*nsamp = n;
	return (temp2);
}

/* StimArray_ClosestITD: 
 *
 * This is used for already computed stimarrays in conjunction with stim
 * types which make ITDs by shifting either the left or right side
 * forward an integral number of samples, e.g. SC_CLICK 
 *
 * It tries to set the ITD for each entry to an integral number of 
 * sampling periods.  This was mainly created so that the ITDs written
 * to xdphys itd files would reflect what the actual ITD was.
 *
 * Returns a new array with the adjusted ITDs, which must be freed
 * when you are done with it.
 *
 * */

int *StimArray_ClosestITD(int *stimarray, int nstims)
{
	int i;
	int *adj_stimarray = NULL;
	int high, low, real;
	float period;

	period = 1.0/syn_dafc() * 1.0e6; /* microseconds */

	assert((adj_stimarray = (int *) calloc(nstims, sizeof(int))) != NULL);

	for (i=0; i<nstims; i++) {
		real=stimarray[i];
		low=(int) (floor(((float)real)/period) * period);
		high=(int) ((floor(((float)real)/period) + 1.0) * period);

		if (abs(real-low) < abs(real-high))
			adj_stimarray[i]=low;
		else
			adj_stimarray[i]=high;
	}
		
	return(adj_stimarray);

}

static int CheckRange(int start, int stop, int step)
{
  if (step == 0) {
    syn_alert("Invalid range");
    return(0); }
  if ((start>stop) && (step>0)) {
    syn_alert("Invalid range");
    return(0); }
  if ((start<stop) && (step<0)) {
    syn_alert("Invalid range");
    return(0); }

  return(1);
}

int StimArray_Parse(char *desc, int **farray, int **iarray, int *n)
{
	char *p;
	int i, from, to, step, nsteps, nn;
	float f,fstep;
	char *remain=NULL;
	char *q=NULL;
	int shift;

	assert(desc != NULL);

	assert((q=(char *)calloc(strlen(desc)+1,sizeof(char)))!=NULL);
	assert((remain=(char *)calloc(strlen(desc)+1,sizeof(char)))!=NULL);
	*remain = '\0';

	shift = get_shift(desc,&q);

	nn=0;
	for (p = strtok(q, ","); p != NULL; p = strtok(NULL, ",")) {
		if ((i = sscanf(p, "%d:%d:%d%s", &from, &to, &step, remain)) >= 3) {
			if(!CheckRange(from,to,step)) {
				if(*farray) free(*farray);
				if(iarray) if(*iarray) free(*iarray);
				return(0); }
			for (f = from; f <= to; f += step) {
				if (nn++ == 0) {
					assert((*farray = (int*)malloc(nn * sizeof(int)))!=NULL);
					if(iarray) assert((*iarray = (int*)malloc(nn * sizeof(int)))!=NULL); }
				else {
					assert((*farray = (int*)realloc(*farray, nn * sizeof(int)))!=NULL);
					if(iarray) assert((*iarray = (int*)realloc(*iarray, nn * sizeof(int)))!=NULL); }
				(*farray)[nn - 1] = f;
				if(iarray) (*iarray)[nn - 1] = 0; }
			if(i==4) { syn_alert("parse error."); return(0); }
		} else if ((i = sscanf(p, "%d:%d:/%d%s", &from, &to, &nsteps, remain)) >= 3) {
			fstep = ((float)to - (float)from) / (float)(nsteps-1);
			for (f = (float)from; rnd2int(f) <= to; f += fstep) {
				if (nn++ == 0) {
					assert((*farray = (int*)malloc(nn * sizeof(int)))!=NULL);
					if(iarray) assert((*iarray = (int*)malloc(nn * sizeof(int)))!=NULL); }
				else {
					assert((*farray = (int*)realloc(*farray, nn * sizeof(int)))!=NULL);
					if(iarray) assert((*iarray = (int*)realloc(*iarray, nn * sizeof(int)))!=NULL); }
				(*farray)[nn - 1] = rnd2int(f);
				if(iarray) (*iarray)[nn - 1] = 0; }
			if(i==4) { syn_alert("parse error."); return(0); }
		} else if ((i = sscanf(p, "%d:%d:@%d%s", &from, &to, &nsteps, remain)) >= 3) {
			fstep = (float)pow((to/from),1.0/(nsteps-1));
			for (f = (float)from; rnd2int(f) <= to; f *= fstep) {
				if (nn++ == 0) {
					assert((*farray = (int*)malloc(nn * sizeof(int)))!=NULL);
					if(iarray) assert((*iarray = (int*)malloc(nn * sizeof(int)))!=NULL); }
				else {
					assert((*farray = (int*)realloc(*farray, nn * sizeof(int)))!=NULL);
					if(iarray) assert((*iarray = (int*)realloc(*iarray, nn * sizeof(int)))!=NULL); }
				(*farray)[nn - 1] = rnd2int(f);
				if(iarray) (*iarray)[nn - 1] = 0; }
			if(i==4) { syn_alert("parse error."); return(0); }
		} else {
			if (nn++ == 0) {
				assert((*farray = (int*)malloc(nn * sizeof(int)))!=NULL);
				if(iarray) assert((*iarray = (int*)malloc(nn * sizeof(int)))!=NULL); }
			else {
				assert((*farray = (int*)realloc(*farray, nn * sizeof(int)))!=NULL);
				if(iarray) assert((*iarray = (int*)realloc(*iarray, nn * sizeof(int)))!=NULL); }
			if(!iarray) {
				i = sscanf(p, "%d%s", &((*farray)[nn-1]), remain);
				if(i!=1) { syn_alert("parse error."); return(0); } }
			else {
				if((i=sscanf(p, "%d(%d)%s", &((*farray)[nn-1]), &((*iarray)[nn-1]),
					remain))>=2) {
					if(i==3) { syn_alert("parse error."); return(0); } }
				else {
					i = sscanf(p, "%d%s", &((*farray)[nn-1]), remain);
					(*iarray)[nn - 1] = 0;
					if(i!=1) { syn_alert("parse error."); return(0); } } } } }

	if(shift!=0) {
		for(i=0; i<nn; i++)
			(*farray)[i] += shift; }

	(*n)=nn;

	if (q!=NULL) free(q);
	if (remain != NULL) free(remain);

	return(1);
}
