/*
 * ** This file is part of XDowl
 * ** Copyright (c) 1994 Jamie Mazer
 * ** California Institute of Technology
 * ** <mazer@asterix.cns.caltech.edu>
 */

/******************************************************************
**  RCSID: $Id: calib.c,v 1.7 2000/11/27 01:49:16 cmalek Exp $
** Program: xdphys
**  Module: earcal.c
**  Author: mazer, bjarthur
** Descrip: earphone calibration file support -- reading/lookup only
**
** Revision History (most recent last)
**
** Thu May  5 23:04:53 1994 mazer
**  this is hideous -- I'm to have written this code..
**  added the figure_scale stuff today... this needs to be
**  rewritten to be cleaner, more consistent and faster..
**
** 96-97 bjarthur
**  found major bug which calibrated the left channel with the
**   right channels calibration data.
**  added phase calibration
**  rewrote a lot of stuff and changed most function names
**  added feature to attenuate waves in buffer by 20dB if the
**   attenuators couldn't attenuate enough.
**
** 97.5 bjarthur
**  added rsfspl, lsfspl which are precomputed scale factors so that
**   cal_FigureCal doesn't have to evaluate a pow() every time.
**   this should greatly speed up syn_noise.
**
** 97.7 bjarthur
**  changed FigureAtten() and FigureSPL() to treat SC_TONE just like
**   everyone else re calibration and the rms fields
**
** 98.11 bjarthur
**  made into independent library, changed name to calib.c
**
*******************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include "misc.h"
#include "calib.h"

float cal_max_atten;

static int nfreqs = 0;					/* number of reference freqs */
static float ref_freqs[CAL_MAX_CAL_PTS];	/* list of reference frequencies */
static float rsfspl[CAL_MAX_CAL_PTS];		/* rdbspl converted to scale factor */
static float lsfspl[CAL_MAX_CAL_PTS];
static float rdbspl[CAL_MAX_CAL_PTS];		/* dBspl of full range sin on */
static float ldbspl[CAL_MAX_CAL_PTS];		/* right, left channels */
static float rusphi[CAL_MAX_CAL_PTS];		/* us phase right channel */
static float lusphi[CAL_MAX_CAL_PTS];		/* us phase left channel */
static int cal_time;

static float min_r_dbspl = 0.0;
static float min_l_dbspl = 0.0;
static float max_r_dbspl = 0.0;
static float max_l_dbspl = 0.0;

/*static int lookup_mag_db(float, float*, float*); */
static int lookup_mag_sf(float, float *, float *);
static int lookup_phi(float, float *, float *);

/* there's a duplicate copy of this in xdphys/mathfns.c.  wish there was */
/* a way not to keep multiple copies of the same code... */

/* doesn't assume xs[] is ordered, or evenly spaced  */
static float lin_interpolate(float x, int n, float *xs, float *ys)
{
  int i, pf, pc, f, c;

  for(i=0, pf=-1, pc=-1; i<n; i++) {
    if(((pf==-1) || (xs[i]>f)) && (xs[i]<=x)) {
      pf = i;
      f = xs[i]; }
    if(((pc==-1) || (xs[i]<c)) && (xs[i]>=x)) {
      pc = i;
      c = xs[i]; } }

  if((pf!=-1)&&(pc!=-1)) {
    if(pf==pc)
      return(ys[pc]);
    else
      return(((ys[pc]-ys[pf]) / (xs[pc]-xs[pf])) * (x-xs[pf]) + ys[pf]); }
  else if(pf==-1)
    return(ys[pc]);
  else if(pc==-1)
    return(ys[pf]);

  assert((pf==-1)&&(pc==-1));

  return(0.0);
}

static void floatVec_minmax(float *v, int n, float *minp, float *maxp)
{
  int i;

  if (n > 0) {
    *minp = *maxp = v[0];
    for (i = 1; i < n; i++) {
      if (v[i] > *maxp)
  *maxp = v[i];
      if (v[i] < *minp)
  *minp = v[i];
    }
  }
}

int cal_GetTime()
{
	return (cal_time);
}

/* side = 0 means left, 1 means right */

float cal_GetOverallMinDBSPL(int side)
{
	assert((side == 0) || (side == 1));
	if (side == 0)
		return (min_l_dbspl);
	else
		return (min_r_dbspl);
}


static int lookup_mag_sf(float freq, float *l, float *r)
{
	if (nfreqs > 2) {
		if (l != NULL)
			*l = lin_interpolate(freq, nfreqs, ref_freqs, lsfspl);
		if (r != NULL)
			*r = lin_interpolate(freq, nfreqs, ref_freqs, rsfspl);
		return (1);
	}
	else if (nfreqs == 1) {
		if (l != NULL)
			*l = lsfspl[0];
		if (r != NULL)
			*r = rsfspl[0];
		return (1);
	}
	else {
		return (0);
	}
}

static int lookup_phi(float freq, float *l, float *r)
{
	if (nfreqs > 2) {
		if (l != NULL)
			*l = lin_interpolate(freq, nfreqs, ref_freqs, lusphi);
		if (r != NULL)
			*r = lin_interpolate(freq, nfreqs, ref_freqs, rusphi);
		return (1);
	}
	else if (nfreqs == 1) {
		if (l != NULL)
			*l = lusphi[0];
		if (r != NULL)
			*r = rusphi[0];
		return (1);
	}
	else {
		return (0);
	}
}

int cal_FakeFlatCal(char **caltype)
{
	nfreqs = 1;
	ldbspl[0] = 70.0;
	rdbspl[0] = 70.0;
	lsfspl[0] = 1.0;
	rsfspl[0] = 1.0;
	lusphi[0] = 0.0;
	rusphi[0] = 0.0;
	cal_time = -1;
	min_r_dbspl = max_r_dbspl = 70.0;
	min_l_dbspl = max_l_dbspl = 70.0;
	(*caltype) = "? 70dbspl/0us ?";
	return (1);
}

void unwrap(float *phis, int nfreqs, float tol)
{
	int i;

  if(nfreqs<1) return;

	while (phis[0] > tol)
		phis[0] -= (2.0 * M_PI);
	while (phis[0] < -tol)
		phis[0] += (2.0 * M_PI);
	for (i = 0; i < nfreqs - 1; i++) {
		if ((phis[i] - phis[i + 1]) > tol) {
			phis[i + 1] += 2 * M_PI;
			i--;
		}
		else if ((phis[i] - phis[i + 1]) < -tol) {
			phis[i + 1] -= 2 * M_PI;
			i--;
		}
	}

}

int cal_SetData(int ndata, float *freqs, float *lmags, float *rmags,
			float *lphis, float *rphis, int time)
{
	int i;

	unwrap(lphis, ndata, M_PI);
	unwrap(rphis, ndata, M_PI);

	for (nfreqs = 0; (nfreqs < ndata) && (nfreqs < CAL_MAX_CAL_PTS); nfreqs++) {
	ref_freqs[nfreqs] = freqs[nfreqs];
		ldbspl[nfreqs] = lmags[nfreqs];
		rdbspl[nfreqs] = rmags[nfreqs];
		lusphi[nfreqs] = 1.0e6 * lphis[nfreqs] / (2.0 * M_PI * freqs[nfreqs]);
		rusphi[nfreqs] = 1.0e6 * rphis[nfreqs] / (2.0 * M_PI * freqs[nfreqs]); }

	cal_time = time;
	floatVec_minmax(rdbspl, nfreqs, &min_r_dbspl, &max_r_dbspl);
	floatVec_minmax(ldbspl, nfreqs, &min_l_dbspl, &max_l_dbspl);

	for (i = 0; i < nfreqs; i++) {
		lsfspl[i] = pow(10.0, (min_l_dbspl - ldbspl[i]) / 20.0);
		rsfspl[i] = pow(10.0, (min_r_dbspl - rdbspl[i]) / 20.0);
	}

	return(1);
}

int cal_FigureAtten(syn_spec * ss, float l, float r, float *la, float *ra)
{
	float lref, rref;
	int ret_val, err;

	assert((ss->class == SC_SILENCE) || (ss->class == SC_TONE) ||
				(ss->class == SC_STACK) || (ss->class == SC_SWEEP) ||
				(ss->class == SC_NOISE) || (ss->class == SC_TWO) || 
				(ss->class == SC_HEMICLICK));

	err = 1;
	if (la != NULL) {
		lref = cal_GetOverallMinDBSPL(0);
		lref += 20.0 * log10(ss->lrms);
	}
	if (ra != NULL) {
		rref = cal_GetOverallMinDBSPL(1);
		rref += 20.0 * log10(ss->rrms);
	}

	if (err) {
		ret_val = 0;
		if (la != NULL) {
			if (l > lref) {
				*la = 0.0;
				ret_val |= CAL_LEFT_TOO_LOUD;
			}
			else if ((lref - l) > cal_max_atten) {
				ret_val |= CAL_LEFT_TOO_SOFT;
			}
			else
				*la = lref - l;
		}
		if (ra != NULL) {
			if (r > rref) {
				*ra = 0.0;
				ret_val |= CAL_RIGHT_TOO_LOUD;
			}
			else if ((rref - r) > cal_max_atten) {
				ret_val |= CAL_RIGHT_TOO_SOFT;
			}
			else
				*ra = rref - r;
		}
		return (ret_val);
	}
	else
		return (CAL_ERROR);
}

int cal_SafeFigureAtten(syn_spec * ss, float l, float r, float *la, float *ra)
{
	int i;
	char msg[1024] = "\0";

	i = cal_FigureAtten(ss, l, r, la, ra);
	if (i == CAL_ERROR) {
		sprintf(msg, "Failure to load/read calibration data");
		syn_alert(msg);
		return (0);
	}
	if (i != 0) {
		if (i & CAL_LEFT_TOO_LOUD)
			strcat(msg, "Requested dB SPL too loud in LEFT channel.");
		if (i & CAL_LEFT_TOO_SOFT)
			strcat(msg, "Requested dB SPL too soft in LEFT channel.");
		if (i & CAL_RIGHT_TOO_LOUD) {
			if (strcmp(msg, ""))
				strcat(msg, "\n");
			strcat(msg, "Requested dB SPL too loud in RIGHT channel.");
		}
		if (i & CAL_RIGHT_TOO_SOFT) {
			if (strcmp(msg, ""))
				strcat(msg, "\n");
			strcat(msg, "Requested dB SPL too soft in RIGHT channel.");
		}
		syn_notify(msg);
		return (0);
	}

	return (1);
}

/* this is just the inverse of FigureAtten:  given the attenuator settings,
 * it returns the dB SPLs produced */

int cal_FigureSPL(syn_spec *ss, float la, float ra, float *l, float *r)
{
	float lref, rref;
	int err;

	assert((ss->class == SC_SILENCE) || (ss->class == SC_TONE) ||
				(ss->class == SC_STACK) || (ss->class == SC_SWEEP) ||
				(ss->class == SC_NOISE) || (ss->class == SC_TWO) ||
				(ss->class == SC_HEMICLICK));

	err = 1;
	if (l != NULL) {
		lref = cal_GetOverallMinDBSPL(0);
		lref += 20.0 * log10(ss->lrms);
	}
	if (r != NULL) {
		rref = cal_GetOverallMinDBSPL(1);
		rref += 20.0 * log10(ss->rrms);
	}

	if (err) {
		if (l != NULL)
			*l = lref - la;
		if (r != NULL)
			*r = rref - ra;
		return (1);
	}
	else
		return (0);
}

int cal_SafeFigureSPL(syn_spec * ss, float la, float ra, float *l, float *r)
{
	if (cal_FigureSPL(ss, la, ra, l, r))
		return (1);
	else {
		syn_alert("Failure to load/read calibration data");
		return (0);
	}
}

int cal_GetData(int *ndata, float **freqs, float **lmags, float **rmags,
			float **lphis, float **rphis)
{
	(*ndata) = nfreqs;
	(*freqs) = ref_freqs;
	(*lmags) = ldbspl;
	(*rmags) = rdbspl;
	(*lphis) = lusphi;
	(*rphis) = rusphi;

	return(1);
}

/* returns scale factors to bring specified freq to the same
 * ** amplitude as the minimum point on the calibration curve.
 * ** also returns phase adjustment needed to delay all frequencies
 * ** equally.
 */

int cal_FigureCal(float freq, float *lscale, float *rscale,
			float *lphi, float *rphi)
{
	float us;

	assert(((lscale != NULL) && (lphi != NULL)) || ((rscale != NULL) && (rphi != NULL)));
	assert(nfreqs > 0);

	if ((lscale != NULL) && (lphi != NULL)) {
		lookup_mag_sf(freq, lscale, NULL);
		lookup_phi(freq, &us, NULL);
		*lphi = (us / 1.0e6) * freq * 2.0 * M_PI;
	}

	if ((rscale != NULL) && (rphi != NULL)) {
		lookup_mag_sf(freq, NULL, rscale);
		lookup_phi(freq, NULL, &us);
		*rphi = (us / 1.0e6) * freq * 2.0 * M_PI;
	}

	return (1);
}

int cal_get_nfreqs(void)
{
	return(nfreqs);
}

float *cal_get_ref_freqs(void)
{
	return(ref_freqs);
}

float *cal_get_rdbspl(void)
{
	return(rdbspl);
}

float *cal_get_ldbspl(void)
{
	return(ldbspl);
}

float *cal_get_rusphi(void)
{
	return(rusphi);
}

float *cal_get_lusphi(void)
{
	return(lusphi);
}

int cal_get_time(void)
{
	return(cal_time);
}
