/*
**	This file is part of XDowl
**	Copyright (c) 1994 Jamie Mazer
**	California Institute of Technology
**	<mazer@asterix.cns.caltech.edu>
*/

/******************************************************************
**  RCSID: $Id: cache.c,v 1.9 2005/01/30 09:32:11 bjorn Exp $
** Program: xdphys
**  Module: cache.c
**  Author: mazer,bjarthur
** Descrip: disk or memory cache handling routines
**
** Revision History (most recent last)
**
** Wed Feb  9 11:09:20 1994 mazer
**  derived from cache.c
**
** 97.3 bjarthur
**  moved SoundSpec_Copy, SoundSpec_Same to synth.c with SoundSpec_parse
**
** 98.12 bjarthur
**  make into independent library
**
*******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#ifndef __linux__
#include <malloc.h>
#endif /* __linux__ */

#include "misc.h"
#include "synth.h"
#include "calib.h"
#include "cache.h"
#include "math.h"

static CachePtr cacheList = NULL;

CachePtr cache_new(syn_spec *ss, int itd, int bc, int nsamps, int nchans,
      /*int rise, int fall*/ int lr, int fc)
{
  CachePtr src;

  assert((ss->class==SC_NOISE) || (ss->class==SC_STACK));

  src = (CachePtr)calloc(1, sizeof(CacheItem));

  src->nextsrc = cacheList;	/* link this node into the list */
  cacheList = src;		/* at the head.. */

  syn_spec_copy(ss,&(src->ss));
  src->itd = itd;
  src->bc = bc;
  src->nsamps = nsamps;
  src->nchans = nchans;
/*
  src->rise = rise;
  src->fall = fall;
*/
  src->lr = lr;
  src->fc = fc;
  src->cal_time = cal_GetTime();
  src->signal = (float*)calloc(nsamps*nchans, sizeof(float));
  return(src);
}

CachePtr cache_get(syn_spec *ss, int itd, int bc, int nsamps, int nchans,
      /*int rise, int fall,*/ int lr, int fc, int cal_time)
{
  CachePtr src;

  for (src = cacheList; src != NULL; src = src->nextsrc) {
    if (syn_spec_same(&(src->ss), ss) &&
        (src->itd == itd) &&
        (src->bc == bc) &&
        (src->nsamps > nsamps) &&
        (src->nchans == nchans) &&
/*
        (src->rise == rise) &&
        (src->fall == fall) &&
*/
        (src->lr == lr) &&
        (src->fc == fc) &&
        (src->cal_time == cal_time))
      return(src); }

  return(NULL);
}

int cache_delete(CachePtr src)
{
  CachePtr asrc, prev_src;

  prev_src = NULL;
  for (asrc = cacheList; asrc != NULL; asrc = asrc->nextsrc) {
    if (src == asrc) {
      if (prev_src == NULL) {
	cacheList  = asrc->nextsrc;
      } else {
	prev_src->nextsrc = asrc->nextsrc;
      }
      free(asrc->signal);
      free(asrc);
      return(1);
    }
    prev_src = asrc;
  }
  return(0);
}

void cache_delete_all(void)
{
  CachePtr ptr1,ptr2;

  ptr1=cacheList;
  while(ptr1!=NULL) {
    ptr2=ptr1->nextsrc;
    cache_delete(ptr1);
    ptr1=ptr2; }
}

int cache(syn_spec *ss, float *outbuf, int outlen,
      int itd, int bc, /*int rise, int fall,*/ int lr, int outskip)
{
  register int si, i,j;
  CachePtr src;

  assert((ss->class==SC_NOISE) || (ss->class==SC_STACK));

  if ((src = cache_get(ss,itd,bc,outlen,outskip,/*rise,fall,*/lr,
        syn_dafc(),cal_GetTime())) ==NULL) {
    return(0);
  }
  
  if (!syn_tone_rad_vary()) {
	  si = 0;
  } else {
	  si = floor(syn_get_rad() / (2.0*M_PI) * (src->nsamps - outlen)) * src->nchans;
  }
  
  if(lr&SYN_LEFT) {
    ss->lrms = src->ss.lrms;
	  if((ss->class==SC_STACK) && (syn_stack_rms()))
			ss->lrms *= sqrt((double) ss->parms.stack.num_freqs);
    for(i=j=0; j<outlen; j++, i+=outskip)
      outbuf[i]=(src->signal+si)[i]; }
  if(lr&SYN_RIGHT) {
    ss->rrms = src->ss.rrms;
	  if((ss->class==SC_STACK) && (syn_stack_rms()))
			ss->rrms *= sqrt((double) ss->parms.stack.num_freqs);
    for(i=j=0; j<outlen; j++, i+=outskip)
      outbuf[i+1]=(src->signal+si+1)[i]; }

	if(ss->class==SC_NOISE)
		ss->parms.noise.res = src->ss.parms.noise.res;
 
  return(1);
}
