/*
 * ** This file is part of XDowl
 * ** Copyright (c) 1994 Jamie Mazer
 * ** California Institute of Technology
 * ** <mazer@asterix.cns.caltech.edu>
 */

/******************************************************************
**  RCSID: $Id: synth.c,v 1.20 2000/11/28 06:41:03 cmalek Exp $
** Program: xdphys
**  Module: synth
**  Author: mazer, bjarthur
** Descrip: sound synthesis routines and tools
**
** Revision History (most recent last)
**
** Aug 96 bjarthur
**  changed all stimulus generation routines to synthesize *both* channels
**  at once.  a single xword* is passed in and they assume that there are
**  two interleaved buffers.  all refs to outskip were removed.  passing in
**  fc was also changed back (now they just use the global var is_daFc).
**  if syn_channel (prev aka syn_use_channel) = 1, then they calibrate.
**  the var lr specifies which channels to synth.
**
** Mon Oct 12 19:59:00 1992 mazer
**  creation date: split out from current iserver.c
**
** Mon Mar 22 12:04:04 1993 mazer
**  replaced currentWave stash array with a cleaner wave stack
**
** Fri May  7 23:33:06 1993 mazer
**  added syn_parseStim() function from itd.c
**
** Tue Jun  1 17:52:57 1993 mazer
**  added global syn_mscale as mult. scale factor
**
** Sat Jun 19 16:48:42 1993 mazer
**  split wavestack stuff out into synstack and deleted
**  syn_clipOutput() since it no longer applies to 16bit
**  waveforms..
**
** Tue Jan 18 13:59:51 1994 mazer
**  note: [ms_from, ms_to] specifed a true range, not a
**  starting point + duration pair ( i.e. syn_shapeOutput)
**
** Tue Jan 18 19:44:33 1994 mazer
**  added syn_env and syn_defineEnvelope()
**
** Mon Feb 21 18:56:45 1994 mazer
**  - deleted syn_simple() --> it's functionality can be accessed
**    by calling syn2_simple(NULL, ...) with essentially same args..
**  - same for syn_mono() --> use syn2_simple with itd=0 and
**    lr=LEFT or RIGHT
**  - deleted syn_parseStim() -- use syn2_toSpec()
**  - deleted syn_complex() -- use syn2_complex() -- eventually
**    should just use syn2_simple()..
**  - got rid of syn_rampOutput(), never used..
**
** Thu Feb 24 00:30:02 1994 mazer
**  merged synth2.c back into synth.c -- revision history for 
**  synth2 follow:
**
** Fri May  6 00:43:35 1994 mazer
**  added syn_channel = LEFT or RIGHT to specify which channel
**  is being synthesized -- this allows for using figure_scale()
**  to adjust for the earphone properties..
**  
**  -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
** Revision History (most recent last)  <-- from synth2.c
**
** Sat Jun 19 17:28:23 1993 mazer
**  creation date :: from synth.c
**  this is a version of synth.c that takes strings for 
**  synth specificiations instead of integers -- for more
**  general sounds!!  This comes up with the introduction
**  of band limited sound sources.
**
** Sun Jun 27 13:58:00 1993 mazer
**  bandlimited noise now generated with phasedBPnoise2().. this
**  should cache waveforms to disk/memory to provide for speed
**
** Mon Nov 22 21:58:13 1993 mazer
**  modified syn2_XXX() functions to allow for either a complex
**  stimspec string OR a simple integer specification of the
**  sounds source, as used by the old syn_XXX() routines..
**
** Tue Feb  8 01:16:20 1994 mazer
**  changed parsing of tonelists from (a,b,c) to stack=<range>
**  and converted syn2_simple() to handle tonelists directly
**
** Mon Feb 21 19:02:03 1994 mazer
**  added lr option to syn2_simple() and deleted syn2_mono()
**
** Wed Feb 23 22:01:34 1994 mazer
**  syn2_complex() is now gone -- and options added to
**  syn2_simple() allow it to replace syn2_complex. syn2_simple()
**  is about to become something else.. Did a global search
**  and replace to eliminated refs to syn2_complex..
**  Ta da -- syn2_simple() is now synthesize()
**
** Fri Nov 11 14:18:22 1994 mazer
**  added support for "nosnd" class
**
** Sep 96 bjarthur
**  moved envelope.c back in here.
**  moved cliperr.c in here
**  moved bits.c in here
**
** 97.3 bjarthur
**  moved noise.c and tone.c in here.  they were too short and
**  intimately connected to keep seperate.  their mod lists are
**  included below.
**  rename phased*() to syn_*()
**  moved SoundSpec_Copy, SoundSpec_Same from cache.c
**  rename SoundSpec to syn_spec, and SoundClass to syn_class
**  
**  -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
** Revision History (most recent last)  <----- from noise.c
**
** Aug 96 bjarthur
**  moved all the noise synth routines into one file, named it noise.c
**  deleted noise2.c.  there's three types of noise:
**    Noise_Uniform   generates random values in the time domain using uniform
**                    statistics, then FFTs to bandpass & calibrate
**    Noise_Gaussian  same, but uses gaussian statistics
**    Noise_FFT       starts in the freq domain and specifies bandpass and
**                    calibration, then iFFTs.
**
** Thu Jun 24 13:28:18 1993 mazer
**  split date :: from noise.c
**
** Sun Jun 27 14:05:59 1993 mazer
**  phasedBPnoisewave2() -> phasedBPnoisewave() and the old
**  old phasedBPnoisewave is now phaseBBnoisewave()..
**  
** Mon Jan 31 00:00:00 1994 mazer
**  changed call to dsp_bandpass to remove the rescaling of the
**  input signal to previous peak amplitude. This used to rescale
**  the filtered waveform to the same peak amp as the input wave-
**  form -- essentially changing the passband gain from unity
**  to something else (either >1 or <1 depending upon the filter
**  parameters, size of buffers, etc -- SEE NOTES ON PASSBAND
**  and STOPBAND RIPPLE PROBLEMS ca 02-Feb-94)
**
** Mon Feb 28 14:57:38 1994 mazer
**  cleanup..
**  
**  -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
** Revision History (most recent last)  <----- from tone.c
**  
** Tue Feb 18 01:58:24 1992 mazer
**  Added rad_vary feature. If this is enabled (by default), then
**  phasedSinewave() will try to pick a phase offset appropriate
**  to the rep number (from is_trial info). e.g. if there are three
**  sets of reps collected, the first set will have a reference
**  phase of 0 rad, the second pi/3 rad and the third 2pi/3 rad.
**  The itd's will still be correct, but the overall phase of the
**  stimuli will vary from repetition to repetion.
**  A similar feature has been implemented for phasedBPnoisewaves
**
** Mon May 31 22:26:08 1993 mazer
**  now tries to detect clippping on WAVE_ADD and WAVE_MULT
**  these are officially flagged with a -1 return value, and
**  should always be reported to stderr as well!!
**
** Tue Jun  1 17:53:29 1993 mazer
**  added support for global syn_mscale as mult. scale factor
**
** Wed Oct 13 16:27:56 1993 mazer
**  changed rad_vary to be settable from the xdowl control panel
**
** Tue Nov 23 16:55:53 1993 mazer
**  changed phasedSinewave() freq arg to float
**  changed phasedFMSweep() fstart, fstop args to floats
**
** Thu Nov 25 15:23:05 1993 mazer
**  added syn_tone_fixed_phase flag
**
** Tue Feb  8 22:27:29 1994 mazer
**  1. added fc args to functions instead of relying on is_daFc
**  2. added (finally) proper handling of the usphase argument
**     in phasedFMSweep (was ignored before)
**
** Mon Feb 21 21:34:48 1994 mazer
**  added warbling to phasedFMSweep()
**
** Fri May  6 00:42:52 1994 mazer
**  added earcal compenstation via figure_scale() for
**    -- tonelists
**    -- fmsweeps
**
** Fri Nov 11 14:17:41 1994 mazer
**  added phasedSilence() junk
**
** 97.3 bjarthur
**  moved rad_vary calculation to iserver.c to centralize it.
**  it was scary having it computed in so many places.
**
**  -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
**
** 97.4 bjarthur
**  replaced linearEnvelope() and syn_shapeOutput() with syn_envelope()
**   and removed the generalizations for arbitrary envelopes.  we only
**   use linear ones and the generalizations made everything confusing.
**  changed everything to work with floats instead of xwords right up
**   until the waveforms were actually used.  this should reduce noise.
**  got rid of WAVE_ stuff.  only WAVE_SET was ever used.
**  rewrote parse_synth_spec to only take integers, but convert them
**   to floats
**  update syn_spec_same() and syn_spec_copy() to include all syn_classes
**
** 97.7 bjarthur
**  added syn_twotone() to take 2 arbitrary sound_specs and combine them
**    for use in 2-tone paradigms.
**  changed syn_tone to use rms fields to calibrate just like everyone else
**
** 97.12 bjarthur
**  modified syn_spec_parse to much more rigorously check syntax.  if there
**  is any portion which doesn't fit, then it sends an syn_alert message
**
** 98.11 bjarthur
**  made into independent library
**  got rid of syn_mscale
**  got rid of "hp" syntax.  doesn't make sense for general purpose lib
**
** 99.01.30 bjarthur
**  added the syn_stack_rms flag function.  before, stacks were always scaled
**   to keep the total power constant.  now one has the option of keeping the
**   power in each band constant.  note that everything works find for the
**   stack=freq1,freq2,...,freqN spec, but for stack=freq1(iid1),freq2(iid2),
**   ...,freqN(iidN), the resulting abi is in general not as specified,
**   particularly for large values of iidx.
**
** 99.03.18 bjarthur
**  pulled bc syn_spec out into parameter just like itd.
**
** 99.04 bjarthur
**  added syn_use_cache for RE
**
*******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#ifndef __linux__
#include <malloc.h>
#else
#include <limits.h>
#endif /* __linux__ */
#include <limits.h>
#include <assert.h>
#include <ctype.h>
#include <unistd.h>
#include <nr.h>

#include "misc.h"
#include "synth.h"
#include "calib.h"
#include "cache.h"
#include "stimarray.h"

#ifdef __linux__
#define MAXSHORT SHRT_MAX
#define MINSHORT SHRT_MIN
#endif

static float correct_crc(float);
static void syn_clear(float *, int, int, int);
static float syn_rescale(float *, int, int);
/*
static void realft(float data[], unsigned long, int);
*/

#define REAL(mag,phi)   ((mag)*cos(phi))
#define IMAG(mag,phi)   ((mag)*sin(phi))

int syn_calibrate = 1;
int syn_use_cache = 1;
int (*syn_tone_rad_vary)(void);
int (*syn_stack_rms)(void);
int syn_tone_fixed_phase = 0;		/* flag to allow overriding of rad_vary */

float (*syn_bb_low_freq)(void);
float (*syn_bb_high_freq)(void);

int (*syn_dafc)(void);
double (*syn_get_rad)(void);


static float limited_uniform(long *seedp)
{
  register float u;

  do {
    u = (2.0 * ran1(seedp)) - 1.0;
  } while (u > 1.0 || u < -1.0);
  return(u);
}

static float syn_rescale(float *holdbuf, int outlen, int outskip)
{
	double max;
	int i, j;

	for (i = j = 0, max = 0.0; i < outlen; i++, j += outskip) {
		if (fabs((double) holdbuf[j]) > max)
			max = fabs((double) holdbuf[j]);
	}
	max = (FRANGE - 1.0) / max;		/* this -1.0 is a hack to avoid clipping */
	for (i = j = 0; i < outlen; i++, j += outskip)
		holdbuf[j] *= max;

	return (max);
}

static xword to_xword(float s)
{
	if (s > 0.0) {
		if (s < FRANGE)
			return ((xword) (s + 0.5));
		else
			return ((xword) (FRANGE));
	}
	else if (s < 0.0) {
		if (s > -FRANGE)
			return ((xword) (s - 0.5));
		else
			return ((xword) (-FRANGE));
	}
	else {
		return (0);
	}
}

int syn_transfer_to_outbuf(float *holdbuf, xword * outbuf, int outlen,
			int outskip)
{
	int i, j;
	int clip_err = 0;
	float s;

	for (i = j = 0; j < outlen; j++, i += outskip) {
		s = holdbuf[i];
		outbuf[i] = to_xword(s);
		if (overflow(s))
			clip_err++;
	}

	return (clip_err);
}

static void syn_envelope(float *fwave, int lr, int ms_from, int ms_to,
			int ms_rise, int ms_fall, int outskip)
{
	int from, to;
	int rise, fall;
	float *outbuf;
	int outlen;
	float m, mm;
	register int i;
	int tmp;

	from = (int) ((float) ms_from * (float) syn_dafc() / 1000.0);
	to = (int) ((float) ms_to * (float) syn_dafc() / 1000.0);
	outlen = to - from;

	outbuf = fwave + (outskip * from);

	rise = (int) ((float) ms_rise * (float) syn_dafc() / 1000.0);
	fall = (int) ((float) ms_fall * (float) syn_dafc() / 1000.0);

	m = 1.0 / (float) rise;
	tmp = rise * outskip;
	if (lr & SYN_LEFT)
		for (mm = 0.0, i = 0; i < tmp; i += outskip, mm += m)
			outbuf[i] *= mm;
	if (lr & SYN_RIGHT)
		for (mm = 0.0, i = 0; i < tmp; i += outskip, mm += m)
			outbuf[i + 1] *= mm;

	m = 1.0 / (float) fall;
	tmp = outlen * outskip;
	if (lr & SYN_LEFT)
		for (mm = 1.0, i = (outlen - fall) * outskip; i < tmp; i += outskip, mm -= m)
			outbuf[i] *= mm;
	if (lr & SYN_RIGHT)
		for (mm = 1.0, i = (outlen - fall) * outskip; i < tmp; i += outskip, mm -= m)
			outbuf[i + 1] *= mm;
}


/*
 * ** this accepts the following specification strings:
 * **  bb................. Broadband Noise 
 * **  ....0.............. synonym for bb
 * **  tone=NNN .......... pure tone of NNN hz
 * **  ....NNN............ synonym for pure tone
 * **  bp=lo-hi .......... band pass from f1-f2
 * **  ....[f1,f2]........ syn for bp=
 * **  ....[f1-f2]........ syn for bp=
 * **  CF=<cf>,BW=<w> .... bandpass from cf-(bw/2) to cf+(bw/2)
 * **  ....{cf,bw} ....... synonym for bove (cf=...)
 * **  stack=<range>...... pure tone list ("tone stack")
 * **  fmsweep=from-to:n.. FM Sweep (n == 1, if not specified..)
 */

char *synthesize_help(void)
{
	return ("Legal syn_classes are:\n"
 "NOSND                                 no sound at all\n"
 "BB                                    broadband noise\n"
 "TONE=nnn                              pure tone of nnn hz\n"
 "BP=lo-hi, or [lo,hi], or [lo-hi]      band pass noise from f1-f2\n"
 "CF=cf,BW=w, or {cf,bw}                bandpass from cf-(bw/2) to cf+(bw/2)\n"
 "STACK=range                           pure tone list (\"tone stack\")\n"
 "FM=from-to, or FMSWEEP=from-to:n      FM sweep in hz \n" 
 "rare_click=nnn                        negative square pulse of nnn samples\n" 
 "cond_click=nnn                        positive square pulse of nnn samples\n" 
 "click=nnn                             positive half cycle of a tone of nnn Hz\n");
}

static float *int_to_float(int *vec, int num)
{
	float *ret_val;
	int i;

	assert((ret_val = (float *) malloc(num * sizeof(float))) != NULL);
	for (i = 0; i < num; i++)
		ret_val[i] = (float) vec[i];
	free(vec);
	return (ret_val);
}

int syn_spec_parse(char *spec, int altf, syn_spec * s)
{
	int cf, width;
	char *SPEC, *p;
	int d1, d2, i, ntones, n;
	float tmp;
	int *tlist, *ilist;
	char remain[128];

	if (s == NULL)
		return (0);
	if (spec == NULL) {
		/* no spec string --> use pure tone or noise from <altf> */
		if (altf > 0) {
			s->class = SC_TONE;
			s->parms.tone.freq = (float) altf;
			sprintf(s->descr, "%d", altf);
		} else {
			s->class = SC_NOISE;
			s->parms.noise.low = syn_bb_low_freq();
			s->parms.noise.high = syn_bb_high_freq();
			sprintf(s->descr, "0");
		}
		SPEC = NULL;
	} else {
		/* make local copy of spec in uppercase to avoid case problems..
		 * convert spec to UPPERCASE to avoid problems.. */
		SPEC = (char *) calloc(strlen(spec) + 1, sizeof(char));
		strcpy(SPEC, spec);
		for (p = SPEC; *p; p++) {
			if (islower(*p)) {
#ifndef __linux__
				extern char toupper();
#endif				/* __linux__ */
				*p = toupper(*p);
			}
		}
		if (!strcmp(SPEC, "")) {
			s->class = SC_INVALID;
			syn_alert("Stim field is blank.");
		} else if ((i = sscanf(SPEC, "TONE=%d%s", &d1, remain))) {
			/* tone=1000... --> pure tone */
			s->class = SC_TONE;
			s->parms.tone.freq = (float) d1;
			sprintf(s->descr, "%s", SPEC);
			if (i != 1)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else
		    if ((i = sscanf(SPEC, "RARE_CLICK=%d%s", &d1, remain)))
		{
			s->class = SC_CLICK;
			s->parms.click.type = 0;	/* rarefaction click */
			s->parms.click.nsamples = (int) d1;
			sprintf(s->descr, "%s", SPEC);
			if (i != 1)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else
		    if ((i = sscanf(SPEC, "COND_CLICK=%d%s", &d1, remain)))
		{
			s->class = SC_CLICK;
			s->parms.click.type = 1;	/* condensation click */
			s->parms.click.nsamples = (int) d1;
			sprintf(s->descr, "%s", SPEC);
			if (i != 1)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else if ((i = sscanf(SPEC, "CLICK=%d%s", &d1, remain))) {
			s->class = SC_HEMICLICK;
			s->parms.hemiclick.freq = (float) d1;
			sprintf(s->descr, "%s", SPEC);
			if (i != 1)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else if (strncmp(SPEC, "NOSND", 5) == 0) {
			/* nosnd --> no sound..
			 */
			s->class = SC_SILENCE;
			sprintf(s->descr, "%s", SPEC);
			if (strlen(SPEC) != 5)
				syn_alert("\"%s\" unmatched in Stim field.",
				     SPEC + 5);
		} else if (strncmp(SPEC, "BB%s", 2) == 0) {
			/* bb --> broadband noise */
			s->class = SC_NOISE;
			s->parms.noise.low = syn_bb_low_freq();
			s->parms.noise.high = syn_bb_high_freq();
			sprintf(s->descr, "%s", SPEC);
			if (strlen(SPEC) != 2)
				syn_alert("\"%s\" unmatched in Stim field.",
				     SPEC + 2);
		}
#if(0)
		else if (strncmp(SPEC, "HP", 2) == 0) {
			/* hp --> high pass noise (4-13k avoid low inter-aural transmission) */
			s->class = SC_NOISE;
			s->parms.noise.low = syn_hp_low_freq;
			s->parms.noise.high = syn_hp_high_freq;
			if (strlen(SPEC) != 2)
				syn_alert("\"%s\" unmatched in Stim field.",
				     SPEC + 2);
		}
#endif
		else if (strncmp(SPEC, "STACK=", 6) == 0) {
			/* stack=1,2,3... --> tonelist */
			tlist = ilist = NULL;
			if (StimArray_Parse
			    (SPEC + 6, &tlist, &ilist, &ntones)) {
				s->class = SC_STACK;
				s->parms.stack.num_freqs = ntones;
				s->parms.stack.freqs =
				    int_to_float(tlist, ntones);
				s->parms.stack.iids =
				    int_to_float(ilist, ntones);
				sprintf(s->descr, "%s", SPEC);
			} else
				s->class = SC_INVALID;
		}
			else
		    if (
			(i =
			 sscanf(SPEC, "FMSWEEP=%d-%d:%d%s", &d1, &d2, &n,
				remain)) >= 2) {
			/* fmsweep=from-to */
			s->class = SC_SWEEP;
			s->parms.sweep.low = (float) d1;
			s->parms.sweep.high = (float) d2;
			s->parms.sweep.n = (i >= 3) ? n : 1;
			sprintf(s->descr, "%s", SPEC);
			if (i == 4)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else
		    if ((sscanf(SPEC, "BP=%d,%d%s", &d1, &d2, remain) >= 2)
			|| (sscanf(SPEC, "BP=%d-%d%s", &d2, &d2, remain) >=
			    2)) {
			/* bandpass'd noise: specified as locut and hicut */
			s->class = SC_NOISE;
			if (d1 < 0) {
				fprintf(stderr,
					"synth: bp low freq < 0 (=%d): using 0\n",
					d1);
				d1 = 0;
			}
			s->parms.noise.low = (float) d1;
			s->parms.noise.high = (float) d2;
			sprintf(s->descr, "%s", SPEC);
			if (i == 3)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else
		    if ((sscanf(SPEC, "[%d,%d]%s", &d1, &d2, remain) >= 2)
			|| (sscanf(SPEC, "[%d-%d]%s", &d1, &d2, remain) >=
			    2)) {
			s->class = SC_NOISE;
			if (d1 < 0) {
				fprintf(stderr,
					"synth: bp low freq < 0 (=%d): using 0\n",
					d1);
				d1 = 0;
			}

			s->parms.noise.low = (float) d1;
			s->parms.noise.high = (float) d2;
			sprintf(s->descr, "%s", SPEC);
			if (i == 3)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else
		    if (
			(sscanf(SPEC, "CF=%d,BW=%d%s", &cf, &width, remain)
			 >= 2)
			||
			(sscanf(SPEC, "CF=%d BW=%d%s", &cf, &width, remain)
			 >= 2)) {
			/* bandpass'd noise: specified as center freq and width */
			s->class = SC_NOISE;
			s->parms.noise.low = cf - ((float) width / 2.0);
			s->parms.noise.high = cf + ((float) width / 2.0);
			sprintf(s->descr, "%s", SPEC);
			if (s->parms.noise.low < 0) {
				fprintf(stderr,
					"synth: bp low freq < 0 (=%f): using 0\n",
					s->parms.noise.low);
				s->parms.noise.low = 0;
			}
			if (i == 3)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else if (sscanf(SPEC, "{%d,%d}%s", &cf, &width, remain)
			   >= 2) {
			s->class = SC_NOISE;
			s->parms.noise.low = cf - ((float) width / 2.0);
			s->parms.noise.high = cf + ((float) width / 2.0);
			sprintf(s->descr, "%s", SPEC);
			if (s->parms.noise.low < 0) {
				fprintf(stderr,
					"synth: bp low freq < 0 (=%f): using 0\n",
					s->parms.noise.low);
				s->parms.noise.low = 0;
			}

			if (i == 3)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else if ((i = sscanf(SPEC, "%d%s", &d1, remain)) >= 1) {
			/* pure tone or broadband noise (tone of 0 freq) synonym..  */
			if (d1 == 0) {
				s->class = SC_NOISE;
				s->parms.noise.low = syn_bb_low_freq();
				s->parms.noise.high = syn_bb_high_freq();
			} else {
				s->class = SC_TONE;
				s->parms.tone.freq = d1;
			}
			sprintf(s->descr, "%s", SPEC);
			if (i == 2)
				syn_alert("\"%s\" unmatched in Stim field.",
				     remain);
		} else {
			/* invalid or unparseable synth specification */
			s->class = SC_INVALID;
		}
	}

	if (s->class == SC_NOISE) {
		if (s->parms.noise.low > s->parms.noise.high) {
			tmp = s->parms.noise.low;
			s->parms.noise.low = s->parms.noise.high;
			s->parms.noise.high = tmp;
		}
	}

	if (SPEC)
		free(SPEC);
	return (1);
}

#define FREE(x) if(x!=NULL) free(x);

void syn_spec_free(syn_spec * ss)
{
	switch(ss->class) {
		case(SC_STACK):
      FREE(ss->parms.stack.freqs);
      FREE(ss->parms.stack.iids);
      break;
		default:
      break; }
}


void syn_spec_copy(syn_spec * from, syn_spec * to)
{
	int i;

	to->class = from->class;

	switch (from->class) {
		case (SC_TONE):
			to->parms.tone.freq = from->parms.tone.freq;
			to->parms.tone.freq = from->parms.tone.freq;
			break;
		case (SC_STACK):
			to->parms.stack.num_freqs = from->parms.stack.num_freqs;
			assert((to->parms.stack.freqs = (float *) malloc(
												to->parms.stack.num_freqs * sizeof(float))) != NULL);
			assert((to->parms.stack.iids = (float *) malloc(
												to->parms.stack.num_freqs * sizeof(float))) != NULL);
			for (i = 0; i < to->parms.stack.num_freqs; i++) {
				to->parms.stack.freqs[i] = from->parms.stack.freqs[i];
				to->parms.stack.iids[i] = from->parms.stack.iids[i];
			}
			break;
		case (SC_SWEEP):
			to->parms.sweep.low = from->parms.sweep.low;
			to->parms.sweep.high = from->parms.sweep.high;
			to->parms.sweep.n = from->parms.sweep.n;
			break;
		case (SC_NOISE):
			to->parms.noise.low = from->parms.noise.low;
			to->parms.noise.high = from->parms.noise.high;
			to->parms.noise.res = from->parms.noise.res;
			break;
		default:
			fprintf(stderr, "syn_spec_copy() currently only supports SC_NOISE\n");
			exit(0);
	}

	to->lrms = from->lrms;
	to->rrms = from->rrms;

#if(0)
	to->l20 = from->l20;
	to->r20 = from->r20;
#endif
}

int syn_spec_same(syn_spec * one, syn_spec * two)
{
	int i;

	if ((one == NULL) || (two == NULL))
		return (0);
	if (one->class != two->class)
		return (0);
	switch (one->class) {
		case (SC_TONE):
			if ((one->parms.tone.freq == two->parms.tone.freq))
				return (1);
			else
				return (0);
			break;
		case (SC_STACK):
			/* note, this gives a false negative if the freqs are just in
			 * a different order */
			if (one->parms.stack.num_freqs == two->parms.stack.num_freqs) {
				for (i = 0; i < one->parms.stack.num_freqs; i++) {
					if (one->parms.stack.freqs[i] != two->parms.stack.freqs[i])
						return (0);
					if (one->parms.stack.iids[i] != two->parms.stack.iids[i])
						return (0);
				}
				return (1);
			}
			else
				return (0);
			break;
		case (SC_SWEEP):
			if ((one->parms.sweep.low == two->parms.sweep.low) &&
						(one->parms.sweep.high == two->parms.sweep.high) &&
						(one->parms.sweep.n == two->parms.sweep.n))
				return (1);
			else
				return (0);
			break;
		case (SC_NOISE):
			if ((one->parms.noise.low == two->parms.noise.low) &&
						(one->parms.noise.high == two->parms.noise.high))
				return (1);
			else
				return (0);
			break;
/*
		case (SC_BC):
			if (one->parms.bc.bc == two->parms.bc.bc)
				return (1);
			else
				return (0);
			break;
*/
		case (SC_CLICK):
			if ((one->parms.click.nsamples == two->parms.click.nsamples)
				&& (one->parms.click.type == two->parms.click.type))
				return (1);
			else
				return (0);
			break;
		case (SC_HEMICLICK):
			if (one->parms.hemiclick.freq == two->parms.hemiclick.freq)
				return (1);
			else
				return (0);
			break;
		default:
			fprintf(stderr, "syn_spec_same() default case.\n");
			exit(0);
	}
}

int syn_silence(float *outbuf, int outlen, float *lscale, float *rscale,
			int lr, int outskip)
{
	register int i, j;

	if (lr & SYN_LEFT) {
		for (i = 0, j = 0; i < outlen; i++, j += outskip) {
			outbuf[j] = 0.0;
		}
	}
	if (lr & SYN_RIGHT) {
		outbuf++;
		for (i = 0, j = 0; i < outlen; i++, j += outskip) {
			outbuf[j] = 0.0;
		}
	}

	if (lscale != NULL)
		(*lscale) = 1.0;
	if (rscale != NULL)
		(*rscale) = 1.0;

	return (1);
}

/* ---------------------------------------------------------------------
 * syn_click:
 * 
 * The output is a rectangular pulse which has maximum voltage for n 
 * sampling points  (n = 1,2,..10,100).
 *
 * Will do both rarefaction (negative voltage) and condensation (positive
 * voltage) depending on value of the type field: 0=rarefaction, 
 * 1=condensation.
 * 
 * This is UNCALIBRATED.  
 *
 * Note: Can only shift ITD in integral multiples of sampling period.
 *
 * Note: nsamples is not checked here to ensure it is < length of outbuf
 * 
 * This function was included at the request of Catherine Carr.
 * Modified from full cycle to half cycle at the request of Kazuo Funabiki
 * --------------------------------------------------------------------- */

int syn_click(float *outbuf,  /* buffer to synth into */
				int *tk_dur,  /* duration in samples */
				int nsamples, /* length of click in samples */
				int type,     /* rarefaction(0)/condensation(1) */
				int usphase,  /* itd in microseconds, >0 means right leads*/
				int lr,       /* monaural/binaural */
				int outskip)  /* da_nchans */
{
	register int i, j;
	float period;
	int itd_shift,loffset,roffset;
	int high, low;

	assert((type==0) || (type==1));

	if (syn_calibrate) {
		fprintf(stderr, "synth: ignoring syn_calibrate for type SC_CLICK\n");
	}
	period = 1.0/syn_dafc() * 1.0e6;

	low=(int) floor(((float)usphase)/period);
	high=(int) (floor(((float)usphase)/period) + 1.0);

	if (abs(usphase-(int)(low*period)) < abs(usphase-(int)(high*period)))
		itd_shift=low;
	else
		itd_shift=high;

	if ((int)(itd_shift*period) != usphase)
		fprintf(stderr, "synth: syn_click: unable to get itd of %d, using %d \n",
			usphase, (int)(itd_shift*period));
		
	*tk_dur = nsamples+abs(itd_shift);

	if (itd_shift<0) {
		loffset=abs((outskip*itd_shift));
		roffset=0;
	} else {
		loffset=0;
		roffset=abs((outskip*itd_shift));
	}

	if (lr & SYN_LEFT) {
		if (type==0) 
			for (i = j = 0; j < *tk_dur; i += outskip, j++)
				outbuf[i+loffset] = MINSHORT;
		else
			for (i = j = 0; j < *tk_dur; i += outskip, j++)
				outbuf[i+loffset] = MAXSHORT;

	}
	if (lr & SYN_RIGHT) {
		if (type==0) 
			for (i = j = 0; j < *tk_dur; i += outskip, j++)
				outbuf[i + 1 + roffset] = MINSHORT;
		else
			for (i = j = 0; j < *tk_dur; i += outskip, j++)
				outbuf[i + 1 + roffset] = MAXSHORT;
	}

	return (1);
}


int syn_hemiclick(float *outbuf, int *tk_dur, float freq, int usphase,
			float *lscale, float *rscale, int lr, int outskip)
{
	register int i, j;
	register double omega, lphi, rphi;
	float tmp1, tmp2;
	float lrms, rrms;
	double rad_vary;
	float itd_phi;

	if (syn_tone_fixed_phase || !syn_tone_rad_vary()) {
		rad_vary = 0.0;
	} else
		rad_vary = syn_get_rad();

	if (syn_calibrate) {
		cal_FigureCal(freq, &lrms, &rrms, &tmp1, &tmp2);
		if (lscale != NULL)
			(*lscale) = 1.0 / lrms;
		if (rscale != NULL)
			(*rscale) = 1.0 / rrms;
		lphi = tmp1;
		rphi = tmp2;
	} else {
		if (lscale != NULL)
			(*lscale) = 1.0;
		if (rscale != NULL)
			(*rscale) = 1.0;
		lphi = rphi = 0.0;
	}
	omega = freq / (double) syn_dafc() * 2.0 * M_PI;	/* radians/tick */
	itd_phi = (usphase / 1.0e6) * freq * 2.0 * M_PI;
	lphi = -lphi + rad_vary;	/* usphase > 0 means right leads */
	rphi = -rphi + rad_vary + itd_phi;
	*tk_dur = (int) (1.0 / (2.0 * omega));

	if (lr & SYN_LEFT) {
		for (i = j = 0; j < *tk_dur; i += outskip, j++)
			outbuf[i] = (FRANGE * sin((omega * j) + lphi));
	}
	if (lr & SYN_RIGHT) {
		for (i = j = 0; j < *tk_dur; i += outskip, j++)
			outbuf[i + 1] = (FRANGE * sin((omega * j) + rphi));
	}

	return (1);
}

int syn_tone(float *outbuf, int outlen, float freq, int usphase,
			float *lscale, float *rscale, int lr, int outskip)
{
	register int i, j;
	register double omega, lphi, rphi;
	double rad_vary;
	float tmp1, tmp2;
	float itd_phi;
	float lrms, rrms;

	if (syn_tone_fixed_phase || !syn_tone_rad_vary()) {
		rad_vary = 0.0;
	}
	else 
		rad_vary = syn_get_rad();

	if (syn_calibrate) {
		cal_FigureCal(freq, &lrms, &rrms, &tmp1, &tmp2);
		if (lscale != NULL)
			(*lscale) = 1.0 / lrms;
		if (rscale != NULL)
			(*rscale) = 1.0 / rrms;
		lphi = tmp1;
		rphi = tmp2;
	}
	else {
		if (lscale != NULL)
			(*lscale) = 1.0;
		if (rscale != NULL)
			(*rscale) = 1.0;
		lphi = rphi = 0.0;
	}

	omega = freq / (double) syn_dafc() *2.0 * M_PI;		/* radians/tick */
	itd_phi = (usphase / 1.0e6) * freq * 2.0 * M_PI;
	lphi = -lphi + rad_vary;			/* usphase > 0 means right leads */
	rphi = -rphi + rad_vary + itd_phi;

	if (lr & SYN_LEFT) {
		for (i = j = 0; j < outlen; i += outskip, j++)
			outbuf[i] = (FRANGE * sin((omega * j) + lphi));
	}
	if (lr & SYN_RIGHT) {
		for (i = j = 0; j < outlen; i += outskip, j++)
			outbuf[i + 1] = (FRANGE * sin((omega * j) + rphi));
	}

	return (1);
}

/*   syn_stack_rms=0 specifies that the power in each band constant, regardless
**   of the number of tones.  note that everything works fine for the
**   stack=freq1,freq2,...,freqN spec, but for stack=freq1(iid1),freq2(iid2),
**   ...,freqN(iidN), the resulting abi is in general not as specified,
**   particularly for large values of iidx. */

int syn_stack(float *outbuf, int outlen, float *freqlist,
			float *iidlist, int ntones, long *seed, int usphase,
			float *lscale, float *rscale, int lr, int outskip)
{
	register int i, j, n;
	register float freq;
	register double omega, lphi, rphi;
	double rad_vary;
	float lmag, rmag, tmp1, tmp2;
	float itd_phi, rand_phi;
	float scale;

	if (syn_tone_fixed_phase || !syn_tone_rad_vary()) {
		rad_vary = 0.0;
	}
	else
		rad_vary = syn_get_rad();

	for (i = j = 0; i < outlen; i++, j += outskip)
		outbuf[j] = outbuf[j + 1] = 0.0;

	for (n = 0; n < ntones; n++) {
		freq = freqlist[n];

		if (syn_calibrate) {
			cal_FigureCal(freq, &lmag, &rmag, &tmp1, &tmp2);
			lphi = tmp1;
			rphi = tmp2;
		}
		else {
			lmag = rmag = 1.0;
			lphi = rphi = 0.0;
		}

		if (iidlist[n] != 0.0) {
			lmag *= pow(10.0, -iidlist[n] / 40.0);
			rmag *= pow(10.0, iidlist[n] / 40.0);
		}

		omega = freq / (double) syn_dafc() *2.0 * M_PI;		/* radians/tick */
		rand_phi = M_PI * limited_uniform(seed);
		itd_phi = (usphase / 1.0e6) * freq * 2.0 * M_PI;
		lphi = -lphi + rad_vary + rand_phi;		/* usphase > 0 means right leads */
		rphi = -rphi + rad_vary + rand_phi + itd_phi;

		if (lr & SYN_LEFT)
			for (i = j = 0; j < outlen; i += outskip, j++)
				outbuf[i] += (FRANGE * lmag * sin((omega * j) + lphi));
		if (lr & SYN_RIGHT)
			for (i = j = 0; j < outlen; i += outskip, j++)
				outbuf[i + 1] += (FRANGE * rmag * sin((omega * j) + rphi));
	}

	if (lr & SYN_LEFT) {
		scale = syn_rescale(outbuf, outlen, outskip);
		if (lscale != NULL)
			(*lscale) = scale;
	}
	else if (lscale != NULL)
		(*lscale) = 1.0;

	if (lr & SYN_RIGHT) {
		scale = syn_rescale(outbuf + 1, outlen, outskip);
		if (rscale != NULL)
			(*rscale) = scale;
	}
	else if (rscale != NULL)
		(*rscale) = 1.0;

	return (1);
}

int syn_sweep(float *outbuf, int outlen, float fstart,
			float fstop, float n, int usphase,
			float *lscale, float *rscale, int lr, int outskip)
{
	register int i, j;
	double omega_of_t;
	double phase_of_t, dt;
	double f, df;
	float lmag, rmag;
	float lphi, rphi;
	double rad_vary;
	float scale;

	if (syn_tone_fixed_phase || !syn_tone_rad_vary()) {
		rad_vary = 0.0;
	}
	else 
		rad_vary = syn_get_rad();

	f = fstart;
	df = (fstop - fstart) / ((double) outlen) * n;

	phase_of_t = 0.0;
	dt = 1.0 / (double) syn_dafc();

	for (j = i = 0; i < outlen; j += outskip, i++, f += df) {
		/* check to see if it's time to reverse the direction
		 * of the sweep..
		 */
		if (fstop > fstart) {
			if (f > fstop || f < fstart) {
				df = -df;
			}
		}
		else {
			if (f > fstart || f < fstop) {
				df = -df;
			}
		}

		if (syn_calibrate)
			cal_FigureCal(f, &lmag, &rmag, &lphi, &rphi);
		else {
			lmag = rmag = 1.0;
			lphi = rphi = 0.0;
		}

		omega_of_t = f * 2.0 * M_PI;
		phase_of_t += omega_of_t * dt;

		lphi = -lphi + rad_vary;		/* usphase > 0 means right leads */
		rphi = -rphi + rad_vary + (double) usphase *omega_of_t / 1.0e6;

		if (lr & SYN_LEFT) {
			outbuf[j] = FRANGE * lmag * sin(phase_of_t + lphi);
		}
		if (lr & SYN_RIGHT) {
			outbuf[j + 1] = FRANGE * rmag * sin(phase_of_t + rphi);
		}
	}

	if (lr & SYN_LEFT) {
		scale = syn_rescale(outbuf, outlen, outskip);
		if (lscale != NULL)
			(*lscale) = scale;
	}
	else if (lscale != NULL)
		(*lscale) = 1.0;

	if (lr & SYN_RIGHT) {
		scale = syn_rescale(outbuf + 1, outlen, outskip);
		if (rscale != NULL)
			(*rscale) = scale;
	}
	else if (rscale != NULL)
		(*rscale) = 1.0;

	return (1);
}

#ifndef next2
#define next2(n, n2) for ((n2) = 1; (n2) < (n); (n2) <<= 1) ;
#endif /* next2 */

int syn_noise_fft(float *outbuf, int outlen, long *seed,
			int usitd, float low, float high, float *res,
			float *lscale, float *rscale, int lr, int outskip)
{
	int fft_len, i, j;
	float freq, rand_phi, itd_phi;
	float *leftholdbuf, *rightholdbuf;
	float lmag, lphi, rmag, rphi;
	float scale;

	/* determine size and alloc mem */
	next2(outlen, fft_len);
	if ((leftholdbuf = (float *) malloc(sizeof(float) * fft_len)) == NULL) {
		fprintf(stderr, "syn_noise_fft: couldn't malloc left wave\n");
		/* beep(1); */
		return (-1);
	}
	if ((rightholdbuf = (float *) malloc(sizeof(float) * fft_len)) == NULL) {
		fprintf(stderr, "syn_noise_fft: couldn't malloc right wave\n");
		/* beep(1); */
		return (-1);
	}

	/* set desired mag & phi characteristics */
	leftholdbuf[0] = rightholdbuf[0] = 0.0;		/* DC */
	leftholdbuf[1] = rightholdbuf[1] = 0.0;		/* Fc */
	for (i = 1; i < fft_len / 2; i++) {
		freq = (float) i / (((float) fft_len - 1.0) / (float) syn_dafc());
		if ((freq < low) || (freq > high)) {
			leftholdbuf[(i * 2) + 0] = rightholdbuf[(i * 2) + 0] = 0.0;
			leftholdbuf[(i * 2) + 1] = rightholdbuf[(i * 2) + 1] = 0.0;
		}
		else {
			rand_phi = M_PI * limited_uniform(seed);
			itd_phi = (usitd / 1.0e6) * freq * 2.0 * M_PI;
			if (syn_calibrate)
				cal_FigureCal(freq, &lmag, &rmag, &lphi, &rphi);
			else {
				lmag = rmag = 1.0;
				lphi = rphi = 0.0;
			}
			/* normally, one would multiply by FRANGE*fft_len/2.0 here.  but with
			 * NR's implementation of the ifft, you have to multiply the result by
			 * 2.0/fft_len.  to save a step, we do it to the input. */
			lmag *= FRANGE;
			rmag *= FRANGE;

			leftholdbuf[(i * 2) + 0] = (float) REAL((double) lmag, (double) (rand_phi + lphi));
			leftholdbuf[(i * 2) + 1] = (float) IMAG((double) lmag, (double) (rand_phi + lphi));

			rightholdbuf[(i * 2) + 0] = (float) REAL((double) rmag,
						(double) (rand_phi - itd_phi + rphi));
			rightholdbuf[(i * 2) + 1] = (float) IMAG((double) rmag,
						(double) (rand_phi - itd_phi + rphi));
		}
	}

	/* do the inverse fft, rescale, and transfer to outputbuf */
	if (lr & SYN_LEFT) {
		realft(leftholdbuf - 1, (unsigned long) fft_len, -1);
		scale = syn_rescale(leftholdbuf + (fft_len - outlen) / 2, outlen, 1);
		scale *= sqrt(floor((high - low) / ((double) syn_dafc() / ((double) fft_len - 1.0))));
		if (lscale != NULL)
			(*lscale) = scale;
		for (i = j = 0; j < outlen; j++, i += outskip)
			outbuf[i] = (leftholdbuf + (fft_len - outlen) / 2)[j];
	}
	else if (lscale != NULL)
		(*lscale) = 1.0;

	if (lr & SYN_RIGHT) {
		realft(rightholdbuf - 1, (unsigned long) fft_len, -1);
		scale = syn_rescale(rightholdbuf + (fft_len - outlen) / 2, outlen, 1);
		scale *= sqrt(floor((high - low) / ((double) syn_dafc() / ((double) fft_len - 1.0))));
		if (rscale != NULL)
			(*rscale) = scale;
		for (i = j = 0; j < outlen; j++, i += outskip)
			outbuf[i + 1] = (rightholdbuf + (fft_len - outlen) / 2)[j];
	}
	else if (rscale != NULL)
		(*rscale) = 1.0;

	if (res != NULL)
		(*res) = (double) syn_dafc() / ((double) fft_len - 1.0);

	free(leftholdbuf);
	free(rightholdbuf);
	return (1);
}
#undef next2

int syn_noise(float *outbuf, int outlen, long *seed,
      int usitd, float low, float high, int bc100, float *res,
			float *lscale, float *rscale, int lr, int outskip)
{
	float *tmpbuf, lfoo, rfoo;
	long i, j;
	float abs_crc, crr_crc;
  float bc;

	bc = bc100/100.0;

	syn_noise_fft(outbuf, outlen, seed, usitd, low, high,
				res, lscale, rscale, lr, outskip);
  if(bc != 1.0) {
    assert((tmpbuf = (float*)malloc(outskip*outlen*sizeof(float))) != NULL);
    syn_noise_fft(tmpbuf, outlen, seed, usitd, low, high,
          NULL, &lfoo, &rfoo, SYN_RIGHT, outskip); }

	if (bc == 0.0) {
		for (i = j = 0; i < outlen; i++, j += outskip)
			(outbuf + 1)[j] = (tmpbuf + 1)[j];
		(*rscale) = rfoo; }
	else if (bc == -1.0) {
		for (i = j = 0; i < outlen; i++, j += outskip)
			(outbuf + 1)[j] = -(outbuf + 1)[j]; }
	else if (bc != 1.0) {
	  crr_crc = correct_crc(bc);
		abs_crc = (crr_crc < 0.0) ? -crr_crc : crr_crc;
		for (i = j = 0; i < outlen; i++, j += outskip)
			(outbuf + 1)[j] = crr_crc * (outbuf+1)[j] + (1.0-abs_crc) * (tmpbuf+1)[j];
		(*rscale) = sqrt(abs_crc * abs_crc * (*rscale) * (*rscale) +
					(1.0 - abs_crc) * (1.0 - abs_crc) * rfoo * rfoo);
		(*rscale) *= syn_rescale(outbuf + 1, outlen, 2); }

  if(bc != 1.0)
	  free(tmpbuf);

  return(1);
}

static float correct_crc(float crc)
{
	float c;

	if ((crc == 0.0) || (crc == 1.0) || (crc == -1.0))
		return crc;
	c = fabs(crc);
	c = 1.0 / (1.0 + sqrt(1.0 / c / c - 1.0));
	return (crc < 0.0) ? -c : c;
}

static void syn_clear(float *wave, int lr, int ntks, int outskip)
{
	int i, j;

	if (lr & SYN_LEFT)
		for (i = j = 0; i < ntks; i++, j += outskip)
			wave[j] = 0.0;
	if (lr & SYN_RIGHT)
		for (i = j = 0; i < ntks; i++, j += outskip)
			wave[j + 1] = 0.0;
}

/*
 * synthesize two independent sounds using the normal functions, and add
 * them together in the specified fashion.  ss1 is the 'fixed' sound, meaning
 * the abi and iid set by the call to cal_FigureAtten() sets the abi and
 * iid of ss1.  ss2 is the 'second' sound, meaning its abi is the fixed abi
 * + del_abi and its iid is the fixed iid + del_iid.  the delay, dur, itd,
 * rise/fall, and channel are all set independently for both sounds, but the
 * epoch is shared.  rms=1 specifies that when computing the abi and iid
 * of either sound, the rms of the time-series should be used.  the alternate,
 * rms=0, specifies that the maximum energy across all frequencies should
 * be used.  the resulting syn_spec is returned in ss1.
 *
 * KNOWN BUGS/LIMITATIONS: lr1 must be SYN_BOTH.  if it's not, then the
 * attenuator settings on the blank channel are wrong.  why?  because
 * how can one specify a del_iid and del_abi when the dB of one of the
 * reference channels is -Inf?
 *
 * the rms 'feature' might actually be a bug.  the relative intensities
 * between the two sounds are probably correct, but the overall intensity
 * seems to be much too loud.  didn't have time to test it...
 */

void two_sound_synthesize(syn_spec *ssout, float *fwave,
			xword *outbuf, int outsize, int da_channels,
			int rms, int ms_epoch, float del_db_iid, float del_db_abi,
			syn_spec * ss1, int ms_delay1, int ms_dur1, int us_itd1, int bc1,
			int ms_rise1, int ms_fall1, int lr1,
			syn_spec * ss2, int ms_delay2, int ms_dur2, int us_itd2, int bc2,
			int ms_rise2, int ms_fall2, int lr2)
{
	float *fwave1, *fwave2;
	float lscale, rscale, lrms, rrms;
	int i, j, tk_epoch;
	float del_db_left, del_db_right;
	float adj1, adj2;

	assert(lr1 == SYN_BOTH);
	assert(rms == 1);

	/* No SYN_CLICK allowed here */
	assert((rms == 1) ||
	       (((ss1->class == SC_TONE) || (ss1->class == SC_STACK) ||
		 (ss1->class == SC_HEMICLICK) || (ss1->class == SC_NOISE) ||
		 (ss1->class == SC_SILENCE)) &&
		((ss2->class == SC_TONE) || (ss2->class == SC_STACK) ||
		 (ss2->class == SC_HEMICLICK) || (ss2->class == SC_NOISE) ||
		 (ss2->class == SC_SILENCE))));

	tk_epoch = (int) ((float) ms_epoch * (float) syn_dafc() / 1000.0);

	if (fwave == NULL)
		assert(
		       (fwave1 =
			(float *) malloc(da_channels * tk_epoch *
					 sizeof(float))) != NULL);
	else
		fwave1 = fwave;
	assert(
	       (fwave2 =
		(float *) malloc(da_channels * tk_epoch *
				 sizeof(float))) != NULL);

	/* synthesize both sound specs */
	synthesize(ss1, fwave1, outbuf, outsize, da_channels,
		   ms_delay1, ms_dur1, ms_epoch, us_itd1, bc1, ms_rise1,
		   ms_fall1, lr1);
	synthesize(ss2, fwave2, outbuf, outsize, da_channels, ms_delay2,
		   ms_dur2, ms_epoch, us_itd2, bc2, ms_rise2, ms_fall2,
		   lr2);

	/* add an appropriately scaled #2 to #1 */
	if (!rms) {
		if (ss1->class == SC_STACK)
			adj1 = sqrt((double) ss1->parms.stack.num_freqs);
		else if (ss1->class == SC_NOISE)
			adj1 =
			    sqrt((double)
				 floor(
				       (ss1->parms.noise.high -
					ss1->parms.noise.low) /
				       ss1->parms.noise.res));
		else
			adj1 = 1.0;
		if (ss2->class == SC_STACK)
			adj2 = sqrt((double) ss2->parms.stack.num_freqs);
		else if (ss2->class == SC_NOISE)
			adj2 =
			    sqrt((double)
				 floor(
				       (ss2->parms.noise.high -
					ss2->parms.noise.low) /
				       ss2->parms.noise.res));
		else
			adj2 = 1.0;
	} else
		adj1 = adj2 = 1.0;

	if (lr2 & SYN_LEFT) {
		lscale = ss1->lrms / ss2->lrms;
		del_db_left = del_db_abi - del_db_iid / 2.0;
		lscale *= (float) pow(10.0, (double) (del_db_left) / 20.0);
		if (!(lr1 & SYN_LEFT))
			syn_clear(fwave1, SYN_LEFT, tk_epoch, da_channels);
		for (i = 0, j = 0; i < tk_epoch; i++, j += da_channels)
			fwave1[j] =
			    fwave1[j] * adj1 + fwave2[j] * adj2 * lscale;
	}
	if (lr2 & SYN_RIGHT) {
		rscale = ss1->rrms / ss2->rrms;
		del_db_right = del_db_abi + del_db_iid / 2.0;
		rscale *=
		    (float) pow(10.0, (double) (del_db_right) / 20.0);
		if (!(lr1 & SYN_RIGHT))
			syn_clear(fwave1, SYN_RIGHT, tk_epoch,
				  da_channels);
		for (i = 0, j = 0; i < tk_epoch; i++, j += da_channels)
			fwave1[j + 1] =
			    fwave1[j + 1] * adj1 + fwave2[j +
							  1] * adj2 *
			    rscale;
	}

	/* rescale the sum to fit in the buffers */
	if ((lr1 | lr2) & SYN_LEFT) {
		lrms = syn_rescale(fwave1, tk_epoch, da_channels);
		ssout->lrms =
		    ss1->lrms *
		    lrms /* * sqrt(adj1 * adj1 + adj2 * adj2) */ ;
	} else
		ssout->lrms = 1.0;

	if ((lr1 | lr2) & SYN_RIGHT) {
		rrms = syn_rescale(fwave1 + 1, tk_epoch, da_channels);
		ssout->rrms =
		    ss1->rrms *
		    rrms /* * sqrt(adj1 * adj1 + adj2 * adj2) */ ;
	} else
		ssout->rrms = 1.0;

	/* put the result in the output buffers if specified */
	if (fwave == NULL) {
		if ((lr1 | lr2) & SYN_LEFT)
			syn_transfer_to_outbuf(fwave1, outbuf, tk_epoch,
					       da_channels);
		if ((lr1 | lr2) & SYN_RIGHT)
			syn_transfer_to_outbuf(fwave1 + 1, outbuf + 1,
					       tk_epoch, da_channels);
		free(fwave1);
	}

	free(fwave2);

	ssout->class = SC_TWO;
}

void synthesize(syn_spec *ss, float *fwave,
			xword *outbuf, int outsize, int da_channels,
			int ms_delay, int ms_dur, int ms_epoch,
			int us_itd, int bc, int ms_rise, int ms_fall, int lr)
{
	int tk_delay, tk_dur, tk_epoch;
	CacheItem *cache_item;
	static long seed = -1;
	float *tmp;

	assert((bc == 100)
	       || ((ss->class == SC_NOISE) && (lr == SYN_BOTH)));

	assert(ms_dur > 0);
	tk_dur = (int) ((float) ms_dur * (float) syn_dafc() / 1000.0);
	tk_delay = (int) ((float) ms_delay * (float) syn_dafc() / 1000.0);
	tk_epoch = (int) ((float) ms_epoch * (float) syn_dafc() / 1000.0);

	if (fwave == NULL)
		assert(
		       (tmp =
			(float *) malloc(da_channels * tk_epoch *
					 sizeof(float))) != NULL);
	else
		tmp = fwave;
	syn_clear(tmp, lr, tk_epoch, da_channels);
	tmp += (tk_delay * da_channels);

	switch (ss->class) {
	case SC_SILENCE:
		syn_silence(tmp, tk_dur, &(ss->lrms), &(ss->rrms), lr,
			    da_channels);
		break;
	case SC_TONE:
		syn_tone(tmp, tk_dur, ss->parms.tone.freq, us_itd,
			 &(ss->lrms), &(ss->rrms), lr, da_channels);
		break;
	case SC_STACK:
		if (syn_use_cache) {
			if (!cache
			    (ss, tmp, tk_dur, us_itd, bc,
			     /*ms_rise,ms_fall, */ lr, da_channels)) {
				cache_item =
				    cache_new(ss, us_itd, bc, 2 * tk_dur,
					      da_channels,
					      /*ms_rise, ms_fall, */ lr,
					      syn_dafc());
				syn_stack(cache_item->signal, 2 * tk_dur,
					  cache_item->ss.parms.stack.freqs,
					  cache_item->ss.parms.stack.iids,
					  cache_item->ss.parms.stack.
					  num_freqs, &seed, us_itd,
					  &(cache_item->ss.lrms),
					  &(cache_item->ss.rrms), lr,
					  da_channels);
				cache(ss, tmp, tk_dur, us_itd, bc,
				      /*ms_rise,ms_fall, */ lr,
				      da_channels);}
		} else {
			syn_stack(tmp, tk_dur, ss->parms.stack.freqs,
				  ss->parms.stack.iids,
				  ss->parms.stack.num_freqs, &seed, us_itd,
				  &(ss->lrms), &(ss->rrms), lr,
				  da_channels);}
		break;
	case SC_SWEEP:
		syn_sweep(tmp, tk_dur, ss->parms.sweep.low,
			  ss->parms.sweep.high, ss->parms.sweep.n, us_itd,
			  &(ss->lrms), &(ss->rrms), lr, da_channels);
		break;
	case SC_NOISE:
		if (syn_use_cache) {
			if (!cache
			    (ss, tmp, tk_dur, us_itd, bc,
			     /*ms_rise,ms_fall, */ lr, da_channels)) {
				cache_item =
				    cache_new(ss, us_itd, bc, 2 * tk_dur,
					      da_channels,
					      /*ms_rise, ms_fall, */ lr,
					      syn_dafc());
				assert((ss->parms.noise.low >= 0)
				       && (ss->parms.noise.high >= 0));
				syn_noise(cache_item->signal, 2 * tk_dur,
					  &seed, us_itd,
					  cache_item->ss.parms.noise.low,
					  cache_item->ss.parms.noise.high,
					  bc,
					  &(cache_item->ss.parms.noise.
					    res), &(cache_item->ss.lrms),
					  &(cache_item->ss.rrms), lr,
					  da_channels);
				cache(ss, tmp, tk_dur, us_itd, bc,
				      /*ms_rise,ms_fall, */ lr,
				      da_channels);}
		} else {
			syn_noise(tmp, tk_dur, &seed, us_itd,
				  ss->parms.noise.low,
				  ss->parms.noise.high, bc,
				  &(ss->parms.noise.res), &(ss->lrms),
				  &(ss->rrms), lr, da_channels);
		}
		break;
	case SC_CLICK:
		syn_click(tmp, &tk_dur, ss->parms.click.nsamples,
			  ss->parms.click.type, us_itd, lr, da_channels);
		break;
	case SC_HEMICLICK:
		syn_hemiclick(tmp, &tk_dur, ss->parms.hemiclick.freq, us_itd,
			      &(ss->lrms), &(ss->rrms), lr, da_channels);
		break;
	default:
		fprintf(stderr,
			"synthesize() called with invalid syn_spec.class\n");
		exit(0);
	}

	if ((ss->class != SC_CLICK) && (ss->class != SC_HEMICLICK))
		syn_envelope(tmp, lr, 0, ms_dur, ms_rise, ms_fall,
			     da_channels);

	if (fwave == NULL) {
		tmp -= (tk_delay * da_channels);
		if (lr & SYN_LEFT)
			syn_transfer_to_outbuf(tmp, outbuf, tk_epoch,
					       da_channels);
		if (lr & SYN_RIGHT)
			syn_transfer_to_outbuf(tmp + 1, outbuf + 1,
					       tk_epoch, da_channels);
		free(tmp);
	}
}
