/******************************************************************
**  RCSID: $Id: syn-matlab.c,v 1.14 2000/11/27 02:35:52 cmalek Exp $
** Program: synth
**  Module: syn-matlab.c
**  Author: cmalek
**
**  NOTE:  Although synth can accept > 2 output channels, only
**         channels 1 and 2 can be calibrated.  I have forced syn-matlab
**         to use only two channels.
**
*******************************************************************/
	
  
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#ifdef __linux__
#include <unistd.h>
#else
#include <sys/types.h>
#endif /* __linux__ */
#include <sys/stat.h>
#include <sys/param.h>
#include <string.h>
#ifdef sparc
#include <strings.h>
#else
#include <string.h>
#endif
#include <assert.h>
#include <errno.h>
#include <mex.h>
#include "synth.h"
#include "calib.h"

/*-----------------------------------------------------------------------
  Typedefs
  ----------------------------------------------------------------------- */
typedef struct syn_data_struct {
	int		ldb;
	float	latten;
	int		rdb;
	float	ratten;
	int		dur;		
	int 	rise;
	int		fall;
	syn_spec 	ss;
	int		delay;
	int 	itd;
	int 	bc;
	int 	side;
	char	 spec[256];
	int		ignore;
} SynData;

/*-----------------------------------------------------------------------
  Global Variables
  ----------------------------------------------------------------------- */
struct synmat_global_vars {
	int 	atexit_registered;
	int 	debugflag;
	double 	bb_low;
	double 	bb_high;
	int 	daFc;
	int 	da_nchans;
	int 	epoch;
	int 	rad_vary;
	int 	cal_loaded;
	int 	twosound_rms;
	int 	stack_rms;
	int 	inited;
} globals = {0,0,500.0,13000.0,48000,2,1000,0,0,1,1,0};



/*-----------------------------------------------------------------------
  Prototypes
  ----------------------------------------------------------------------- */
static void synthExitFcn(void);
static void usage(void);
static void init(int, mxArray *plhs[], int, const mxArray *prhs[]);
static void synth(int, mxArray *plhs[], int, const mxArray *prhs[]);
static void synth_click(int, mxArray *plhs[], int, const mxArray *prhs[]);
static void twosound(int, mxArray *plhs[], int, const mxArray *prhs[]);
static void set_calib(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]);
static void get_calib(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]);
static void synthesize_usage(void);
static void click_usage(void);
static void export_globals(void);
static void update_local_from_globals(void);
static int get_field_double(mxArray *, int, char *, double *);
static int get_field_int(mxArray *, int, char *, int *);
static void set_field_double(mxArray *, int, char *, double);
static void set_field_int(mxArray *, int, char *, int);
static void init_syn_data(SynData *);
static int get_field_string(mxArray *, int, char *, char **);
static void set_field_string(mxArray *, int, char *, char *);
static int parse_syn_data(SynData *, const mxArray *args[]);
static int parse_click_data(SynData *, int, const mxArray *args[]);
static int parse_twosound_data(SynData *, SynData *, const mxArray *args[]);
static int parse_sound_data(SynData *, int, const mxArray *args[]);
/*static int parse_side(SynData *, int, const mxArray *args[]);*/
static int check_syn_data(SynData *, char *);
static int check_click_data(SynData *, char *);
static void twosound_usage(void);
static void compute_iid_abi(SynData *, SynData *, float *, float *);
static void compute_stack_db(SynData *);

static void synth_prepare_flags(SynData *, const mxArray *args[]);
static void click_prepare_flags(SynData *, const mxArray *args[]);
static void synth_prepare_output(SynData *, xword *, int, mxArray *args[]);
static void click_prepare_output(SynData *, xword *, int, mxArray *args[]);
static void synth_setup_calib(void);
static void twosound_prepare_flags(SynData *, SynData *, const mxArray *args[]);
static int parse_epoch(SynData *, int i, const mxArray *args[]);

static int synmat_daFc(void);
static int synmat_stack_rms(void);
static float synmat_bb_low_freq(void);
static float synmat_bb_high_freq(void);
static double synmat_get_rad(void);
static int synmat_tone_rad_vary(void);

static void synmat_init(void);

/*----------------------------------------------------------------------- */
/*----------------------------------------------------------------------- */

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
	char command[30];

	if (nrhs < 1)  
		usage();

	if (!globals.atexit_registered) {
		if (mexAtExit(synthExitFcn))
			mexPrintf("synth: synthExitFcn not registered properly\n");
		globals.atexit_registered=1;
		/* Assume we're using TDT PA4s */
	}

	synmat_init();

	if (!mxIsChar(prhs[0])) 
		usage();

	mxGetString(prhs[0], command, 29);

	if (!strncmp(command,"init",4)) init(nlhs, plhs, nrhs, prhs);
	if (!strncmp(command,"set_calib",9)) set_calib(nlhs, plhs, nrhs, prhs);
	if (!strncmp(command,"get_calib",9)) get_calib(nlhs, plhs, nrhs, prhs);
	if (!strncmp(command,"synthesize",10)) synth(nlhs, plhs, nrhs, prhs);
	if (!strncmp(command,"click",5)) synth_click(nlhs, plhs, nrhs, prhs);
	if (!strncmp(command,"twosound",8)) twosound(nlhs, plhs, nrhs, prhs);
}

/* ------------------------------------------------------------------------
   synthExitFcn
   ------------------------------------------------------------------------ */
static void synthExitFcn(void) {

	mexPrintf("synth: WARNING: MEX-file is being unloaded\n");

}


/* ------------------------------------------------------------------------
   usage
   ------------------------------------------------------------------------ */
static void usage(void) 
{
	mexPrintf("Usage: synth('init');\n");
	mexPrintf("       synth('set_calib', freqs, mags, phases);\n");
	mexPrintf("       [freqs, mags, phases] = synth('get_calib');\n");
	mexPrintf("       synth('get_calib', freqs, mags, phases);\n");
	mexPrintf("       [data att] = synth('synthesize', <stim>, ...);\n");
	mexPrintf("       [data] = synth('click', <stim>, ...);\n");
	mexErrMsgTxt("       [data att] = synth('twosound', <stim1>, ...);\n");
}

/* ---------------------------------------------------------------------
   synmat_init
   -------------------------------------------------------------------- */
static void synmat_init(void)
{
	if (!globals.inited) {
		syn_get_rad = synmat_get_rad;

		syn_tone_rad_vary = synmat_tone_rad_vary;

		syn_bb_low_freq = synmat_bb_low_freq;
		syn_bb_high_freq = synmat_bb_high_freq;

		syn_dafc = synmat_daFc;

		syn_stack_rms = synmat_stack_rms;

		if (cal_max_atten == 0.0)
			cal_max_atten=90.0;

		syn_use_cache=0;

		globals.inited=1;
	}
}

/* ------------------------------------------------------------------------
   init: just export syn_info
   ------------------------------------------------------------------------ */
static void init(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) 
{
	if (nrhs != 1)
		mexErrMsgTxt("usage: synth('init');\n");
	if (nlhs > 0)
		mexErrMsgTxt("usage: synth('init');\n");

	update_local_from_globals();

	export_globals();
}

/* ------------------------------------------------------------------------
   set_calib: Assume that we always have 2 channels of calibration data.
   ------------------------------------------------------------------------ */
static void set_calib(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) 
{
	int nfreqs, i;
	double *f, *m, *p;
	float *freqs, *lmags, *lphis, *rmags, *rphis;

	if (nrhs != 4)
		mexErrMsgTxt("usage: synth('set_calib', freqs, mags, phases);\n");
	if (nlhs > 0)
		mexErrMsgTxt("usage: synth('set_calib', freqs, mags, phases);\n");


	update_local_from_globals();

	/* prhs[1] = freqs, should be Mx1, a vector */
	/* prhs[2] = mags, should be Mx2, one column for each side  */
	/* prhs[3] = phases, should be Mx2, one column for each side*/
	if ((mxGetN(prhs[1])!=1) || (mxGetM(prhs[2])!=2) || 
	    (mxGetM(prhs[3])!=2)) {
	  mexErrMsgTxt("set_calib: freqs should be Mx1; mags and phases must be 2xM\n");
	}
	  
	if ((mxGetM(prhs[1]) != mxGetN(prhs[2])) || 
		(mxGetN(prhs[2]) != mxGetN(prhs[3]))) {
		mexPrintf("set_calib: freqs, mags, and phases must all be the);\n");
		mexErrMsgTxt("           same size.\n");
	}

	nfreqs=mxGetM(prhs[1]);

	mxAssert(((freqs = (float *) mxCalloc(nfreqs, sizeof(float))) != NULL), 
		"set_calib: mxCalloc returns NULL!");
	mxAssert(((lmags = (float *) mxCalloc(nfreqs, sizeof(float))) != NULL), 
		"set_calib: mxCalloc returns NULL!");
	mxAssert(((rmags = (float *) mxCalloc(nfreqs, sizeof(float))) != NULL), 
		"set_calib: mxCalloc returns NULL!");
	mxAssert(((lphis = (float *) mxCalloc(nfreqs, sizeof(float))) != NULL), 
		"set_calib: mxCalloc returns NULL!");
	mxAssert(((rphis = (float *) mxCalloc(nfreqs, sizeof(float))) != NULL), 
		"set_calib: mxCalloc returns NULL!");

	f=mxGetPr(prhs[1]);
	m=mxGetPr(prhs[2]);
	p=mxGetPr(prhs[3]);

	for (i=0; i<nfreqs; i++) {
		freqs[i]=(float) f[i];
		lmags[i]=(float) m[2*i];
		rmags[i]=(float) m[2*i+1];
		lphis[i]=(float) p[2*i];
		rphis[i]=(float) p[2*i+1];
	}

	cal_SetData(nfreqs, freqs, lmags, rmags, lphis, rphis, 0);
	globals.cal_loaded = 1;

	mxFree(rphis);
	mxFree(lphis);
	mxFree(rmags);
	mxFree(lmags);
	mxFree(freqs);

	export_globals();
}

/* ------------------------------------------------------------------------
   get_calib 
   ------------------------------------------------------------------------ */
static void get_calib(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) 
{
	int nfreqs, i;
	double *f, *m, *p;
	float *freqs, *lmags, *lphis, *rmags, *rphis;

	if (nrhs != 1)
		mexErrMsgTxt("usage: [freqs, mags, phases] = synth('get_calib');\n");
	if (nlhs != 3)
		mexErrMsgTxt("usage: [freqs, mags, phases] = synth('get_calib');\n");


	update_local_from_globals();

	if (!globals.cal_loaded)
		mexErrMsgTxt("get_calib: no calibration data loaded.");

	nfreqs = cal_get_nfreqs(); 
	freqs = cal_get_ref_freqs();
	lmags = cal_get_ldbspl();
	rmags = cal_get_rdbspl();
	lphis = cal_get_lusphi();
	rphis = cal_get_rusphi();

	/* freqs */
	plhs[0]=mxCreateDoubleMatrix(nfreqs,1,mxREAL);
	f=mxGetPr(plhs[0]);
	for (i=0; i<nfreqs; i++) 
		f[i] = freqs[i];

	/* mags */
	plhs[1]=mxCreateDoubleMatrix(2,nfreqs,mxREAL);
	m=mxGetPr(plhs[1]);

	/* phis */
	plhs[2]=mxCreateDoubleMatrix(2,nfreqs,mxREAL);
	p=mxGetPr(plhs[2]);

	for (i=0; i<nfreqs; i++) {
		m[2*i] = lmags[i];
		m[2*i+1] = rmags[i];
		p[2*i] = ((2.0 * M_PI * freqs[i])/1.0e6) * lphis[i];
		p[2*i+1] = ((2.0 * M_PI * freqs[i])/1.0e6) * rphis[i];
	}

	export_globals();
}

/* ---------------------------------------------------------------------
   synth
   -------------------------------------------------------------------- */
static void synth(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) 
{
	SynData	flags;
	double *output;
	xword *outbuf;
	int outsize;
	int retval;
	double *atten;
	char msg[1024] = "\0";

	if (nrhs != 11)
		synthesize_usage();
	if (nlhs > 2)
		synthesize_usage();

	update_local_from_globals();

	synth_prepare_flags(&flags, prhs);
	synth_setup_calib();

	outsize = (int) (((double) globals.epoch/1000) * (double) globals.daFc);
	mxAssert(((outbuf = (xword *) mxCalloc(outsize*globals.da_nchans, 
		sizeof(xword)))!=NULL), "synthesize: mxCalloc returns NULL!");

	synthesize(&flags.ss, NULL, outbuf, outsize, globals.da_nchans, 
		flags.delay, flags.dur, globals.epoch, flags.itd, flags.bc,
		flags.rise, flags.fall, flags.side);

	retval = cal_FigureAtten(&flags.ss, flags.ldb, flags.rdb, &flags.latten, 
		&flags.ratten);

	if (retval != 0) {
		if ((retval & CAL_LEFT_TOO_LOUD) && (flags.side & SYN_LEFT))
			strcat(msg, "Requested dB SPL too loud in LEFT channel.");
		if ((retval & CAL_LEFT_TOO_SOFT) && (flags.side & SYN_LEFT))
			strcat(msg, "Requested dB SPL too soft in LEFT channel.");
		if ((retval & CAL_RIGHT_TOO_LOUD) && (flags.side & SYN_RIGHT)){
			if (strcmp(msg, ""))
				strcat(msg, "\n");
			strcat(msg, "Requested dB SPL too loud in RIGHT channel.");
		}
		if ((retval & CAL_RIGHT_TOO_SOFT) && (flags.side & SYN_RIGHT)){
			if (strcmp(msg, ""))
				strcat(msg, "\n");
			strcat(msg, "Requested dB SPL too soft in RIGHT channel.");
		}
		fprintf(stderr, "%s\n", msg);

		if (flags.side == SYN_BOTH) {
			flags.latten = -1;
			flags.ratten = -1;
		} else {
			if (((retval & CAL_RIGHT_TOO_LOUD) || 
				(retval & CAL_RIGHT_TOO_SOFT)) && (flags.side == SYN_RIGHT)) {
				flags.ratten = -1;
			}
			if (((retval & CAL_LEFT_TOO_LOUD) || 
				(retval & CAL_LEFT_TOO_SOFT)) && (flags.side == SYN_LEFT)) {
				flags.latten = -1;
			}
				
		}
	}

	synth_prepare_output(&flags, outbuf, outsize, plhs);

	export_globals();
}

/* ---------------------------------------------------------------------
   synth_click: 
   
   	SC_CLICK stimuli take strange args, so we have a different function

	They are also not calibrated.
   -------------------------------------------------------------------- */
static void synth_click(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) 
{
	SynData	flags;
	xword *outbuf;
	int outsize;

	if (nrhs != 5)
		click_usage();
	if (nlhs > 1)
		click_usage();

	update_local_from_globals();

	click_prepare_flags(&flags, prhs);

	outsize = (int) (((double) globals.epoch/1000) * (double) globals.daFc);
	mxAssert(((outbuf = (xword *) mxCalloc(outsize*globals.da_nchans, 
		sizeof(xword)))!=NULL), "synthesize: mxCalloc returns NULL!");

	synthesize(&flags.ss, NULL, outbuf, outsize, globals.da_nchans, 
		flags.delay, flags.dur, globals.epoch, flags.itd, flags.bc,
		flags.rise, flags.fall, flags.side);

	click_prepare_output(&flags, outbuf, outsize, plhs);

	export_globals();
}


/* ---------------------------------------------------------------------
   synth_prepare_output
   -------------------------------------------------------------------- */
static void synth_prepare_output(SynData *flags, xword *outbuf, int outsize, 
					 mxArray *plhs[])
{
	double *atten;
	double *output;
	int i,j;

	plhs[0]=mxCreateDoubleMatrix(outsize,globals.da_nchans,mxREAL);
	output=mxGetPr(plhs[0]);
	for (i=0; i<outsize; i++) {
		for (j=0; j<globals.da_nchans; j++) 
			output[j*outsize+i]= (double) outbuf[globals.da_nchans*i+j];
	}
	mxFree(outbuf);

	plhs[1]=mxCreateDoubleMatrix(1,2,mxREAL);
	atten=mxGetPr(plhs[1]);

	atten[0]=flags->latten;
	atten[1]=flags->ratten;

	if (flags->side == SYN_LEFT) 
		atten[1]=cal_max_atten;
	 else
		 if (flags->side == SYN_RIGHT) 
			atten[0]=cal_max_atten;

}

/* ---------------------------------------------------------------------
   click_prepare_output
   -------------------------------------------------------------------- */
static void click_prepare_output(SynData *flags, xword *outbuf, int outsize, 
					 mxArray *plhs[])
{
	double *output;
	int i,j;

	plhs[0]=mxCreateDoubleMatrix(outsize,globals.da_nchans,mxREAL);
	output=mxGetPr(plhs[0]);
	for (i=0; i<outsize; i++) {
		for (j=0; j<globals.da_nchans; j++) 
			output[j*outsize+i]= (double) outbuf[globals.da_nchans*i+j];
	}
	mxFree(outbuf);

}
/* ---------------------------------------------------------------------
   synth_prepare_flags
   -------------------------------------------------------------------- */
static void synth_prepare_flags(SynData *flags, const mxArray *prhs[])
{
	init_syn_data(flags);
	parse_syn_data(flags, prhs);
	check_syn_data(flags, "synthesize");
	compute_stack_db(flags);

	syn_spec_parse(flags->spec, 0, &(flags->ss));
	if (flags->ss.class == SC_INVALID) 
		mexErrMsgTxt("synthesize: invalid stimspec!\n");

}

/* ---------------------------------------------------------------------
   click_prepare_flags
   -------------------------------------------------------------------- */
static void click_prepare_flags(SynData *flags, const mxArray *prhs[])
{
	init_syn_data(flags);
	parse_click_data(flags, 1, prhs);
	check_click_data(flags, "synthesize");

	syn_spec_parse(flags->spec, 0, &(flags->ss));
	if (flags->ss.class == SC_INVALID) 
		mexErrMsgTxt("synthesize: invalid stimspec!\n");

}

/* ---------------------------------------------------------------------
   synth_setup_calib
   -------------------------------------------------------------------- */
static void synth_setup_calib(void)
{
	char *foo;

	syn_calibrate = 1;

	if (!globals.cal_loaded) {
		cal_FakeFlatCal(&foo);
		globals.cal_loaded = 1;
		mexPrintf("synthesize: no calib data loaded; using flat calib\n");
	}
}


/* ---------------------------------------------------------------------
   twosound:
   -------------------------------------------------------------------- */
static void twosound(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) 
{
	float del_db_iid, del_db_abi;
	SynData	flags1, flags2;
	syn_spec ss_two;
	xword *outbuf;
	int outsize;

	if (nrhs != 21)
		twosound_usage();
	if (nlhs > 2)
		twosound_usage();

	update_local_from_globals();

	twosound_prepare_flags(&flags1, &flags2, prhs);
	if (flags1.side != SYN_BOTH) 
		mexErrMsgTxt("ERROR: side1 must be 'both'. other values not currently support\n");
	synth_setup_calib();

	compute_iid_abi(&flags1, &flags2, &del_db_iid, &del_db_abi);

	outsize = (int) (((double) globals.epoch/1000) * (double) globals.daFc);
	mxAssert(((outbuf = (xword *) mxCalloc(outsize*globals.da_nchans, 
		sizeof(xword)))!=NULL), "synthesize: mxCalloc returns NULL!");

	two_sound_synthesize(&ss_two, NULL, outbuf, outsize, globals.da_nchans, 
		globals.twosound_rms,
		globals.epoch, del_db_iid, del_db_abi, 
		&flags1.ss,  flags1.delay, flags1.dur, flags1.itd, flags1.bc,
		flags1.rise, flags1.fall, flags1.side, 
		&flags2.ss, flags2.delay, flags2.dur, flags2.itd, flags2.bc,
		flags2.rise, flags2.fall, flags2.side);

	if (!cal_SafeFigureAtten(&ss_two, flags1.ldb, flags1.rdb, 
		  &flags1.latten, &flags1.ratten)) {
	  /*
		if (!flags1.ignore) 
			mexErrMsgTxt("twosound: error calibrating waveforms!\n");
	  */
	}

	synth_prepare_output(&flags1, outbuf, outsize, plhs);

	export_globals(); }

/* ------------------------------------------------------------------------
	twosound_prepare_flags
   ------------------------------------------------------------------------ */
static void twosound_prepare_flags(SynData *flags1, SynData *flags2, const mxArray *prhs[])
{
	init_syn_data(flags1);
	init_syn_data(flags2);
	parse_twosound_data(flags1, flags2, prhs);
	check_syn_data(flags1, "twosound, stim #1");
	check_syn_data(flags2, "twosound, stim #2");
	compute_stack_db(flags1);
	compute_stack_db(flags2);

	syn_spec_parse(flags1->spec, 0, &(flags1->ss));
	if (flags1->ss.class == SC_INVALID) 
		mexErrMsgTxt("twosound: invalid stimspec for sound #1!\n");
	syn_spec_parse(flags2->spec, 0, &(flags2->ss));
	if (flags2->ss.class == SC_INVALID) 
		mexErrMsgTxt("twosound: invalid stimspec for sound #2!\n");


}

/* ------------------------------------------------------------------------
	compute_iid_abi
   ------------------------------------------------------------------------ */
static void compute_iid_abi(SynData *flags1, SynData *flags2, 
							float *del_db_iid, float* del_db_abi)
{
  float iid1, iid2, abi1, abi2;

	if (flags1->side == SYN_LEFT)
    flags1->rdb = flags1->ldb;
	if (flags1->side == SYN_RIGHT)
    flags1->ldb = flags1->rdb;
	if (flags2->side == SYN_LEFT)
    flags2->rdb = flags2->ldb;
	if (flags2->side == SYN_RIGHT)
    flags2->ldb = flags2->rdb;

  iid1 = (float)(flags1->rdb - flags1->ldb);
  iid2 = (float)(flags2->rdb - flags2->ldb);
  abi1 = ((float)(flags1->rdb + flags1->ldb))/2.0;
  abi2 = ((float)(flags2->rdb + flags2->ldb))/2.0;

  *del_db_iid = iid2 - iid1;
  *del_db_abi = abi2 - abi1;

/*
	if (flags1->side == SYN_BOTH) {
		*del_db_iid = (float) ((flags2->rdb - flags2->ldb) - (flags1->rdb - 
			flags1->ldb));
		*del_db_abi = ((float) flags2->rdb + (float) flags2->ldb)/2.0 - 
			((float) flags1->rdb + (float) flags1->ldb)/2.0;
	} else if (flags1->side == SYN_LEFT) {
		*del_db_iid = 0.0;
		*del_db_abi = (float) (flags2->ldb-flags1->ldb);
	} else if (flags1->side == SYN_RIGHT) {
		*del_db_iid = 0.0;
		*del_db_abi = (float) (flags2->rdb-flags1->rdb);
	}
*/
}

/* ------------------------------------------------------------------------
	compute_stack_db
   ------------------------------------------------------------------------ */
static void compute_stack_db(SynData *flags)
{
	char buf[256];
	char *tok;
	int i,nfreqs=1;
	int	*freq=NULL;
	int *db=NULL;
	int	adj=0;


	if (strncmp(flags->spec,"stack",5)!=0) return;

	if ((flags->side == SYN_LEFT) || (flags->side == SYN_RIGHT)) {

		if (index(flags->spec,'(')!=NULL) {
			/* first, count the number of frequencies */
			for (i=0;i<strlen(flags->spec);i++) 
				if (flags->spec[i] == ',') nfreqs++;

			mxAssert(((freq=(int *)mxCalloc(nfreqs,sizeof(int)))!=NULL), 
				"compute_stack_db: mxCalloc returns NULL!");
			mxAssert(((db=(int *)mxCalloc(nfreqs,sizeof(int)))!=NULL),
				"compute_stack_db: mxCalloc returns NULL!");

			/* extract the freq's and db's from the stimspec */
			if (flags->side==SYN_LEFT) 
				adj=-2;
			else 
				adj=2;

			strcpy(buf, index(flags->spec,'=')+1);
			tok=strtok(buf,",");
			for (i=0;i<nfreqs;i++) {
				if (tok!=NULL) {
					sscanf(tok,"%d(%d)",&(freq[i]),&(db[i]));
					db[i]=adj*db[i];
				}
				tok=strtok(NULL,",");
			}
			/* now reassemble the stimspec */
			sprintf(buf,"stack=%d(%d)",freq[0],db[0]);
			if (nfreqs>1) {
				for (i=1;i<nfreqs;i++) 
					sprintf(buf,"%s,%d(%d)",buf,freq[i],db[i]);
			}
			strcpy(flags->spec,buf);

			mxFree(freq);
			mxFree(db);
		}
	}
}

/* ---------------------------------------------------------------------
   click_usage
   -------------------------------------------------------------------- */
static void click_usage(void)
{
	mexPrintf("usage: [data att] = synth('click',<spec>,<delay>,<itd>,<side>)\n");
	mexPrintf("      <spec>        syn spec (see below)\n");
	mexPrintf("      <delay> (ms)  delay before onset of stimulus\n");
	mexPrintf("      <itd> (us)    itd b/t left and right channel\n");
	mexPrintf("      <side>        'left', 'right' or 'both'\n\n");

	mexPrintf("synth_click sound specs (for <spec>):\n\n");
	mexPrintf("       rare_click=<nsamples>...... rarefaction click\n");
	mexErrMsgTxt("       cond_click=<nsamples ...... condensation click)\n");
}

/* ---------------------------------------------------------------------
   synthesize_usage
   -------------------------------------------------------------------- */
static void synthesize_usage(void)
{
	mexPrintf("usage: [data att] = synth('synthesize',<spec>,<dur>,<delay>,<itd>,<rise>,<fall>,<ldb>,<rdb>,<side>)\n");
	mexPrintf("      <spec>        syn spec (see below)\n");
	mexPrintf("      <dur> (ms)    duration of stimulus\n");
	mexPrintf("      <delay> (ms)  delay before onset of stimulus\n");
	mexPrintf("      <itd> (us)    itd b/t left and right channel\n");
	mexPrintf("      <rise> (ms)   rise time of stimulus envelope\n");
	mexPrintf("      <fall> (ms)   fall time of stimulus envelope\n");
	mexPrintf("      <ldb> (db)    left absolute intensity\n");
	mexPrintf("      <rdb> (dB)    right absolute intensity\n");
	mexPrintf("      <side>        'left', 'right' or 'both'\n\n");

	mexPrintf("synthesize sound specs (for <spec>):\n\n");
	mexPrintf("       bb................. Broadband Noise\n");
	mexPrintf("       hp................. high pass noise (4-13k)\n");
	mexPrintf("       tone=NNN .......... pure tone of NNN hz\n");
	mexPrintf("       bp=lo-hi .......... band pass from f1-f2\n");
	mexPrintf("       CF=<cf>,BW=<w> .... bandpass from cf-(bw/2) to cf+(bw/2)\n");
	mexPrintf("       stack=<range>...... pure tone list (tone stack)\n");
	mexErrMsgTxt("       fmsweep=from-to:n.. FM Sweep (n == 1, if not specified..)\n");
}

/* ---------------------------------------------------------------------
   twosound_usage
   -------------------------------------------------------------------- */
static void twosound_usage(void)
{
	mexPrintf("usage: [data att] = synth('twosound',<spec1>,<dur1>,<delay1>,<itd1>,<rise1>,<fall1>,<ldb1>,<rdb1>,<side1>,\n");
	mexPrintf("                          <spec2>,<dur2>,<delay2>,<itd2>,<rise2>,<fall2>,<ldb2>,<rdb2>,<side2>)\n");
	mexPrintf("      <spec<N>>        syn spec (see below)\n");
	mexPrintf("      <dur<N>> (ms)    duration of stimulus\n");
	mexPrintf("      <delay<N>> (ms)  delay before onset of stimulus\n");
	mexPrintf("      <itd<N>> (us)    itd b/t left and right channel\n");
	mexPrintf("      <rise<N>> (ms)   rise time of stimulus envelope\n");
	mexPrintf("      <fall<N>> (ms)   fall time of stimulus envelope\n");
	mexPrintf("      <ldb<N>> (db)    left absolute intensity\n");
	mexPrintf("      <rdb<N>> (dB)    right absolute intensity\n");
	mexPrintf("      <side<N>>           'left', 'right' or 'both'\n\n");

	mexPrintf("synthesize sound specs (for <spec>):\n\n");
	mexPrintf("       bb................. Broadband Noise\n");
	mexPrintf("       hp................. high pass noise (4-13k)\n");
	mexPrintf("       tone=NNN .......... pure tone of NNN hz\n");
	mexPrintf("       bp=lo-hi .......... band pass from f1-f2\n");
	mexPrintf("       CF=<cf>,BW=<w> .... bandpass from cf-(bw/2) to cf+(bw/2)\n");
	mexPrintf("       stack=<range>...... pure tone list (tone stack)\n");
	mexErrMsgTxt("       fmsweep=from-to:n.. FM Sweep (n == 1, if not specified..)\n");
}

/* ---------------------------------------------------------------------
   init_syn_data
   -------------------------------------------------------------------- */
static void init_syn_data(SynData *flags)
{
	flags->ldb = 0;     /* dBspl */
	flags->latten = 0;
	flags->rdb = 0;     /* dBspl */
	flags->ratten = 0;
	flags->dur=100;		 /* ms */
	flags->rise=10;      /* ms */
	flags->fall=10;      /* ms */
	flags->delay=0;      /* ms */
	flags->itd=0;        /* us */
	flags->bc=100;        /* us */
	flags->side=SYN_BOTH;
	strcpy(flags->spec, "BB");
	flags->ignore=0;
}

/* ---------------------------------------------------------------------
   parse_syn_data
   -------------------------------------------------------------------- */
static int parse_syn_data(SynData *flags, const mxArray *args[])
{
	parse_sound_data(flags, 1, args);
/*
	parse_side(flags, 10, args);
*/
}

/* ---------------------------------------------------------------------
   parse_twosound_data
   -------------------------------------------------------------------- */
static int parse_twosound_data(SynData *flags1, SynData *flags2, 
	const mxArray *args[])
{
	parse_sound_data(flags1, 1, args);
	parse_sound_data(flags2, 11 /*9*/, args);
/*
	parse_side(flags1, 19, args);
	flags2->side=flags1->side;
*/
}

/* ---------------------------------------------------------------------
   parse_click_data
   -------------------------------------------------------------------- */
static int parse_click_data(SynData *flags, int offset, const mxArray *args[])
{
	char buf[255];
	char first;

	/* syn spec */
	if (mxIsChar(args[offset])) {
		if (mxGetString(args[offset], flags->spec, 255)) 
			mexPrintf("synth: <spec> longer than 255 chars: truncated\n");
	} else 
		mexErrMsgTxt("synth: <spec> must be a string");

	/* delay */
	if (mxIsDouble(args[offset+1]))
		flags->delay = mxGetScalar(args[offset+1]);
	else
		mexErrMsgTxt("synth: <delay> must be a scalar");

	/* itd */
	if (mxIsDouble(args[offset+2]))
		flags->itd = mxGetScalar(args[offset+2]);
	else
		mexErrMsgTxt("synth: <itd> must be a scalar");

	/* side */
	if (mxIsChar(args[offset+3])) {
		mxGetString(args[offset+3], buf, 255);
		first = toupper(buf[0]);
		switch (first) {
			case 'L':
				flags->side = SYN_LEFT;
				break;
			case 'R':
				flags->side = SYN_RIGHT;
				break;
			case 'B':
				flags->side = SYN_BOTH;
				break;
			default:
				mexErrMsgTxt("synth: use 'left', 'right' or 'both' for <side>\n"); 
		}
	} else
		mexErrMsgTxt("synth: <side> must be a string");
}


/* ---------------------------------------------------------------------
   parse_sound_data
   -------------------------------------------------------------------- */
static int parse_sound_data(SynData *flags, int offset, const mxArray *args[])
{
	char buf[255];
	char first;

	/* syn spec */
	if (mxIsChar(args[offset])) {
		if (mxGetString(args[offset], flags->spec, 255)) 
			mexPrintf("synth: <spec> longer than 255 chars: truncated\n");
	} else 
		mexErrMsgTxt("synth: <spec> must be a string");

	/* duration */
	if (mxIsDouble(args[offset+1]))
		flags->dur = mxGetScalar(args[offset+1]);
	else
		mexErrMsgTxt("synth: <dur> must be a scalar");


	/* delay */
	if (mxIsDouble(args[offset+2]))
		flags->delay = mxGetScalar(args[offset+2]);
	else
		mexErrMsgTxt("synth: <delay> must be a scalar");

	/* itd */
	if (mxIsDouble(args[offset+3]))
		flags->itd = mxGetScalar(args[offset+3]);
	else
		mexErrMsgTxt("synth: <itd> must be a scalar");

	/* bc */
	if (mxIsDouble(args[offset+4]))
		flags->bc = mxGetScalar(args[offset+4]);
	else
		mexErrMsgTxt("synth: <itd> must be a scalar");

	/* rise */
	if (mxIsDouble(args[offset+5]))
		flags->rise = mxGetScalar(args[offset+5]);
	else
		mexErrMsgTxt("synth: <rise> must be a scalar");

	/* fall */
	if (mxIsDouble(args[offset+6]))
		flags->fall = mxGetScalar(args[offset+6]);
	else
		mexErrMsgTxt("synth: <fall> must be a scalar");

	/* ldb */
	if (mxIsDouble(args[offset+7]))
		flags->ldb = mxGetScalar(args[offset+7]);
	else
		mexErrMsgTxt("synth: <ldb> must be a scalar");

	/* rdb */
	if (mxIsDouble(args[offset+8]))
		flags->rdb = mxGetScalar(args[offset+8]);
	else
		mexErrMsgTxt("synth: <rdb> must be a scalar");

	/* side */
	if (mxIsChar(args[offset+9])) {
		mxGetString(args[offset+9], buf, 255);
		first = toupper(buf[0]);
		switch (first) {
			case 'L':
				flags->side = SYN_LEFT;
				break;
			case 'R':
				flags->side = SYN_RIGHT;
				break;
			case 'B':
				flags->side = SYN_BOTH;
				break;
			default:
				mexErrMsgTxt("synth: use 'left', 'right' or 'both' for <side>\n"); 
		}
	} else
		mexErrMsgTxt("synth: <side> must be a string");
}

#if(0)
/* ---------------------------------------------------------------------
   parse_side
   -------------------------------------------------------------------- */
static int parse_side(SynData *flags, int i, const mxArray *args[])
{
	char buf[255];
	char first;

	if (mxIsChar(args[i])) {
		mxGetString(args[i], buf, 255);
		first = toupper(buf[0]);
		switch (first) {
			case 'L':
				flags->side = SYN_LEFT;
				break;
			case 'R':
				flags->side = SYN_RIGHT;
				break;
			case 'B':
				flags->side = SYN_BOTH;
				break;
			default:
				mexErrMsgTxt("synth: use 'left', 'right' or 'both' for <side>\n"); 
		}
	} else
		mexErrMsgTxt("synth: <side> must be a string");
}
#endif

/* ---------------------------------------------------------------------
   check_syn_data
   -------------------------------------------------------------------- */
static int check_syn_data(SynData *flags, char *id)
{
	char buf[80];

	if (flags->dur > globals.epoch) {
		sprintf(buf, "%s: stim duration (%d) > epoch (%d)",
			id, flags->dur, globals.epoch);
		mexErrMsgTxt(buf);
	}

	if ((flags->dur+flags->delay) > globals.epoch) {
		sprintf(buf, "%s: stim delay (%d) + dur (%d) > epoch (%d)",
			id, flags->delay, flags->dur, globals.epoch);
		mexErrMsgTxt(buf);
	}
}

/* ---------------------------------------------------------------------
   check_click_data
   -------------------------------------------------------------------- */
static int check_click_data(SynData *flags, char *id)
{
	char buf[80];

	if (flags->delay > globals.epoch) {
		sprintf(buf, "%s: stim delay (%d) > epoch (%d)",
			id, flags->dur, globals.epoch);
		mexErrMsgTxt(buf);
	}
}


/* ------------------------------------------------------------------------
   export_globals
   ------------------------------------------------------------------------ */
static void export_globals(void)
{
	mxArray *matglobals = NULL;
	char *field_names[] = {"debug", "bb_low", "bb_high", "cal_loaded", 
						   "daFc", "epoch", "attMaxAtten", "rad_vary", 
						   "twosound_rms", "stack_rms", "use_cache"};
	int nfields = 11;
	int dims[2] = {1,1};
	int ndims = 2;

	matglobals = mxCreateStructArray(ndims, dims, nfields, 
			(const char **)field_names);
	mxSetName(matglobals, "syn_info");

	mxAssert((matglobals != NULL), "synth: unable to get or allocate syn_info!");

	set_field_double(matglobals, 0, "bb_low", globals.bb_low);
	set_field_double(matglobals, 0, "bb_high", globals.bb_high);
	set_field_int(matglobals, 0, "daFc", globals.daFc);
	set_field_int(matglobals, 0, "stack_rms", globals.stack_rms);
	set_field_int(matglobals, 0, "epoch", globals.epoch);
	set_field_int(matglobals, 0, "rad_vary", globals.rad_vary);
	set_field_int(matglobals, 0, "cal_loaded", globals.cal_loaded);
	set_field_double(matglobals, 0, "attMaxAtten", cal_max_atten);
	set_field_int(matglobals, 0, "debug", globals.debugflag);
	set_field_int(matglobals, 0, "twosound_rms", globals.twosound_rms);
	set_field_int(matglobals, 0, "use_cache", syn_use_cache);

	mexPutArray(matglobals, "global");
}

/* ------------------------------------------------------------------------
   set_field_double
   ------------------------------------------------------------------------ */
static void set_field_double(mxArray *data, int index, char *name, double val)
{
	double *new = NULL;
	mxArray *mat = NULL;
	mxArray *old = NULL;

	mat = mxCreateDoubleMatrix(1,1,mxREAL);
	new = mxGetPr(mat);
	*new = val;
	old = mxGetField(data, 0, name);
	if (old != NULL)
		mxDestroyArray(old);
	mxSetField(data, 0, name, mat);
}

/* ------------------------------------------------------------------------
   set_field_string
   ------------------------------------------------------------------------ */
static void set_field_string(mxArray *data, int index, char *name, char *val)
{
	mxArray *mat = NULL;
	mxArray *old = NULL;
	char test[256];

	mat = mxCreateString(val);
	mxGetString(mat, test, 256);
	old = mxGetField(data, 0, name);
	if (old != NULL)
		mxDestroyArray(old);
	mxSetField(data, 0, name, mat);
}

/* ------------------------------------------------------------------------
   set_field_int
   ------------------------------------------------------------------------ */
static void set_field_int(mxArray *data, int index, char *name, int val) {
	double *new = NULL;
	mxArray *mat = NULL;
	mxArray *old = NULL;

	mat = mxCreateDoubleMatrix(1,1,mxREAL);
	new = mxGetPr(mat);
	*new = (double) val;
	old = mxGetField(data, 0, name);
	if (old != NULL)
		mxDestroyArray(old);
	mxSetField(data, 0, name, mat);
}

/* ------------------------------------------------------------------------
   update_local_from_globals
   ------------------------------------------------------------------------ */
static void update_local_from_globals(void)
{
	mxArray *matglobals = NULL;
	double tmp;

	if ((matglobals = mexGetArray("syn_info", "global")) == NULL) {
		export_globals();
		return;
	}

	mxAssert((matglobals != NULL), 
		"synth: unable to get or allocate syn_info!");

	get_field_int(matglobals, 0, "debug", &globals.debugflag);
	get_field_double(matglobals, 0, "bb_low", &globals.bb_low);
	get_field_double(matglobals, 0, "bb_high", &globals.bb_high);
	get_field_int(matglobals, 0, "daFc", &globals.daFc);
	get_field_int(matglobals, 0, "stack_rms", &globals.stack_rms);
	get_field_int(matglobals, 0, "epoch", &globals.epoch);
	get_field_double(matglobals, 0, "attMaxAtten", (double *)&tmp);
	cal_max_atten=(float) tmp;
	get_field_int(matglobals, 0, "twosound_rms", &globals.twosound_rms);
	get_field_int(matglobals, 0, "rad_vary", &globals.rad_vary);
	get_field_int(matglobals, 0, "use_cache", (int *)&syn_use_cache);
}

/* ------------------------------------------------------------------------
   get_field_int
   ------------------------------------------------------------------------ */
static int get_field_int(mxArray *data, int index, char *name, int *val)
{
	mxArray *old = NULL;
	double *tmp;

	old = mxGetField(data, 0, name);
	if (old) {
		tmp =  mxGetPr(old);
		*val=*tmp;
		return(1);
	}

	return(0);
}

/* ------------------------------------------------------------------------
   get_field_double
   ------------------------------------------------------------------------ */
static int get_field_double(mxArray *data, int index, char *name, 
							   double *val)
{
	mxArray *old = NULL;
	double *tmp;

	old = mxGetField(data, 0, name);
	if (old) {
		tmp = mxGetPr(old);
		*val=*tmp;
		return(1);
	}
	return(0);
}

/* ------------------------------------------------------------------------
   get_field_string: remember to free the string when you're done
   ------------------------------------------------------------------------ */
static int get_field_string(mxArray *data, int index, char *name, char **val)
{
	mxArray *old = NULL;

	old = mxGetField(data, 0, name);
	if (old) {
		mxAssert(((*val = (char *) mxCalloc(255, sizeof(char))) != NULL),
			"get_field_string: mxCalloc returns NULL!");
		mxGetString(old, *val, 255);
		return(1);
	}
	return(0);
}

static int synmat_daFc(void) {
	return(globals.daFc);
}

static int synmat_stack_rms(void) {
	return(globals.stack_rms);
}

static float synmat_bb_low_freq(void) {
	return(globals.bb_low);
}

static float synmat_bb_high_freq(void) {
	return(globals.bb_high);
}

static double synmat_get_rad(void) {
	return(0);
}

static int synmat_tone_rad_vary(void) {
	return(globals.rad_vary);
}
