/******************************************************************
**  RCSID: $Id: psdriver.c,v 1.3 1999/05/14 21:41:58 cmalek Exp $
** Program: libpsdriver.a
**  Module: psdriver.c
**  Author: wilson -> mazer
** Descrip: x11->postscript driver
**
**   >>> NOTE: Original copyright notice at end of file <<<
**
** Revision History (most recent last)
**  Sat Feb 11 ??:??:?? 1992 mazer
**   ps output is now EPS compatible
**
** Sun May  3 16:51:33 1992 mazer
**  added partial support for writing XFIG 2.1 files
**
** Wed Dec  9 15:59:52 1992 mazer
**  Made things use switches and incorported Segment stuff
**  from Mike Lewicki's version of psdriver
**
** Tue Nov 23 13:22:51 1993 mazer
**  changed XPS_Set_Filemode(NULL) to return the current mode string
**
** Fri Dec 10 13:25:37 1993 mazer
**  cleaned some stuff up -- made truly private functions static
**  and changed all the Prepare/Finish stuff to XPS_xxxFIG() or
**  XPS_xxxPS() and deleted At_NewpagePS()
**
** Tue Feb 14 14:30:01 1995 mazer
**  patched ala Upi/Bilitch for R6 support..
**
** 1.97 bjarthur
**  changed naming convention back to original XPS*()
**  took off XPS prefix on static functions
**  replaced library version with mazer's that has R6 patch
**  made ANSI compatible
**  added GetContextFont() to complete R6 support
**  fixed XPS_DrawString() to terminate at right length
**  fixed PSSetPixel() to actually use pixel, and not just inverse
**    (it now does fills with background color correctly)
**
** 10.98 bjarthur
**  added XPS_PutImage(), XPS_DrawImageStringVertical()
**  changed PSSetPixel() to PSSetColor() ... now color output is supported
**  added support for Helvetica, Times, and Courier fonts
**		(Bold and/or/neither Italic)
**  scaled width of PS strings to match that of X strings
**
*******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <math.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <strings.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>

#include "psdriver.h"

#define DPI		300
#define PAGEHEIGHT	11.0
#define PAGEWIDTH	8.5
#define PTS_PER_INCH	72				/* ps default units */
#define MAXCHUNK 	10000					/* maximum number of linesegs at once */

typedef enum {
	Xmode, PSmode, FIGmode,
} OutMode;

#if 0
static unsigned long GetContextForeground(Display *, GC);
static int GetContextLineWidth(Display *, GC);
static int GetContextLineStyle(Display *, GC);
static unsigned long GetContextFont(Display *, GC);
#endif
static void prep_clip(Display *, Drawable, int *, int *);
static int try_clip_point(int, int, int *, int *);
static int try_clip_line(int, int, int *, int *, int *, int *);
static void PSSetColor(Display*, Drawable, GC);
static int GetLineWidth(Display*,GC);
static void PSSetLineWidth(Display*,GC);
static int GetLineStyle(Display*,GC);
static void PSSetLineStyle(Display*,GC);
static void PSClosefill(void);
static void PSNewpath(void);
static void PSMoveto(int, int);
static void PSLineto(int, int);
static void PSPointat(int, int);
static void PSStroke(void);
static void PSShow(char *);
static void PSHeader(XWindowAttributes *, float, int, char *);
static void PSTrailer(void);
static void FIGHeader(XWindowAttributes *);
static void FIGTrailer(void);

static OutMode currentOutMode = Xmode;
static int global_window_height;
static int global_window_width;
static float pix_scale;
static float page_scale;
static int current_width = -1;
static int current_style = -1;
static int current_color = -1;
static int color = 1;
static int page_count = 0;
static int center_output = 1;		/* 25-Mar-93: this was 1 */
static short file_output = 0;		/* write to file or stdout? */
static char *filename_out;			/* name of file to write to */
static char filemode_out[5] = "a";	/* output mode: "w" or "a" */
static FILE *outfp = NULL;			/* current output file handle */
static char xfig_font_num = 16;	/* Helvetica */
static int figxoff = 10;
static int figyoff = 10;

static Drawable G_drawable;

#define NO_CLIP		 0
#define CLIPPED		 1
#define ELIMINATE	-1

static void prep_clip(Display * dpy, Drawable drawable, int *w, int *h)
{
	XWindowAttributes info;

	XGetWindowAttributes(dpy, drawable, &info);
	*w = info.width;
	*h = info.height;
}

static int try_clip_point(int w, int h, int *x, int *y)
{

	if (*x < 0 || *x > w) {
		return (ELIMINATE);
	}

	if (*y < 0 || *y > h) {
		return (ELIMINATE);
	}
	return (NO_CLIP);
}

static int try_clip_line(int w, int h, int *x1, int *y1, int *x2, int *y2)
{
	if (*x1 < 0 || *x1 > w || *x2 < 0 || *x2 > w) {
		return (ELIMINATE);
	}

	if (*y1 < 0 || *y1 > h || *y2 < 0 || *y2 > h) {
		return (ELIMINATE);
	}
	return (NO_CLIP);
}

void XPS_SetFilename(char *name)
{
	filename_out = name;
}

char *XPS_SetFilemode(char *newmode)
{
	if (newmode == NULL)
		return (filemode_out);
	else
		strncpy(filemode_out, newmode, sizeof(filemode_out));
	return (filemode_out);
}

void XPS_SetFileOutput(int state)
{
	file_output = state;
}


static void PSSetColor(Display *dpy, Drawable drawable, GC context)
{
	XWindowAttributes retval;
	XGCValues gcval;
	XColor xcolor;
	long tmp;

	if(color) {
		XGetWindowAttributes(dpy, drawable, &retval);
		XGetGCValues(dpy, context, GCForeground, &gcval);
		xcolor.flags = DoRed & DoGreen & DoBlue;
		xcolor.pixel = gcval.foreground;
		XQueryColor(dpy, retval.colormap, &xcolor);
		if((tmp = xcolor.red/256 + xcolor.green + xcolor.blue*256)
					!= current_color) {
			current_color = tmp;
			fprintf(outfp,"%f %f %f setrgbcolor\n",
						xcolor.red/65535.0, xcolor.green/65535.0, xcolor.blue/65535.0); } }
	else
		fprintf(outfp,"0.0 setgray\n");
}

static int GetLineWidth(Display *dpy, GC gc)
{
	XGCValues val;

	XGetGCValues(dpy, gc, GCLineWidth, &val);
	return(val.line_width);
}

static void PSSetLineWidth(Display *dpy, GC gc)
{
	int width = GetLineWidth(dpy,gc);

	if (width == 0)
		width = 1;
	if (current_width != width) {
		current_width = width;
		fprintf(outfp, "%f setlinewidth\n", current_width / pix_scale);
	}
}

static int GetLineStyle(Display *dpy, GC gc)
{
	XGCValues val;

	XGetGCValues(dpy, gc, GCLineStyle, &val);
	return(val.line_style);
}

static void PSSetLineStyle(Display *dpy, GC gc)
{
	XGCValues val;
	int style = GetLineStyle(dpy,gc);

	if (current_style != style) {
		current_style = style;
		switch (current_style) {
			case LineSolid:
				fprintf(outfp, "[] 0 setdash\n");
				break;
			case LineOnOffDash:
				XGetGCValues(dpy, gc, GCDashList, &val);
				if(val.dashes != 0)
					fprintf(outfp, "[%d %d] 0 setdash\n",val.dashes,val.dashes);
				else
					fprintf(outfp, "[1 2] 0 setdash\n");
				break;
		}
	}
}

static void PSArc(float x, float y, float w, float h, float a1, float a2)
{
	if (w != h)
		fprintf(stderr, "warning: ellipses not supported\n");
	fprintf(outfp, "%f %f %f %f %f arc ",
				x + w/2.0, global_window_height - y - w/2.0,
				w/2.0, a1/64.0, a2/64.0);
}

static void PSClosefill(void)
{
	fprintf(outfp, "closepath fill ");
}

static void PSNewpath(void)
{
	fprintf(outfp, "newpath\n");
}

static void PSMoveto(int x, int y)
{
	fprintf(outfp, "%d %d moveto\n", x, global_window_height - y);
}

static void PSLineto(int x, int y)
{
	fprintf(outfp, "%d %d lineto ", x, global_window_height - y);
}

static void PSPointat(int x, int y)
{
	fprintf(outfp, "%d %d 0.25 0 360 arc fill ", x, global_window_height - y);
}

static void PSStroke(void)
{
	fprintf(outfp, "stroke\n");
}

static void PSShow(char *s)
{
	fprintf(outfp, "(%s) show\n", s);
}

static void PSSelectfont(char *s, int p)
{
	fprintf(outfp,"/%s %d selectfont\n",s,p);
}

void XPS_PutImage(Display * dpy, Drawable drawable, GC context, XImage * image,
			int src_x, int src_y, int dest_x, int dest_y,
			unsigned int width, unsigned int height)
{
	int i, j;
	XWindowAttributes retval;
	XColor xcolor;

	switch (currentOutMode) {
		case Xmode:
			XPutImage(dpy, drawable, context, image,
						src_x, src_y, dest_x, dest_y, width, height);
			break;
		case PSmode:
			XGetWindowAttributes(dpy, drawable, &retval);
			xcolor.flags = DoRed & DoGreen & DoBlue;

			assert(8 % image->depth == 0);	/* can't handle depth==12 */
			fprintf(outfp, "gsave\n");
			fprintf(outfp, "/picstr %d string def\n", 3 * image->width);
			fprintf(outfp, "%d %d translate\n", dest_x, global_window_height - dest_y);
			fprintf(outfp, "%d %d scale\n", width, height);
			fprintf(outfp, "%d %d 8\n", image->width, image->height);
			fprintf(outfp, "[%d 0 0 %d 0 0]\n", image->width, image->height);
			fprintf(outfp, "{currentfile picstr readhexstring pop}\n");
			fprintf(outfp, "false 3\n");
			fprintf(outfp, "colorimage\n");
			for (j = image->height - 1; j >= 0; j--) {
				for (i = 0; i < image->width; i++) {
					xcolor.pixel = XGetPixel(image, i + src_x, j + src_y);
					XQueryColor(dpy, retval.colormap, &xcolor);
					fprintf(outfp, "%02x%02x%02x",
								xcolor.red / 256, xcolor.green / 256, xcolor.blue / 256);
				}
				fprintf(outfp, "\n");
			}
			fprintf(outfp, "grestore\n");
			break;
		case FIGmode:
			break;
	}
}

static void get_ps_font(Display *dpy, GC context, char **font, int *size)
{
	XFontStruct *old;
	XGCValues gcval;
  unsigned long family, weight, slant, point;
	char ret_weight[128], ret_slant[128];
	static char ret_name[128];
	
	XGetGCValues(dpy, context, GCFont, &gcval);
	old = XQueryFont(dpy, gcval.font);

  XGetFontProperty(old, XInternAtom(dpy, "FAMILY_NAME", 0), &family);
  sprintf(ret_name, "%s", XGetAtomName(dpy, (Atom) family));
  XGetFontProperty(old, XInternAtom(dpy, "WEIGHT_NAME", 0), &weight);
  sprintf(ret_weight, "%s", XGetAtomName(dpy, (Atom) weight));
  XGetFontProperty(old, XInternAtom(dpy, "SLANT", 0), &slant);
  sprintf(ret_slant, "%s", XGetAtomName(dpy, (Atom) slant));
  XGetFontProperty(old, XInternAtom(dpy, "POINT_SIZE", 0), &point);
	*size = point/10;

	if(!strcasecmp(ret_weight,"bold"))
		strcat(ret_name,"-Bold");

	if(!strcasecmp(ret_slant,"o")) {
		if(!strcasecmp(ret_weight,"medium"))
			strcat(ret_name,"-");
		if(!strcasecmp(ret_name,"times"))
			strcat(ret_name,"Italic");
		else
			strcat(ret_name,"Oblique"); }
		
	if(!strcasecmp(ret_name,"times") && !strcasecmp(ret_weight,"medium") &&
				!strcasecmp(ret_slant,"r"))
		strcat(ret_name,"-Roman");

	*font = ret_name;
}

void XPS_DrawImageStringVertical(Display *dpy, Drawable drawable, GC context,
			int x, int y, char *s, int length)
{
	Pixmap pixmap;
	int h, w;
	char *tmp,tmp2[64];
	OutMode tmpOutMode;
	XFontStruct *fontstruct;
	XWindowAttributes wa;
	XGCValues gcval;
	char *font;
	int size;
	int i,j;
  XImage *image, *rotated_image;
  char *data;
	unsigned long tmpul;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
    	XGetGCValues(dpy, context, GCForeground | GCBackground | GCFont, &gcval);
			fontstruct = XQueryFont(dpy, gcval.font);
      w = XTextWidth(fontstruct,s,length);
      h = fontstruct->ascent + fontstruct->descent;
      XGetWindowAttributes(dpy, drawable, &wa);
      pixmap = XCreatePixmap(dpy, drawable, w, h, wa.depth);
    	tmpul = gcval.foreground;
      XSetForeground(dpy, context, gcval.background);
      XFillRectangle(dpy, pixmap, context, 0, 0, w, h);
      XSetForeground(dpy, context, tmpul);
      XDrawString(dpy, pixmap, context, 0, fontstruct->ascent, s, length);
      image = XGetImage(dpy, pixmap, 0, 0, w, h, AllPlanes, XYPixmap);
      data = (char *) malloc(wa.depth * w * (h + 7) / 8);
      rotated_image = XCreateImage(dpy, wa.visual, wa.depth, XYPixmap, 0, data,
            h, w, 8, (h + 7) / 8);
      for (i = 0; i < w; i++)
        for (j = 0; j < h; j++)
          XPutPixel(rotated_image, j, w - i - 1, XGetPixel(image, i, j));
    	XPutImage(dpy, drawable, context, rotated_image, 0, 0, x, y-w, h, w);
      XDestroyImage(image); XDestroyImage(rotated_image);
      XFreePixmap(dpy, pixmap);
      XFreeFontInfo(NULL, fontstruct, 1);
			break;
		case PSmode:
			strncpy(tmp2,s,length);
			tmp2[length]='\0';
			get_ps_font(dpy,context,&font,&size);
			PSSetColor(dpy, drawable, context);
			PSSelectfont(font,size);
			PSMoveto((int)(x+0.9*size),y);
			fprintf(outfp, "gsave\n");
			XGetGCValues(dpy, context, GCFont, &gcval);
			fontstruct = XQueryFont(dpy, gcval.font);
			fprintf(outfp,"%d (%s) stringwidth pop div 1.0 scale\n",
					XTextWidth(fontstruct,s,length),tmp2);
			fprintf(outfp,"90 rotate\n");
			PSShow(tmp2);
			fprintf(outfp, "grestore\n");
      XFreeFontInfo(NULL, fontstruct,1);
			break;
		case FIGmode:
			assert((tmp = (char *) malloc((1 + length) * sizeof(char))) != NULL);
			strncpy(tmp, s, length);
			tmp[length] = '\0';
			fprintf(outfp, "4 0 %d -1 -1 -1 1 0.000 6 12 %d %d %d %s\001\n",
						xfig_font_num,
						6 * length, x + figxoff, y + figyoff, tmp);
			free(tmp);
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_DrawString(Display *dpy, Drawable drawable, GC context, int x, int y,
			char *s, int length)
{
	char *tmp,tmp2[64];
	OutMode tmpOutMode;
	XFontStruct *fontstruct;
	XGCValues gcval;
	char *font;
	int size;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XDrawString(dpy, drawable, context, x, y, s, length);
			break;
		case PSmode:
			strncpy(tmp2,s,length);
			tmp2[length]='\0';
			get_ps_font(dpy,context,&font,&size);
			PSSetColor(dpy, drawable, context);
			PSSelectfont(font,size);
			PSMoveto(x,y);
			fprintf(outfp, "gsave\n");
			XGetGCValues(dpy, context, GCFont, &gcval);
			fontstruct = XQueryFont(dpy, gcval.font);
			fprintf(outfp,"%d (%s) stringwidth pop div 1.0 scale\n",
					XTextWidth(fontstruct,s,length),tmp2);
			PSShow(tmp2);
			fprintf(outfp, "grestore\n");
      XFreeFontInfo(NULL, fontstruct, 1);
			break;
		case FIGmode:
			assert((tmp = (char *) malloc((1 + length) * sizeof(char))) != NULL);
			strncpy(tmp, s, length);
			tmp[length] = '\0';
			fprintf(outfp, "4 0 %d -1 -1 -1 1 0.000 6 12 %d %d %d %s\001\n",
						xfig_font_num,
						6 * length, x + figxoff, y + figyoff, tmp);
			free(tmp);
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_DrawLine(Display * dpy, Drawable drawable, GC context, int x1, int y1,
			int x2, int y2)
{
	int w, h;
	int width,style;
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XDrawLine(dpy, drawable, context, x1, y1, x2, y2);
			break;
		case PSmode:
			/* get the current foreground color from the graphics context */
			PSSetColor(dpy, drawable, context);
			PSSetLineWidth(dpy, context);
			PSSetLineStyle(dpy, context);

			PSNewpath();
			prep_clip(dpy, drawable, &w, &h);
			if (try_clip_line(w, h, &x1, &y1, &x2, &y2) != ELIMINATE) {
				PSMoveto(x1, y1);
				PSLineto(x2, y2);
				PSStroke();
			}
			break;
		case FIGmode:
			width = GetLineWidth(dpy,context);
			style = GetLineStyle(dpy,context);
			prep_clip(dpy, drawable, &w, &h);
			if (try_clip_line(w, h, &x1, &y1, &x2, &y2) != ELIMINATE) {
				fprintf(outfp,
							"2 1 %d %d -1 0 0 0 4.000 -1 0 0\n\t %d %d %d %d 9999 9999\n",
							(style == LineSolid) ? 0 : 1,
							width ? width : 1,
							x1 + figxoff, y1 + figyoff, x2 + figxoff, y2 + figyoff);
			}
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_DrawPoint(Display * dpy, Drawable drawable, GC context, int x1, int y1)
{
	int w, h;
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XDrawPoint(dpy, drawable, context, x1, y1);
			break;
		case PSmode:
			prep_clip(dpy, drawable, &w, &h);
			if (try_clip_point(w, h, &x1, &y1) != ELIMINATE) {
				PSSetColor(dpy, drawable, context);
				PSSetLineWidth(dpy, context);
				PSNewpath();
				PSPointat(x1, y1);
				PSStroke();
			}
			break;
		case FIGmode:
			prep_clip(dpy, drawable, &w, &h);
			if (try_clip_point(w, h, &x1, &y1) != ELIMINATE) {
				fprintf(outfp,
							"1 3 0 1 0 0 -1 21 0.000 1 0.000 %d %d 1 1 %d %d %d %d\n",
							x1 + figxoff, y1 + figyoff, x1 + figxoff, y1 + figyoff,
							x1 + figxoff, y1 + 1 + figyoff);
			}
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_DrawPoints(Display * dpy, Drawable drawable, GC context, XPoint * points,
			int npoints, int mode)
{
	int x, y, w, h;
	int i;
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XDrawPoints(dpy, drawable, context, points, npoints, mode);
			break;
		case PSmode:
			x = y = 0;
			prep_clip(dpy, drawable, &w, &h);
			for (i = 0; i < npoints; i++) {
				PSSetColor(dpy, drawable, context);
				PSSetLineWidth(dpy, context);
				PSNewpath();
				if (mode == CoordModeOrigin) {
					x = points[i].x;
					y = points[i].y;
				}
				else {
					x += points[i].x;
					y += points[i].y;
				}
				if (try_clip_point(w, h, &x, &y) != ELIMINATE) {
					PSPointat(x, y);
					PSStroke();
				}
			}
			break;
		case FIGmode:
			prep_clip(dpy, drawable, &w, &h);
			for (i = 0; i < npoints; i++) {
				if (mode == CoordModeOrigin || i == 0) {
					x = points[i].x;
					y = points[i].y;
				}
				else {
					x += points[i].x;
					y += points[i].y;
				}
				if (try_clip_point(w, h, &x, &y) != ELIMINATE) {
					fprintf(outfp,
								"1 3 0 1 0 0 -1 21 0.000 1 0.000 %d %d %d %d %d %d %d %d\n",
								x + figxoff, y + figxoff, 1, 1, x + figxoff, y + figyoff,
								x + figxoff, y + figyoff + 1);
				}
			}
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_DrawLines(Display * dpy, Drawable drawable, GC context, XPoint * coord,
			int ncoords, int mode)
	 /* mode; CoordModeOrigin or CoordModePrevious */
{
	int i;
	int pcount;
	int nchunks;
	int chunksize;
	int x1, y1, x2, y2, needopen, needclose;
	int w, h;
	int width,style;
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			/* avoid the limit on the length of a multiple line vector
			 * by doing it in multiple calls
			 */
			nchunks = (ncoords - 1) / MAXCHUNK + 1;
			for (i = 0; i < nchunks; i++) {
				if ((chunksize = ncoords - i * MAXCHUNK) > MAXCHUNK) {
					chunksize = MAXCHUNK;
				}
				/* draw one point past to connect this chunk with the 
				 * next. Dont do it for the last chunk
				 */
				if (i < nchunks - 1)
					chunksize++;
				XDrawLines(dpy, drawable, context,
							coord + i * MAXCHUNK, chunksize, mode);
			}
			break;
		case PSmode:
			if (ncoords <= 0)
				return;
			PSNewpath();
			PSSetColor(dpy, drawable, context);
			PSSetLineWidth(dpy, context);
			PSSetLineStyle(dpy, context);
			PSMoveto(coord[0].x, coord[0].y);
			pcount = 0;
			for (i = 1; i < ncoords; i++) {
				if ((coord[i].x == coord[i - 1].x) && (coord[i].y == coord[i - 1].y) &&
							i < ncoords - 1) {
					continue;
				}
				PSLineto(coord[i].x, coord[i].y);
				/* break it up into managable chunks */
				if (pcount > 200) {
					PSStroke();
					PSNewpath();
					PSMoveto(coord[i].x, coord[i].y);
					pcount = 0;
				}
				pcount++;
			}
			PSStroke();
			break;
		case FIGmode:
			width = GetLineWidth(dpy,context);
			style = GetLineStyle(dpy,context);
			prep_clip(dpy, drawable, &w, &h);
			x2 = y2 = 0;
			for (needclose = 0, needopen = 1, i = 0; i < ncoords; i++) {
				x1 = coord[i].x;
				y1 = coord[i].y;
				if (mode == CoordModePrevious) {
					x1 += x2;
					y1 += y2;
					x2 = x1;
					y2 = y1;
				}
				if (try_clip_point(w, h, &x1, &y1) != ELIMINATE) {
					if (needopen) {
						if (needclose) {
							fprintf(outfp, "\t9999 9999\n");
							needclose = 0;
						}
						fprintf(outfp, "2 1 %d %d -1 0 0 0 4.000 -1 0 0\n",
									(style == LineSolid) ? 0 : 1,
									width ? width : 1);
						needopen = 0;
						needclose = 1;
					}
					fprintf(outfp, "\t%d %d\n", x1 + figxoff, y1 + figyoff);
				}
				else {
					needopen = 1;
					x2 = y2 = 0;
				}
			}
			if (needclose) {
				fprintf(outfp, "\t9999 9999\n");
				needclose = 0;
			}
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_DrawSegments(Display * dpy, Drawable drawable, GC context,
			XSegment * segments, int nsegments)
{
	int i;
	int pcount;
	int nchunks;
	int chunksize;
	int x1, y1, x2, y2;
	int w, h;
	XSegment *segp;
	int width,style;
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			/* avoid the limit on the length of a multiple line vector
			 * by doing it in multiple calls
			 */
			nchunks = (nsegments - 1) / MAXCHUNK + 1;
			for (i = 0; i < nchunks; i++) {
				if ((chunksize = nsegments - i * MAXCHUNK) > MAXCHUNK)
					chunksize = MAXCHUNK;
				if (i < nchunks - 1)
					chunksize++;
				XDrawSegments(dpy, drawable, context,
							segments + i * MAXCHUNK, chunksize);
			}
			break;
		case PSmode:
			if (nsegments <= 0)
				return;
			PSNewpath();
			PSSetColor(dpy, drawable, context);
			PSSetLineWidth(dpy, context);
			PSSetLineStyle(dpy, context);
			pcount = 0;
			for (i = 0, segp = &segments[0]; i < nsegments; i++, segp++) {
				/* We could check for redundant segments, but it slows things down.
				 * Eliminate redundancy at the source.
				 */
				PSMoveto(segp->x1, segp->y1);
				PSLineto(segp->x2, segp->y2);
				/* break it up into managable cuhnks */
				if (pcount > 100) {
					PSStroke();
					PSNewpath();
					pcount = 0;
				}
				pcount++;
			}
			PSStroke();
			break;
		case FIGmode:
			width = GetLineWidth(dpy,context);
			style = GetLineStyle(dpy,context);
			prep_clip(dpy, drawable, &w, &h);
			for (i = 0; i < nsegments; i++) {
				x1 = segments[i].x1;
				y1 = segments[i].y1;
				x2 = segments[i].x2;
				y2 = segments[i].y2;
				if (try_clip_line(w, h, &x1, &y1, &x2, &y2) != ELIMINATE) {
					fprintf(outfp,
								"2 1 %d %d -1 0 0 0 4.000 -1 0 0\n\t %d %d %d %d 9999 9999\n",
								(style == LineSolid) ? 0 : 1,
								width ? width : 1,
								x1 + figxoff, y1 + figyoff, x2 + figxoff, y2 + figyoff);
				}
			}
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_DrawRectangles(Display * dpy, Drawable drawable, GC context,
			XRectangle rectangles[], int nrectangles)
{
	int i;

	for (i = 0; i < nrectangles; i++)
		XPS_DrawRectangle(dpy, drawable, context,
					rectangles[i].x, rectangles[i].y,
					rectangles[i].width, rectangles[i].height);
}

void XPS_DrawRectangle(Display * dpy, Drawable drawable, GC context, int x,
			int y, int w, int h)
{
	int width,style;
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XDrawRectangle(dpy, drawable, context, x, y, w, h);
			break;
		case PSmode:
			PSSetColor(dpy, drawable, context);
			PSSetLineWidth(dpy, context);
			PSSetLineStyle(dpy, context);
			PSNewpath();
			PSMoveto(x, y);
			PSLineto(x + w, y);
			PSLineto(x + w, y + h);
			PSLineto(x, y + h);
			PSLineto(x, y);
			PSStroke();
			break;
		case FIGmode:
			width = GetLineWidth(dpy,context);
			style = GetLineStyle(dpy,context);
			fprintf(outfp, "2 1 %d %d -1 0 0 %d 4.000 -1 0 0\n",
						(style == LineSolid) ? 0 : 1,
						width ? width : 1, 0);
			fprintf(outfp, "\t %d %d %d %d\n",
						x + figxoff, y + figxoff, x + w + figxoff, y + figyoff);
			fprintf(outfp, "\t %d %d %d %d\n",
						x + w + figxoff, y + figyoff, x + w + figxoff, y + h + figyoff);
			fprintf(outfp, "\t %d %d %d %d\n",
						x + w + figxoff, y + h + figyoff, x + figxoff, y + h + figyoff);
			fprintf(outfp, "\t %d %d %d %d\n",
						x + figxoff, y + h + figyoff, x + figxoff, y + figyoff);
			fprintf(outfp, "\t 9999 9999\n");
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_FillRectangles(Display * dpy, Drawable drawable, GC context,
			XRectangle rectangles[], int nrectangles)
{
	int i;

	for (i = 0; i < nrectangles; i++)
		XPS_FillRectangle(dpy, drawable, context,
					rectangles[i].x, rectangles[i].y,
					rectangles[i].width, rectangles[i].height);
}

void XPS_FillRectangle(Display * dpy, Drawable drawable, GC context, int x,
			int y, int w, int h)
{
	int fill_color;
	int width,style;
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XFillRectangle(dpy, drawable, context, x, y, w, h);
			break;
		case PSmode:
			PSSetColor(dpy, drawable, context);
			PSNewpath();
			PSMoveto(x, y);
			PSLineto(x + w, y);
			PSLineto(x + w, y + h);
			PSLineto(x, y + h);
			PSClosefill();
			PSStroke();
			break;
		case FIGmode:
			width = GetLineWidth(dpy,context);
			style = GetLineStyle(dpy,context);
			fill_color = 0;
			fprintf(outfp, "2 1 %d %d -1 0 0 %d 4.000 -1 0 0\n",
						(style == LineSolid) ? 0 : 1,
						width ? width : 1,
						fill_color);
			fprintf(outfp, "\t %d %d %d %d\n",
						x + figxoff, y + figxoff, x + w + figxoff, y + figyoff);
			fprintf(outfp, "\t %d %d %d %d\n",
						x + w + figxoff, y + figyoff, x + w + figxoff, y + h + figyoff);
			fprintf(outfp, "\t %d %d %d %d\n",
						x + w + figxoff, y + h + figyoff, x + figxoff, y + h + figyoff);
			fprintf(outfp, "\t %d %d %d %d\n",
						x + figxoff, y + h + figyoff, x + figxoff, y + figyoff);
			fprintf(outfp, "\t 9999 9999\n");
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_DrawArc(Display * dpy, Drawable drawable, GC context, int x, int y,
			int w, int h, int a1, int a2)
{
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XDrawArc(dpy, drawable, context, x, y, w, h, a1, a2);
			break;
		case PSmode:
			PSSetColor(dpy, drawable, context);
			PSSetLineWidth(dpy, context);
			PSSetLineStyle(dpy, context);
			PSNewpath();
			PSArc(x,y,w,h,a1,a2);
			PSStroke();
			break;
		case FIGmode:
			fprintf(stderr, "XPS_DrawArc(): upsupported for xfig output\n");
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_FillArc(Display * dpy, Drawable drawable, GC context, int x, int y,
			int w, int h, int a1, int a2)
{
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XFillArc(dpy, drawable, context, x, y, w, h, a1, a2);
			break;
		case PSmode:
			PSSetColor(dpy, drawable, context);
			PSSetLineWidth(dpy, context);
			PSSetLineStyle(dpy, context);
			PSNewpath();
			PSArc(x,y,w,h,a1,a2);
			PSClosefill();
			PSStroke();
			break;
		case FIGmode:
			fprintf(stderr, "XPS_FillArc(): upsupported for xfig output\n");
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

void XPS_FillPolygon(Display * dpy, Drawable drawable, GC context, XPoint * coord,
			int ncoords, int shape, int mode)
{
	int i;
	int pcount;
	OutMode tmpOutMode;

	if (G_drawable != drawable) {
		tmpOutMode = currentOutMode;
		currentOutMode = Xmode;
	}

	switch (currentOutMode) {
		case Xmode:
			XFillPolygon(dpy, drawable, context, coord, ncoords, shape, mode);
			break;
		case PSmode:
			if (ncoords <= 0)
				return;
			PSNewpath();
			PSSetColor(dpy, drawable, context);
			PSMoveto(coord[0].x, coord[0].y);
			pcount = 0;
			for (i = 1; i < ncoords; i++) {
				PSLineto(coord[i].x, coord[i].y);
				/* break it up into managable chunks */
				if (pcount > 200) {
					PSClosefill();
					PSMoveto(coord[i].x, coord[i].y);
					pcount = 0;
				}
				pcount++;
			}
			PSClosefill();
			break;
		case FIGmode:
			fprintf(stderr,
						"Warning: FIGmode doesn't support XPS_FillPolygon()\n");
			break;
	}

	if (G_drawable != drawable)
		currentOutMode = tmpOutMode;
}

static void PSHeader(XWindowAttributes * info, float requested_scale,
			int box, char *header)
{
	float scale, scalex, scaley;
	int startx, starty, endx, endy;
	time_t clock, time();
	char headerstr[200];
	char *namestr;
	char *ptr;
	char clockstr[100];

	namestr = getenv("LOGNAME");
	if (namestr == NULL) {
		namestr = "?User?";
	}

	global_window_width = info->width;
	global_window_height = info->height;

	/* calculate the scale factors in inches/screenpixel */
	scalex = PAGEWIDTH / global_window_width;
	scaley = PAGEHEIGHT / global_window_height;

	/* use the smaller scale factor so that everything will fit on the page */
	scale = requested_scale * ((scalex < scaley) ? scalex : scaley);

	/* calculate the dots/screenpixel scale factor */
	pix_scale = scale * DPI;

	/* convert to postscriptpoints/screenpixel */
	page_scale = scale * PTS_PER_INCH;

	if (center_output) {
		startx = (int) (PTS_PER_INCH*(PAGEWIDTH - ((double)global_window_width)*scale) / 2.0);
		starty = (int) (PTS_PER_INCH*(PAGEHEIGHT - ((double)global_window_height)*scale) / 2.0);
		endx = startx + (int) (PTS_PER_INCH*(((double)global_window_width)*scale));
		endy = starty + (int) (PTS_PER_INCH*(((double)global_window_height)*scale));

	} else {
		startx = 0;
		starty = 0;
		endx = (int) (PTS_PER_INCH*(((double)global_window_width)*scale));
		endy = (int) (PTS_PER_INCH*(((double)global_window_height)*scale));
	}

	if (page_count == 0) {
		fprintf(outfp, "%%!PS-Adobe-3.0 EPSF-3.0\n");
		fprintf(outfp, "%%%%Title: X11 psdriver window dump\n");
		fprintf(outfp, "%%%%Creator: psdriver.c (wilson..mazer)\n");
		time(&clock);
		fprintf(outfp, "%%%%CreationDate: %s", ctime(&clock));
		fprintf(outfp, "%%%%For: %s\n", namestr);
		fprintf(outfp, "%%%%Pages: (atend)\n");
		fprintf(outfp, "%%%%BoundingBox: %d %d %d %d\n", startx, starty, endx, endy);
		fprintf(outfp, "%%%%EndComments\n\n");
		fprintf(outfp, "save\n");
#ifdef ELLIPSE
		fprintf(outfp, "\
/ellipsedict 8 dict def ellipsedict /mtrx matrix put\n\
/ellipse\n\
  { ellipsedict begin\n\
    /startangle exch def\n\
    /endangle exch def\n\
    /yrad exch def\n\
    /xrad exch def\n\
    /y exch def\n\
    /x exch def\n\
    /savematrix mtrx currentmatrix def\n\
    x y translate\n\
    xrad yrad scale\n\
    0 0 1 startangle endangle arc\n\
    savematrix setmatrix\n\
    end } def\n");
#endif
#ifdef INITGRAPHICS
		fprintf(outfp, "initgraphics\n");
#endif
		page_count = 1;
	}
	else {
		fprintf(outfp, "showpage\n");
		fprintf(outfp, "initgraphics\n");
		page_count += 1;
	}
	fprintf(outfp, "%%%%Page: ? %d\n", page_count);

	if (header != NULL) {
		time(&clock);
		strcpy(clockstr, ctime(&clock));
		if ((ptr = index(clockstr, '\n'))) {
			*ptr = '\0';
		}
		sprintf(headerstr, "%s ::  %s ::  %s",
					clockstr, header, namestr);
		PSNewpath();
		fprintf(outfp, "75 15 moveto\n");
		PSShow(headerstr);
		PSStroke();
	}

	/* center it on the page */
	if (!file_output || center_output) {
		fprintf(outfp, "%f %f translate\n",
					PTS_PER_INCH * (PAGEWIDTH - scale * global_window_width) / 2.0,
					PTS_PER_INCH * (PAGEHEIGHT - scale * global_window_height) / 2.0);

		fprintf(outfp, "%f %f scale\n", page_scale, page_scale);
	}

	if (box) {
		fprintf(outfp, "0.5 setgray\n");
		PSNewpath();
		PSMoveto(0, 0);
		PSLineto(0, global_window_height);
		PSLineto(global_window_width, global_window_height);
		PSLineto(global_window_width, 0);
		PSLineto(0, 0);
		PSStroke();
	}

	PSNewpath();
	PSMoveto(0, 0);
	PSLineto(0, global_window_height);
	PSLineto(global_window_width, global_window_height);
	PSLineto(global_window_width, 0);
	PSLineto(0, 0);
	fprintf(outfp, "eoclip\n");
}

static void PSTrailer(void)
{
	fprintf(outfp, "showpage\n");
	fprintf(outfp, "restore\n");
	fprintf(outfp, "%%%%Trailer\n");
	fprintf(outfp, "%%%%Pages: ? %d\n", page_count);
	page_count = 0;
}

void XPS_PreparePS(Display * dpy, Window window, double scale, int box,
			char *header, int color_flag)
{
	char command[80];
	XWindowAttributes info;
	char *printer;

	color = color_flag;

	current_color = -1;
	current_width = -1;
	current_style = -1;

	XGetWindowAttributes(dpy, window, &info);
	if (file_output) {
		if (strcmp(filename_out, "-") == 0) {
			outfp = stdout;
		}
		else {
			outfp = fopen(filename_out, filemode_out);
		}
	}
	else {
		if ((printer = (char *) getenv("PRINTER")) == NULL) {
			printer = "lpr";
		}
		sprintf(command, "lpr -P%s", printer);
		outfp = popen(command, "w");
	}
	PSHeader(&info, (float) scale, box, header);
	currentOutMode = PSmode;
	G_drawable = window;
	fflush(outfp);
}

void XPS_FinishPS(void)
{
	PSTrailer();
	fflush(outfp);
	currentOutMode = Xmode;
	if (file_output) {
		if (outfp != stdout)
			fclose(outfp);
	}
	else {
		pclose(outfp);
	}
}

static void FIGHeader(XWindowAttributes * info)
{
	fprintf(outfp, "#FIG 2.1\n");	/* fig version number */
	fprintf(outfp, "80 2\n");			/* resolution, origin */
	fprintf(outfp, "6 %d %d 0 0\n",		/* begin compound object */
				info->width + figxoff, info->height + figyoff);
}

static void FIGTrailer(void)
{
	fprintf(outfp, "-6\n");				/* end compound object */
}

void XPS_PrepareFIG(Display * dpy, Window window)
{
	XWindowAttributes info;

	current_color = -1;
	current_width = -1;
	current_style = -1;
	XGetWindowAttributes(dpy, window, &info);
	global_window_width = info.width;
	global_window_height = info.height;
	if (strcmp(filename_out, "-") == 0) {
		outfp = stdout;
	}
	else {
		outfp = fopen(filename_out, filemode_out);
	}
	FIGHeader(&info);
	currentOutMode = FIGmode;

	fflush(outfp);
}

void XPS_FinishFIG(void)
{
	FIGTrailer();
	fflush(outfp);
	currentOutMode = Xmode;
	if (outfp != stdout)
		fclose(outfp);
}

/*****************************************************************************
 **		PSdriver
 ** 		X11 -> postscript library routines
 **		Copyright 1989  Matt Wilson 
 ** 		California Institute of Technology
 **		wilson@smaug.cns.caltech.edu
 **
 ** Permission to use, copy, modify, and distribute this
 ** software and its documentation for any purpose and without
 ** fee is hereby granted, provided that the above copyright
 ** notice appear in all copies and that both that copyright
 ** notice and this permission notice appear in supporting
 ** documentation, and that the name of the California Institute
 ** of Technology not be used in advertising or publicity pertaining 
 ** to distribution of the software without specific, written prior 
 ** permission.  Neither the author nor California Institute of Technology 
 ** make any representations about the suitability of this software for 
 ** any purpose. It is provided "as is" without express or implied warranty.
 **
 ** To use these functions
 ** replace the X11 calls in your code with calls to the
 ** comparable driver functions. 
 ** e.g. instead of 
 **	XDrawLine (dpy, drawable, context, x1, y1, x2, y2)
 ** use
 **	XPS_DrawLine (dpy, drawable, context, x1, y1, x2, y2)
 **
 ** output can be directed to a file or directly to a printer using 
 **	XPS_Set_FileOutput(state)		1=file 0=printer
 **	XPS_Set_Filename(name)
 **
 ** to produce postscript, output add this sequence of calls to your code
 **
 ** XPS_PreparePS(dpy,window,scale,box,header);
 ** refresh_window(window);		your routine to refresh the screen
 ** XPS_FinishPS();
 **
 ****************************************************************************/

#ifdef NEWPAGEPS
void XPS_NewpagePS(dpy, window, scale, box, header)
		 Display *dpy;
		 Window window;
		 float scale;
		 int box;
		 char *header;
{
	XWindowAttributes info;

	XGetWindowAttributes(dpy, window, &info);
	PSHeader(&info, scale, box, header);
	fflush(outfp);
}

#endif

#if 0
static unsigned long GetContextForeground(Display * display, GC gc)
{
	XGCValues val;

	XGetGCValues(display, gc, GCForeground, &val);
	return (val.foreground);
}
#endif

