#include <stdio.h>
#include <math.h>

/* compile with:
%% cc -g -o @ @.c -lm
*/

#define TRUE		1
#define FALSE		0

#define streq(a,b) (strcmp((a),(b)) == 0)
#define isarg(a,b) (strncmp((a),(b),strlen(a)) == 0)

#define FOOTER		1
#define HEADER		0

#define MAXFILES	100	/* max number of input files */
#define DEFAULTRES	72.0	/* 72 points/inch */
#define PAGEWIDTH	8.5	/* inches */
#define PAGEHEIGHT	11.0	/* inches */
#define LEFTMARGIN	0.5	/* inches */
#define RIGHTMARGIN	0.5	/* inches */
#define TOPMARGIN	0.5	/* inches */
#define BOTTOMMARGIN	0.5	/* inches */
				/* The default is to keep the oringal
				 * aspect ratio. */
#define VARIABLE_AR	0	/* enabled if -max is specified */
#define FIXED_AR	1	/* overrides -max */

extern char *getenv(),*strchr();

static void usage(pname)
     char *pname;
{
  fprintf(stderr, "usage: %s [options] [eps-files]\n\
 Options:\n\
[-o outfile]    output file\n\
[-foot]         add footers\n\
[-head]         add headers\n\
[-title string] set page title (%%d will be replaced with pagenum)\n\
[-{left,right,top,bottom}margin inch]\n\
                define page margins in inches\n\
[-max]          maximize print areas\n\
[-rev]          print on page in reverse order\n\
[-bycols]       print by cols (down first) instead of by rows\n\
[-borders]      draw page margins\n\
[-showbb]       draw bounding boxes\n\
[-aligntop]     align image tops\n\
[-justbb]       JUST show the bounding boxes\n\
[-layout nROWSxnCOLS (default is 2x2)]\n\
[-layout onepage (put all files on one page)]\n\
                specify page layout\n\
[-scale f]      <not implemented>\n\
 File options:\n\
   -@   tells epslayout to skip that spot on the page\n\
  -ar   preceding a psfile will keep the original aspect ratio\n", pname);
  exit(0);
}

static void showfoot(titlestr, fp, pnum, footflag, leftmargin)
     char *titlestr;
     FILE *fp;
     int pnum;
     int footflag;
     float leftmargin;
{
  char buf[1000], footerstr[1000];
  char	clockstr[100], *namestr, *p;
  long	clock;

  if (titlestr) {
    sprintf(footerstr, titlestr, pnum);
  } else {
    if ((namestr = getenv("LOGNAME")) == NULL)
      namestr = "?Unknown User?";
    time(&clock);
    strcpy(clockstr,ctime(&clock));
    if (p = strchr(clockstr,'\n')) {
      *p = '\0';
    }
    sprintf(footerstr, "%s ::  %s ::  %s", 
	    clockstr, getcwd(buf, sizeof(buf)), namestr);
  }
  fprintf(fp, "newpath\n");
  fprintf(fp, "/Helvetica findfont 12 scalefont setfont\n");
  if (footflag == FOOTER)
    fprintf(fp, "%d 15 moveto\n",(int)(leftmargin*DEFAULTRES));
  else
    fprintf(fp, "%d 750 moveto\n",(int)(leftmargin*DEFAULTRES));
  fprintf(fp, "(%s) show\n", footerstr);
  fprintf(fp, "stroke\n");
}

static int calc_layout(filecnt, playoutx, playouty)
     int filecnt;
     int *playoutx, *playouty;
{
  if (filecnt <= 1) {
    *playoutx=1;
    *playouty=1;
  } else if (filecnt <=2) {
    *playoutx=1;
    *playouty=2;
  } else if (filecnt <=4) {
    *playoutx=2;
    *playouty=2;
  } else if (filecnt <=6) {
    *playoutx=2;
    *playouty=3;
  } else if (filecnt <=9) {
    *playoutx=3;
    *playouty=3;
  } else if (filecnt <=12) {
    *playoutx=3;
    *playouty=4;
  } else if (filecnt <=16) {
    *playoutx=3;
    *playouty=4;
  } else {
    *playoutx=(int)sqrt((float)filecnt);
    *playouty=(int)(ceil(filecnt / *playoutx));
  }
}


main(argc,argv)
     int argc;
     char **argv;
{
  int		i, p, px, py;
  int		inside_epsfig;
  int		needshowpage;
  int		ar_flag[MAXFILES];
  char		line[1000];
  int		perpage, pagenumber;
  char		tmpstr[100];
  char		*titlestr = NULL;
  char		*delim;
  FILE		*fpout = stdout, *fpps[MAXFILES];
  float		xscale,yscale;
  float		llx, lly, urx, ury;
  float		userscale = 1.0;
  float		epsblock_width, epsblock_height;
  float		epsblock_xoffset, epsblock_yoffset;
  float		usable_width, usable_height;
  float		pageblock_width, pageblock_height;
  float		leftmargin = LEFTMARGIN;
  float		rightmargin = RIGHTMARGIN;
  float		topmargin = TOPMARGIN;
  float		bottommargin = BOTTOMMARGIN;
  int		argn = 0;
  int		layoutx = 2;
  int		layouty = 2;
  int		filecnt = 0;
  int		epsfig_cnt = 0;
  int		footer = 0;
  int		header = 0;
  int		maximize = 0;
  int		onepage=0;
  int		border = 0;
  int		showbb = 0;
  int		align_tops = 0;
  int		justbb = 0;
  int		bycols = 0;
  int		reverse = 0;

  for (i = 0; i < MAXFILES; i++) {
    ar_flag[i] = VARIABLE_AR;
  }

  for (argn = 1; argn < argc; argn++) {
    if (isarg(argv[argn], "-usage")) {
      usage(argv[0]);
    } else if (isarg(argv[argn], "-o")) {
      if ((fpout = fopen(argv[++argn], "w")) == NULL) {
	perror(argv[argn]);
	exit(0);
      }
    } else if (isarg(argv[argn], "-title")) {
      titlestr = argv[++argn];
    } else if (isarg(argv[argn], "-scale")) {
      sscanf(argv[++argn], "%f", &userscale);
    } else if (isarg(argv[argn], "-foot")) {
      footer = 1;
    } else if (isarg(argv[argn], "-head")) {
      header = 1;
    } else if (isarg(argv[argn], "-rev")) {
      reverse = 1;
    } else if (isarg(argv[argn], "-bycols")) {
      bycols = 1;
    } else if (isarg(argv[argn], "-max")) {
      maximize = 1;
    } else if (isarg(argv[argn], "-borders")) {
      border = 1;
    } else if (isarg(argv[argn], "-showbb")) {
      showbb = 1;
    } else if (isarg(argv[argn], "-aligntop")) {
      align_tops = 1;
    } else if (isarg(argv[argn], "-justbb")) {
      showbb = 1;
      justbb = 1;
    } else if (isarg(argv[argn], "-2")) {
      layoutx = 1;
      layouty = 2;
    } else if (isarg(argv[argn], "-4")) {
      layoutx = 2;
      layouty = 2;
    } else if (isarg(argv[argn], "-6")) {
      layoutx = 2;
      layouty = 3;
    } else if (isarg(argv[argn], "-8")) {
      layoutx = 2;
      layouty = 4;
    } else if (isarg(argv[argn], "-layout")) {
      argn++;
      if (isarg(argv[argn], "onepage")) {
	onepage=1;
      } else if (sscanf(argv[argn], "%dx%d", &layoutx,  &layouty) != 2) {
	if (sscanf(argv[argn], "%d", &layoutx) != 1) {
	  fprintf(stderr, "invalid layout  '%s'\n", argv[argn]);
	  usage(argv[0]);
	} else {
	  layouty = layoutx;
	}
      }
    } else if (isarg(argv[argn], "-rightmargin") ||
	       isarg(argv[argn], "-rmargin")) {
      sscanf(argv[++argn], "%f", &rightmargin);
    } else if (isarg(argv[argn], "-leftmargin") ||
	       isarg(argv[argn], "-lmargin")) {
      sscanf(argv[++argn], "%f", &leftmargin);
    } else if (isarg(argv[argn], "-topmargin") ||
	       isarg(argv[argn], "-tmargin")) {
      sscanf(argv[++argn], "%f", &topmargin);
    } else if (isarg(argv[argn], "-bottommargin") ||
	       isarg(argv[argn], "-bmargin")) {
      sscanf(argv[++argn], "%f", &bottommargin);
    } else if (isarg(argv[argn], "-@")) {
      fpps[filecnt] = NULL;
      filecnt++;
    } else if (isarg(argv[argn], "-ar")) {
      ar_flag[filecnt] = FIXED_AR;
    } else if ((fpps[filecnt] = fopen(argv[argn], "r")) != NULL) {
      filecnt++;
    } else {
      fprintf(stderr, "invalid option '%s'\n",argv[argn]);
      usage(argv[0]);
      exit(0);
    }
  }
    
  if (filecnt == 0) {
    fpps[filecnt++] = stdin;
  } else if (onepage) {
    calc_layout(filecnt,  &layoutx, &layouty);
  }
  perpage = layoutx * layouty;

#ifdef DBUG
	fprintf(stderr, "layout: %d x %d (RxC) == %d pages\n",
			layoutx, layouty, perpage);
#endif

  pagenumber = 0;
  inside_epsfig = FALSE;
  needshowpage = FALSE;

  /* Calculate the usable page dimensions in ps points */
  usable_width =
    (PAGEWIDTH - (leftmargin + rightmargin)) * DEFAULTRES;
  usable_height =
    (PAGEHEIGHT - (topmargin + bottommargin)) * DEFAULTRES;

  /* Calculate the size of the block on the page (again) */
  pageblock_width = usable_width / layoutx;
  pageblock_height = usable_height / layouty;

  for (i = 0; i < filecnt; i++) {
    if (fpps[i] == NULL) { /* skip a figure place */
			if (epsfig_cnt % perpage == 0) {
				if (pagenumber != 0)
					fprintf(fpout, "showpage\n");
				else
					fprintf(fpout, "%%!PS-Adobe-2.0\n");
				if (footer)
					showfoot(titlestr, fpout, pagenumber, FOOTER, leftmargin);
				if (header)
					showfoot(titlestr, fpout, pagenumber, HEADER, leftmargin);
				fprintf(fpout, "%g %g translate\n",
						leftmargin * DEFAULTRES, bottommargin * DEFAULTRES);

				if (border) {
					fprintf(fpout, "  %f %f moveto %f %f lineto %f %f lineto\n",
							0.0, 0.0, usable_width, 0.0,
							usable_width, usable_height);
					fprintf(fpout, "  %f %f lineto %f %f lineto stroke\n",
							0.0, usable_height, 0.0, 0.0); }

				needshowpage = TRUE;
				pagenumber++; }
      epsfig_cnt++; }
		else {
      while (fgets(line,1000,fpps[i]) != NULL) {

				if (sscanf(line, "%%%%BoundingBox: %f%f%f%f",
						&llx, &lly, &urx, &ury) == 4) {

					fprintf(fpout, "%% inserted by epslayout...fig=%d\n", epsfig_cnt);

					if (inside_epsfig)
						fprintf(fpout, "restore\n");
					else
						inside_epsfig = TRUE;

		/*  if beginning of a page:
				o showpage for the previous page (if not 1st)
				o translate for margins at beginning of page
				o output head/footer code */
		if (epsfig_cnt % perpage == 0) {

#ifdef DBUG
	fprintf(stderr, "New Page: Page %d\n", pagenumber + 1);
#endif

			if (pagenumber != 0)
				fprintf(fpout, "showpage\n");
			if (footer)
				showfoot(titlestr, fpout, pagenumber, FOOTER, leftmargin);
			if (header)
				showfoot(titlestr, fpout, pagenumber, HEADER, leftmargin);
			fprintf(fpout, "%g %g translate\n",
					leftmargin * DEFAULTRES, bottommargin * DEFAULTRES);

			if (border) {
				fprintf(fpout, "  %f %f moveto %f %f lineto %f %f lineto\n",
						0.0, 0.0, usable_width, 0.0,
						usable_width, usable_height);
				fprintf(fpout, "  %f %f lineto %f %f lineto stroke\n",
						0.0, usable_height, 0.0, 0.0); }

			needshowpage = TRUE;
			pagenumber++; }

					/* Calculate the width and height of the esp file to be inserted */
					epsblock_width = urx - llx;
					epsblock_height = ury - lly;

					/*  If we're `maximizing' we rescale the eps block dimensions.
							to match the aspect ratio of the pageblocks.
							Otherwise we preserve the original aspect ratio.  */

					if (maximize && ar_flag[i] != FIXED_AR) {
						xscale = pageblock_width / epsblock_width;
						yscale = pageblock_height / epsblock_height;
					} else {
		    		/* Expand epsblock to pageblock borders keeping old aspect ratio */
						if (epsblock_height/pageblock_height
									> epsblock_width/pageblock_width)
							xscale = yscale = pageblock_height / epsblock_height;
						else
							xscale = yscale = pageblock_width / epsblock_width; }

					/* We're printing the pth epsblock */
					if (reverse)
						p = (perpage - epsfig_cnt - 1) % perpage;
					else
						p = epsfig_cnt % perpage;

					if (!bycols) {
						/* This goes left to right from top to bottom (Engligh) */
						px = p % layoutx;	/* 0 to layoutx - 1 */
						py = p / layoutx;	/* 0 to layouty - 1 */ }
					else {
						/* This goes top to bottom from left to right (Chinese)*/
						py = p % layouty;
						px = p / layouty; }

#ifdef DBUG
	fprintf(stderr, "%d %d (%d)\n", px, py, layouty - py - 1);
#endif

					/*  Now calculate how much to offset of the epsblock.
							We go in rows from upper left to lower right.  */
					epsblock_xoffset = px * pageblock_width;
					epsblock_yoffset = (layouty - py - 1) * pageblock_height;
	  
					fprintf(fpout, "save\n");
					if (align_tops)
						epsblock_yoffset -= (yscale * epsblock_height - pageblock_height);

					/* compensate for ll corner possibly being non-zero */
					epsblock_xoffset -= (llx*xscale);
					epsblock_yoffset -= (lly*yscale);

					fprintf(fpout, "%g %g translate\n",
							epsblock_xoffset, epsblock_yoffset);
					fprintf(fpout, "%g %g scale\n",  xscale, yscale);

					if (showbb) {
						fprintf(fpout, "       %f %f moveto %f %f lineto %f %f lineto\n",
								llx, lly, llx, ury, urx, ury);
						fprintf(fpout, "       %f %f lineto %f %f lineto stroke\n",
								urx, lly, llx, lly);
						if (justbb) {
							fprintf(fpout, "  newpath\n");
							fprintf(fpout, "  /Helvetica findfont 24 scalefont setfont\n");
							fprintf(fpout, "  %f %f moveto (Fig %d) show stroke\n",
									(llx + urx) / 2.0, (lly + ury) /2.0, epsfig_cnt + 1); } }

					fprintf(fpout, "%% ...inserted by epslayout\n");
					epsfig_cnt++; }

				else if ((strncmp(line, "showpage", 8) != 0) &&
							(strncmp(line, "endpage", 7) != 0) &&
							(strncmp(line, "%%EOF", 5) != 0)) {
					/* copy all but showpage, endpage, and %%EOF lines into the output */
					if (!justbb)
						fprintf(fpout, "%s",line); } }

			if (inside_epsfig) {
				fprintf(fpout, "restore\n");
				inside_epsfig = FALSE; } } }

  if (needshowpage)
    fprintf(fpout, "showpage\n");
  if (fpout != stdout)
    fclose(stdout);
}

