/*
 *      XYLinePlot.c
 *
 *      The AthenaTools Plotter Widget Set - Version 6.0
 *
 *      klin, Tue Jul  7 13:59:47 1992
 *      klin, Mon Jul 27 14:19:31 1992, patchlevel 2
 *                                      Bugs in DrawPS() and DrawStylePS() fixed.
 *                                      Draw() changed for drawing
 *                                      to a pixmap instead of a window.
 *                                      Plot types steps and bars added.
 *                                      Shorter procedure names.
 *      klin, Fri Aug  7 10:07:59 1992, Cast type converters to keep
 *                                      ANSI C compilers quiet.
 *      klin, Sat Aug 15 10:31:50 1992, patchlevel 4
 *                                      Minor changes in PS output
 *                                      Changed <At/..> to <X11/At/..>.
 *      klin, Thu Dec 30 16:47:21 1993, patchlevel 8
 *                                      New resource XtNmarkColor added.
 *                                      Line types SEGMENTS and SEGMENTPOINTS,
 *                                      mark styles DOT, CIRCLE, HORIZONTALBAR
 *                                      and VERTICALBAR added (by Jamie Mazer).
 *                                      Fast update and all stuff removed.
 *                                      Some changes in drawing for improving
 *                                      speed when extending data.
 *                                      Some more minor changes.
 **
 ** 10.98 bjarthur
 **  changed name to Quad
 **  added support to use cardinal numbers if xdata was unspecified
 **  added support to have optional zdata used for error bars
 **  deleted AtSteps and AtBars  (use QuadBarPlot instead)
 */
static char SCCSid[] = "@(#) Plotter V6.0  93/12/30  XYLinePlot.c";


/*   The line plot widget */

#include "PlotterP.h"
#include "AxisCoreP.h"
#include "QuadLinePlotP.h"
#include "Plot.h"


/*   Forward declare all the private widgetclass routines */

static void Draw(AtQuadLinePlotWidget, Display *, Drawable, Region, int);
static void DrawIcon(AtQuadLinePlotWidget, Display *, Drawable, int, int, int, int, Region);
static void Recalc(AtQuadLinePlotWidget, AtScale *, AtScale *, int, int);
static void Attach(AtQuadPlotWidget, BoundingBox *, int);
static void ClassInitialize(void);
static void Initialize(AtQuadLinePlotWidget, AtQuadLinePlotWidget);
static void Destroy(AtQuadLinePlotWidget);
static Boolean SetValues(AtQuadLinePlotWidget, AtQuadLinePlotWidget, AtQuadLinePlotWidget);

static void DrawLines(AtQuadLinePlotWidget, Display *, Drawable, int);
static void DrawZLines(AtQuadLinePlotWidget, Display *, Drawable, int);
static int CalcMarkOffset(AtQuadLinePlotWidget);
static void DrawOneMark(AtQuadLinePlotWidget, Display *, Drawable, int, int, int, int);
static void DrawMarks(AtQuadLinePlotWidget, Display *, Drawable, int);
static void DrawZMarks(AtQuadLinePlotWidget, Display *, Drawable, int);
static void DrawImpulses(AtQuadLinePlotWidget, Display *, Drawable, int);
static void DrawSegments(AtQuadLinePlotWidget, Display *, Drawable, int);

static Boolean AtCvtStringToPlotType(Display *, XrmValue *,
			Cardinal *, XrmValue *, XrmValue *, XtPointer *);
static Boolean AtCvtStringToPlotLine(Display *, XrmValue *,
			Cardinal *, XrmValue *, XrmValue *, XtPointer *);
static Boolean AtCvtStringToPlotMark(Display *, XrmValue *,
			Cardinal *, XrmValue *, XrmValue *, XtPointer *);
static Boolean AtCvtStringToErrBarSide(Display *, XrmValue *,
			Cardinal *, XrmValue *, XrmValue *, XtPointer *);


/*   The resources */

static double dflt_mark_guage = 50.0;
static double dflt_err_bar_size = 0.5;

static void defFG( Widget w, int off, XrmValue *val)
{
	AtQuadLinePlotWidget pw = (AtQuadLinePlotWidget) w;

	val->addr = (XtPointer) & pw->plot.foreground;
}

#define off(field) XtOffsetOf (AtQuadLinePlotRec, lplot.field)
static XtResource resources[] =
{
			{
						XtNplotType, XtCPlotType,
						XtRPlotType, sizeof(AtPlotType),
						off(type), XtRImmediate, (XtPointer) AtTypeLINES
			},
			{
						XtNplotLine, XtCPlotLine,
						XtRPlotLine, sizeof(AtPlotLine),
						off(line), XtRImmediate, (XtPointer) AtLineSOLID
			},
			{
						XtNplotMark, XtCPlotMark,
						XtRPlotMark, sizeof(AtPlotMark),
						off(mark), XtRImmediate, (XtPointer) AtMarkSQUARE
			},
			{
						XtNplotColor, XtCForeground,
						XtRPixel, sizeof(Pixel),
						off(color), XtRCallProc, (XtPointer) defFG
			},
			{
						XtNmarkGuage, XtCMarkGuage,
						XtRDouble, sizeof(double),
						off(mark_guage), XtRDouble, (XtPointer) & dflt_mark_guage
			},
			{
						XtNmarkFill, XtCMarkFill,
						XtRBoolean, sizeof(Boolean),
						off(mark_fill), XtRImmediate, (XtPointer) True
			},
  {
     XtNerrBarSide, XtCErrBarSide,
     XtRErrBarSide, sizeof(AtPlotSide),
     off(err_bar_side), XtRImmediate, (XtPointer) AtSideBOTH
  },
			{
						XtNerrBarSize, XtCErrBarSize,
						XtRDouble, sizeof(double),
						off(err_bar_size), XtRDouble, (XtPointer) & dflt_err_bar_size
			},
};
#undef  off

AtQuadLinePlotClassRec atQuadLinePlotClassRec =
{
			{													/* core fields */
		/* superclass               */ (WidgetClass) & atQuadPlotClassRec,
		/* class_name               */ "AtQuadLinePlot",
		/* widget_size              */ sizeof(AtQuadLinePlotRec),
		/* class_initialize         */ (XtProc) ClassInitialize,
		/* class_part_initialize    */ NULL,
		/* class_inited             */ FALSE,
		/* initialize               */ (XtInitProc) Initialize,
		/* initialize_hook          */ NULL,
		/* pad                      */ NULL,
		/* pad                      */ NULL,
		/* pad                      */ 0,
		/* resources                */ resources,
		/* num_resources            */ XtNumber(resources),
		/* xrm_class                */ NULLQUARK,
		/* pad                      */ FALSE,
		/* pad                      */ FALSE,
		/* pad                      */ FALSE,
		/* pad                      */ FALSE,
		/* destroy                  */ (XtWidgetProc) Destroy,
		/* pad                      */ NULL,
		/* pad                      */ NULL,
		/* set_values               */ (XtSetValuesFunc) SetValues,
		/* set_values_hook          */ NULL,
		/* pad                      */ NULL,
		/* get_values_hook          */ NULL,
		/* pad                      */ NULL,
		/* version                  */ XtVersion,
		/* callback_private         */ NULL,
		/* pad                      */ NULL,
		/* pad                      */ NULL,
		/* pad                      */ NULL,
		/* pad                      */ NULL
			},
			{													/* atPlot fields */
		/* draw                     */ (AtPlotDrawProc) Draw,
		/* draw_icon                */ (AtPlotDrawIconProc) DrawIcon,
		/* recalc                   */ (AtPlotRecalcProc) Recalc
			},
			{													/* atQuadPlot fields */
		/* attach_data              */ (AtQuadPlotAttachProc) Attach,
			},
			{													/* atQuadLinePlot fields */
		/* empty                    */ 0
			}
};

WidgetClass atQuadLinePlotWidgetClass = (WidgetClass) & atQuadLinePlotClassRec;


/*   The core member procs */

static void ClassInitialize(void)
{
	XtSetTypeConverter(XtRString, XtRPlotType,
				AtCvtStringToPlotType, NULL, 0, XtCacheNone, NULL);
	XtSetTypeConverter(XtRString, XtRPlotLine,
				AtCvtStringToPlotLine, NULL, 0, XtCacheNone, NULL);
	XtSetTypeConverter(XtRString, XtRPlotMark,
				AtCvtStringToPlotMark, NULL, 0, XtCacheNone, NULL);
	XtSetTypeConverter(XtRString, XtRErrBarSide,
				AtCvtStringToErrBarSide, NULL, 0, XtCacheNone, NULL);

	*SCCSid = *SCCSid;						/* Keep gcc quiet */
}


/*   Some helper functions for GC management */

#define DOTTED      0						/* the line style dashes */
static char dotted[] =
{1, 1};
#define DASHED      1
static char dashed[] =
{2, 2};
#define DOTDASHED   2
static char dotdashed[] =
{3, 2, 1, 2};
#define DOTTED2     3
static char dotted2[] =
{1, 2};
#define DOTTED3     4
static char dotted3[] =
{1, 3};
#define DOTTED4     5
static char dotted4[] =
{1, 4};
#define DOTTED5     6
static char dotted5[] =
{1, 5};
#define DASHED3     7
static char dashed3[] =
{3, 3};
#define DASHED4     8
static char dashed4[] =
{4, 4};
#define DASHED5     9
static char dashed5[] =
{5, 5};
#define DOTDASHED2  10
static char dotdashed2[] =
{4, 4, 1, 4};

static struct {									/* The dashes list */
	char *dashes;
	int length;
} dashlist[] = {

	{
		dotted, 2
	},
	{
		dashed, 2
	},
	{
		dotdashed, 4
	},
	{
		dotted2, 2
	},
	{
		dotted3, 2
	},
	{
		dotted4, 2
	},
	{
		dotted5, 2
	},
	{
		dashed3, 2
	},
	{
		dashed4, 2
	},
	{
		dashed5, 2
	},
	{
		dotdashed2, 4
	}
};

#define HUGE_DASH   200

static void GetLineGC(AtQuadLinePlotWidget w);
static void GetLineGC(w)
		 AtQuadLinePlotWidget w;
{
	XGCValues gcv;
	XtGCMask gcmask;
	int dash;

	gcv.foreground = w->lplot.color;
	gcv.background = w->plot.background;
	gcmask = GCForeground | GCBackground;

	/*
	 *   We have to create a unique GC with the appropriate dash list
	 *   for each line type. XtGetGC() shares same GCs between different
	 *   widgets and we had to create a new GC for each line using the
	 *   Xlib function XCreateGC(). Setting the dashes member in the GC
	 *   to a unlikely value (>200) allows us to use XtGetGC() and so
	 *   to share same GCs between line widgets with the same line type.
	 */
	switch (w->lplot.line) {
		case AtLineDOTTED:
			gcv.line_style = LineOnOffDash;
			dash = DOTTED;
			gcv.dashes = (char) HUGE_DASH + DOTTED;
			gcmask |= GCDashList;
			break;
		case AtLineDASHED:
			gcv.line_style = LineOnOffDash;
			dash = DASHED;
			gcv.dashes = (char) HUGE_DASH + DASHED;
			gcmask |= GCDashList;
			break;
		case AtLineDOTDASHED:
			gcv.line_style = LineOnOffDash;
			dash = DOTDASHED;
			gcv.dashes = (char) HUGE_DASH + DOTDASHED;
			gcmask |= GCDashList;
			break;
		case AtLineDOTTED2:
			gcv.line_style = LineOnOffDash;
			dash = DOTTED2;
			gcv.dashes = (char) HUGE_DASH + DOTTED2;
			gcmask |= GCDashList;
			break;
		case AtLineDOTTED3:
			gcv.line_style = LineOnOffDash;
			dash = DOTTED3;
			gcv.dashes = (char) HUGE_DASH + DOTTED3;
			gcmask |= GCDashList;
			break;
		case AtLineDOTTED4:
			gcv.line_style = LineOnOffDash;
			dash = DOTTED4;
			gcv.dashes = (char) HUGE_DASH + DOTTED4;
			gcmask |= GCDashList;
			break;
		case AtLineDOTTED5:
			gcv.line_style = LineOnOffDash;
			dash = DOTTED5;
			gcv.dashes = (char) HUGE_DASH + DOTTED5;
			gcmask |= GCDashList;
			break;
		case AtLineDASHED3:
			gcv.line_style = LineOnOffDash;
			dash = DASHED3;
			gcv.dashes = (char) HUGE_DASH + DASHED3;
			gcmask |= GCDashList;
			break;
		case AtLineDASHED4:
			gcv.line_style = LineOnOffDash;
			dash = DASHED4;
			gcv.dashes = (char) HUGE_DASH + DASHED4;
			gcmask |= GCDashList;
			break;
		case AtLineDASHED5:
			gcv.line_style = LineOnOffDash;
			dash = DASHED5;
			gcv.dashes = (char) HUGE_DASH + DASHED5;
			gcmask |= GCDashList;
			break;
		case AtLineDOTDASHED2:
			gcv.line_style = LineOnOffDash;
			dash = DOTDASHED2;
			gcv.dashes = (char) HUGE_DASH + DOTDASHED2;
			gcmask |= GCDashList;
			break;
		case AtLineSOLID:
		default:
			gcv.line_style = LineSolid;
			dash = -1;
			break;
	}
	gcv.line_width = w->plot.line_width;
	gcmask |= GCLineStyle | GCLineWidth;

	w->lplot.line_gc = XtGetGC(XtParent((Widget) w), gcmask, &gcv);

	if (dash >= 0) {							/* now set the dashlist */
		XSetDashes(XtDisplay(XtParent((Widget) w)), w->lplot.line_gc,
					0, dashlist[dash].dashes, dashlist[dash].length);
	}
}

static void GetMarkGC(AtQuadLinePlotWidget w);
static void GetMarkGC(w)
		 AtQuadLinePlotWidget w;
{
	XGCValues gcv;
	XtGCMask gcmask;

	gcv.line_style = LineSolid;
	gcv.foreground = w->lplot.color;
	gcv.background = w->plot.background;
	gcmask = GCForeground | GCBackground | GCLineStyle;

	w->lplot.mark_gc = XtGetGC(XtParent((Widget) w), gcmask, &gcv);
}

#define FreeLineGC(w) XtReleaseGC((Widget)w, w->lplot.line_gc)
#define FreeMarkGC(w) XtReleaseGC((Widget)w, w->lplot.mark_gc)


/*  The initialize/destroy/setvalues procs */

static void Initialize(request, new)
		 AtQuadLinePlotWidget request, new;
{
	GetLineGC(new);
	GetMarkGC(new);
}

static void Destroy(w)
		 AtQuadLinePlotWidget w;
{
	FreeLineGC(w);
	FreeMarkGC(w);
	if (w->qplot.pix)
		XtFree((char *) w->qplot.pix);
}

static Boolean SetValues(current, request, new)
		 AtQuadLinePlotWidget current, request, new;
{
#define Changed(field) (new->lplot.field != current->lplot.field)
	Boolean redraw = False;

	if ((new->plot.foreground != current->plot.foreground) ||
				(new->plot.background != current->plot.background) ||
				(new->plot.line_width != current->plot.line_width) ||
				(new->lplot.color != current->lplot.color)) {
		FreeLineGC(new);
		FreeMarkGC(new);
		GetLineGC(new);
		GetMarkGC(new);
		redraw = True;
	}
	if (Changed(line)) {
		FreeLineGC(new);
		GetLineGC(new);
		redraw = True;
	}
	if (Changed(color)) {
		FreeMarkGC(new);
		GetMarkGC(new);
		redraw = True;
	}
	if (Changed(type) || Changed(mark)) {
		redraw = True;
	}
	if (redraw) {
		AtPlotterRedrawRequired((Widget) new);
	}
	/* Nothing to redisplay */
	return False;
#undef Changed
}


/*  These routines are the ones called by the parent plot widget */

#define lp ((AtQuadLinePlotWidget)self)
#define PIX ((AtQuadPoint*) lp->qplot.pix)
/* NB: PIX is NOT an lvalue (on some very picky compilers!!!) */


static void Attach(AtQuadPlotWidget self, BoundingBox *bb, int extending)
{
	Cardinal num;
	int i;
  double v, z1, z2;

	/* now calculate the bounding box */
  bb->xmax = bb->ymax = -HUGE_VAL;
  bb->xmin = bb->ymin = HUGE_VAL;
  for (i = 0; i < lp->qplot.num_points; i++) {
    v = AtQuadPlotGetXValue(self, i);
    bb->xmax = Max(bb->xmax, v);
    bb->xmin = Min(bb->xmin, v);
    v = AtQuadPlotGetYValue(self, i);
    bb->ymax = Max(bb->ymax, v);
    bb->ymin = Min(bb->ymin, v);
    if(lp->qplot.z1data != NULL) {
      z1 = AtQuadPlotGetZ1Value(self, i);
      z2 = (lp->qplot.z2data == NULL) ? AtQuadPlotGetZ1Value(self, i) :
      			AtQuadPlotGetZ2Value(self, i);
			if(lp->lplot.err_bar_side != AtSideUPPER)
      	bb->ymin = Min(bb->ymin, v - z1);
			if(lp->lplot.err_bar_side != AtSideLOWER)
      	bb->ymax = Max(bb->ymax, v + z2); } }

	/* Free old data if needed */
	if (!extending) {
		if (lp->qplot.release_when_attached) {
			if (lp->qplot.pix)
				XtFree((char *) lp->qplot.pix);
			lp->qplot.num_alloc = 0;
			lp->qplot.pix = NULL;
		}
	}

	/* We have enough space for the data */
	if (lp->qplot.num_alloc >= lp->qplot.num_points)
		return;

	/* Calc number of data to allocate and allocate them */
	if (lp->qplot.allocate_in_slices)
		num = 1 + (lp->qplot.num_points | 0x7f);
	else
		num = lp->qplot.num_points;
	if (num > lp->qplot.num_alloc) {
		if (extending && lp->qplot.num_alloc > 0)
			lp->qplot.pix = XtRealloc((char *) lp->qplot.pix,
						num * sizeof(AtQuadPoint));
		else
			lp->qplot.pix = XtMalloc(num * sizeof(AtQuadPoint));
		lp->qplot.num_alloc = num;
	}
}


/*  Internal procs for drawing lines/marks/impulses/steps/bars/segments */

static void DrawLines(AtQuadLinePlotWidget self, Display * dpy, Drawable drw,
			int refresh)
{
	int i, start, end;

	if (refresh && lp->qplot.draw_from > 0 &&
				lp->qplot.draw_to >= lp->qplot.draw_from) {
		start = lp->qplot.draw_from;
		end = lp->qplot.draw_to + 1;
	}
	else {
		start = 1;
		end = lp->qplot.num_calc;
	}

	for (i = start; i < end; i++) {
		XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
					PIX[i - 1].x, PIX[i - 1].y,
					PIX[i].x, PIX[i].y);
	}
}

static void DrawZLines(AtQuadLinePlotWidget self, Display * dpy, Drawable drw,
			int refresh)
{
	int i, start, end;

	if (refresh && lp->qplot.draw_from > 0 &&
				lp->qplot.draw_to >= lp->qplot.draw_from) {
		start = lp->qplot.draw_from;
		end = lp->qplot.draw_to + 1;
	}
	else {
		start = 1;
		end = lp->qplot.num_calc;
	}

	if ((lp->qplot.z2data != NULL) ||
				((lp->qplot.z2data == NULL) &&
							(lp->lplot.err_bar_side == AtSideBOTH))) {
		for (i = start; i < end; i++) {
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i - 1].x, PIX[i - 1].z1, PIX[i].x, PIX[i].z1);
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i - 1].x, PIX[i - 1].z2, PIX[i].x, PIX[i].z2);
		}
	}
	else {
		for (i = start; i < end; i++) {
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i - 1].x, PIX[i - 1].z1, PIX[i].x, PIX[i].z1);
		}
	}
}

static int CalcMarkOffset(self)
		 AtQuadLinePlotWidget self;
{
	AtPlotterWidget pw = (AtPlotterWidget) XtParent((Widget) self);
	AtPlotterLayout *pwl = &pw->plotter.layout;

	return ((int) ((double) pwl->width / (double) self->lplot.mark_guage / 2.0
							+ 0.5));
}

static void DrawOneMark(AtQuadLinePlotWidget self, Display * dpy, Drawable drw,
			int x, int y, int off, int hbar)
{
	XPoint pts[5];

	switch (lp->lplot.mark) {
		case AtMarkPLUS:
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x, y - off, x, y + off);
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x - off, y, x + off, y);
			break;
		case AtMarkHBAR:
			if (hbar) {
				XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
							x - off, y, x + off, y);
				break;
			}
			/* Fall thru */
		case AtMarkVBAR:
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x, y - off, x, y + off);
			break;
		case AtMarkXMARK:
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x - off, y - off, x + off, y + off);
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x - off, y + off, x + off, y - off);
			break;
		case AtMarkSTAR:
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x, y - off, x, y + off);
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x - off, y, x + off, y);
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x - off, y - off, x + off, y + off);
			XPS_DrawLine(dpy, drw, lp->lplot.mark_gc,
						x - off, y + off, x + off, y - off);
			break;
		case AtMarkTRIANGLE1:
			pts[0].x = x - off;
			pts[0].y = y + off;
			pts[1].x = x + off;
			pts[1].y = y + off;
			pts[2].x = x;
			pts[2].y = y - off;
			pts[3].x = x - off;
			pts[3].y = y + off;
			if (self->lplot.mark_fill == True)
				XPS_FillPolygon(dpy, drw, lp->lplot.mark_gc,
							pts, 4, Convex, CoordModeOrigin);
			else
				XPS_DrawLines(dpy, drw, lp->lplot.mark_gc,
							pts, 4, CoordModeOrigin);
			break;
		case AtMarkTRIANGLE2:
			pts[0].x = x - off;
			pts[0].y = y - off;
			pts[1].x = x + off;
			pts[1].y = y - off;
			pts[2].x = x;
			pts[2].y = y + off;
			pts[3].x = x - off;
			pts[3].y = y - off;
			if (self->lplot.mark_fill == True)
				XPS_FillPolygon(dpy, drw, lp->lplot.mark_gc,
							pts, 4, Convex, CoordModeOrigin);
			else
				XPS_DrawLines(dpy, drw, lp->lplot.mark_gc,
							pts, 4, CoordModeOrigin);
			break;
		case AtMarkTRIANGLE3:
			pts[0].x = x - off;
			pts[0].y = y - off;
			pts[1].x = x - off;
			pts[1].y = y + off;
			pts[2].x = x + off;
			pts[2].y = y;
			pts[3].x = x - off;
			pts[3].y = y - off;
			if (self->lplot.mark_fill == True)
				XPS_FillPolygon(dpy, drw, lp->lplot.mark_gc,
							pts, 4, Convex, CoordModeOrigin);
			else
				XPS_DrawLines(dpy, drw, lp->lplot.mark_gc,
							pts, 4, CoordModeOrigin);
			break;
		case AtMarkTRIANGLE4:
			pts[0].x = x + off;
			pts[0].y = y - off;
			pts[1].x = x + off;
			pts[1].y = y + off;
			pts[2].x = x - off;
			pts[2].y = y;
			pts[3].x = x + off;
			pts[3].y = y - off;
			if (self->lplot.mark_fill == True)
				XPS_FillPolygon(dpy, drw, lp->lplot.mark_gc,
							pts, 4, Convex, CoordModeOrigin);
			else
				XPS_DrawLines(dpy, drw, lp->lplot.mark_gc,
							pts, 4, CoordModeOrigin);
			break;
		case AtMarkDIAMOND:
			pts[0].x = x;
			pts[0].y = y - off;
			pts[1].x = x - off;
			pts[1].y = y;
			pts[2].x = x;
			pts[2].y = y + off;
			pts[3].x = x + off;
			pts[3].y = y;
			pts[4].x = x;
			pts[4].y = y - off;
			if (self->lplot.mark_fill == True)
				XPS_FillPolygon(dpy, drw, lp->lplot.mark_gc,
							pts, 5, Convex, CoordModeOrigin);
			else
				XPS_DrawLines(dpy, drw, lp->lplot.mark_gc,
							pts, 5, CoordModeOrigin);
			break;
		case AtMarkCIRCLE:
			if (self->lplot.mark_fill == True)
				XPS_FillArc(dpy, drw, lp->lplot.mark_gc,
							x - off, y - off, 2 * off, 2 * off, 0, 360 * 64);
			else
				XPS_DrawArc(dpy, drw, lp->lplot.mark_gc,
							x - off, y - off, 2 * off, 2 * off, 0, 360 * 64);
			break;
		case AtMarkDOT:
			if (off > 0)
				XFillArc(dpy, drw, lp->lplot.mark_gc,
							x - off, y - off, 2 * off, 2 * off, 0, 360 * 64);
			else
				XPS_DrawPoint(dpy, drw, lp->lplot.mark_gc, x, y);
			break;
		case AtMarkSQUARE:
		default:
			if (self->lplot.mark_fill == True)
				XPS_FillRectangle(dpy, drw, lp->lplot.mark_gc,
							x - off, y - off, 2 * off, 2 * off);
			else
				XPS_DrawRectangle(dpy, drw, lp->lplot.mark_gc,
							x - off, y - off, 2 * off, 2 * off);
	}
}

static void DrawMarks(AtQuadLinePlotWidget self, Display * dpy,
			Drawable drw, int refresh)
{
	int i, off;

	off = CalcMarkOffset(self);
	if (refresh && lp->qplot.draw_to >= lp->qplot.draw_from) {
		for (i = lp->qplot.draw_from; i <= lp->qplot.draw_to; i++)
			DrawOneMark(self, dpy, drw, PIX[i].x, PIX[i].y, off, True);
	}
	else {
		for (i = 0; i < lp->qplot.num_calc; i++)
			DrawOneMark(self, dpy, drw, PIX[i].x, PIX[i].y, off, True);
	}
}

static void DrawZMarks(AtQuadLinePlotWidget self, Display * dpy,
			Drawable drw, int refresh)
{
	int i, off, start, end;

	off = (int)ceil(lp->lplot.err_bar_size*CalcMarkOffset(self));
	if (refresh && lp->qplot.draw_to >= lp->qplot.draw_from) {
		start = lp->qplot.draw_from;
		end = lp->qplot.draw_to + 1;
	}
	else {
		start = 0;
		end = lp->qplot.num_calc;
	}

	if ((lp->qplot.z2data != NULL) ||
				((lp->qplot.z2data == NULL) &&
							(lp->lplot.err_bar_side == AtSideBOTH))) {
		for (i = start; i < end; i++) {
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i].x, PIX[i].z1, PIX[i].x, PIX[i].z2);
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i].x - off, PIX[i].z1, PIX[i].x + off, PIX[i].z1);
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i].x - off, PIX[i].z2, PIX[i].x + off, PIX[i].z2);
		}
	}
	else {
		for (i = start; i < end; i++) {
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i].x, PIX[i].z1, PIX[i].x, PIX[i].y);
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i].x - off, PIX[i].z1, PIX[i].x + off, PIX[i].z1);
		}
	}
}

static void DrawImpulses(AtQuadLinePlotWidget self, Display * dpy,
			Drawable drw, int refresh)
{
	int i;

	if (refresh && lp->qplot.draw_to >= lp->qplot.draw_from) {
		for (i = lp->qplot.draw_from; i <= lp->qplot.draw_to; i++)
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i].x, PIX[i].y, PIX[i].x, lp->lplot.impulse_y);
	}
	else {
		for (i = 0; i < lp->qplot.num_calc; i++)
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						PIX[i].x, PIX[i].y, PIX[i].x, lp->lplot.impulse_y);
	}
}

static void DrawSegments(AtQuadLinePlotWidget self, Display * dpy,
			Drawable drw, int refresh)
{
	int i, l;

	l = lp->qplot.num_calc - 1;
	for (i = 0; i < l; i += 2) {
		XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
					PIX[i].x, PIX[i].y, PIX[i + 1].x, PIX[i + 1].y);
	}
}


/*  Draw the line clipped by the given region.  */

static void Draw(AtQuadLinePlotWidget self, Display * dpy, Drawable drw,
			Region region, int refresh)
{
	AtPlotterWidget pw = (AtPlotterWidget) XtParent((Widget) self);
	AtPlotterLayout *pwl = &pw->plotter.layout;
	XRectangle linerectangle, markrectangle;
	Region lineregion, markregion;
	int off;

	if (lp->qplot.num_calc <= 0)	/* Nothing to draw */
		return;

	/* Get clip regions for lines and marks from the plotter layout */
	linerectangle.x = pwl->x1 + 1;
	linerectangle.y = pwl->y1 + 1;
	linerectangle.width = pwl->width - 2;
	linerectangle.height = pwl->height - 2;
	off = CalcMarkOffset(self);
	markrectangle.x = pwl->x1 - off;
	markrectangle.y = pwl->y1 - off;
	markrectangle.width = pwl->width + 2 * off;
	markrectangle.height = pwl->height + 2 * off;

	/* Now create and set the regions */
	lineregion = XCreateRegion();
	markregion = XCreateRegion();
	XUnionRectWithRegion(&linerectangle, lineregion, lineregion);
	XUnionRectWithRegion(&markrectangle, markregion, markregion);
	XSetRegion(dpy, lp->lplot.line_gc, lineregion);
	XSetRegion(dpy, lp->lplot.mark_gc, markregion);

#ifdef TRACE
	fprintf(stderr, "Draw %d lines/points\n", lp->qplot.num_points);
#endif

	if (lp->qplot.draw_all_when_extended)
		refresh = False;

	switch (lp->lplot.type) {
		case AtTypePOINTS:
			DrawMarks(self, dpy, drw, refresh);
			if (lp->qplot.z1data != NULL)
				DrawZMarks(self, dpy, drw, refresh);
			break;
		case AtTypeLINEPOINTS:
			DrawLines(self, dpy, drw, refresh);
			DrawMarks(self, dpy, drw, refresh);
			if (lp->qplot.z1data != NULL)
				DrawZMarks(self, dpy, drw, refresh);
			break;
		case AtTypeIMPULSES:
			DrawImpulses(self, dpy, drw, refresh);
			break;
		case AtTypeLINEIMPULSES:
			DrawLines(self, dpy, drw, refresh);
			DrawImpulses(self, dpy, drw, refresh);
			break;
		case AtTypeSEGMENTS:
			DrawSegments(self, dpy, drw, refresh);
			break;
		case AtTypeSEGMENTPOINTS:
			DrawSegments(self, dpy, drw, refresh);
			DrawMarks(self, dpy, drw, refresh);
			break;
		case AtTypeLINES:
		default:
			DrawLines(self, dpy, drw, refresh);
			if (lp->qplot.z1data != NULL)
				DrawZLines(self, dpy, drw, refresh);
			break;
	}

	/* Unset and destroy the regions */
	XSetClipMask(dpy, lp->lplot.line_gc, None);
	XSetClipMask(dpy, lp->lplot.mark_gc, None);
	XDestroyRegion(lineregion);
	XDestroyRegion(markregion);
}


/*  Draw the "icon" in the given place.  */

static void DrawIcon(AtQuadLinePlotWidget self, Display * dpy, Drawable drw,
			int x1, int y1, int width, int height, Region region)
{
	int off;

	y1 += height >> 1;
	switch (lp->lplot.type) {
		case AtTypePOINTS:
			off = CalcMarkOffset(self);
			x1 += width >> 1;
			DrawOneMark(self, dpy, drw, x1, y1, off, True);
			break;
		case AtTypeLINEPOINTS:
		case AtTypeSEGMENTPOINTS:
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						x1, y1, x1 + width, y1);
			off = CalcMarkOffset(self);
			x1 += width >> 1;
			DrawOneMark(self, dpy, drw, x1, y1, off, False);
			break;
		case AtTypeLINEIMPULSES:
		case AtTypeIMPULSES:
		case AtTypeSEGMENTS:
		case AtTypeLINES:
		default:
			XPS_DrawLine(dpy, drw, lp->lplot.line_gc,
						x1, y1, x1 + width, y1);
			break;
	}
}


/*  Recalc the data according to the passed x and y scales */

static void log_modify(AtScale *xs, double *x)
{
	double l;

	if ((xs->transform == AtTransformLOGARITHMIC) && ((*x) <= 0.0)) {
		l = log10(xs->low) - 5.0;
		(*x) = pow(10.0, l); }
}

static void Recalc(AtQuadLinePlotWidget self, AtScale * xs, AtScale * ys,
			int from, int to)
{
	double x, y, z1, z2;
	int i;

	if (from > to) {							/* Recalc all */
		from = 0;
		to = lp->qplot.num_points - 1;
		lp->qplot.draw_from = 0;
		lp->qplot.draw_to = -1;
	}
	else {												/* Recalc partial */
		if (lp->qplot.num_calc < from)
			from = lp->qplot.num_calc;
		lp->qplot.draw_from = from;
		lp->qplot.draw_to = to;
	}
	lp->qplot.num_calc = to + 1;

	/* Calc pixel positions */
	assert(lp->qplot.ydata != NULL);
	for (i = from; i <= to; i++) {
		x = AtQuadPlotGetXValue((AtQuadPlotWidget) lp, i);
		y = AtQuadPlotGetYValue((AtQuadPlotWidget) lp, i);
		log_modify(xs,&x);
		log_modify(ys,&y);
		PIX[i].x = AtScaleUserToPixel(xs, x);
		PIX[i].y = AtScaleUserToPixel(ys, y);
	}

	if (lp->qplot.z1data != NULL) {
		if ((lp->qplot.z2data != NULL) ||
					((lp->qplot.z2data == NULL) &&
								(lp->lplot.err_bar_side == AtSideBOTH))) {
			for (i = from; i <= to; i++) {
				y = AtQuadPlotGetYValue((AtQuadPlotWidget) lp, i);
				z1 = AtQuadPlotGetZ1Value((AtQuadPlotWidget) lp, i);
				z2 = (lp->qplot.z2data == NULL) ?
							AtQuadPlotGetZ1Value((AtQuadPlotWidget) lp, i) :
							AtQuadPlotGetZ2Value((AtQuadPlotWidget) lp, i);
		    log_modify(ys,&z1);
		    log_modify(ys,&z2);
				PIX[i].z1 = AtScaleUserToPixel(ys, y + z1);
				PIX[i].z2 = AtScaleUserToPixel(ys, y - z2);
			}
		}
		else {
			for (i = from; i <= to; i++) {
				y = AtQuadPlotGetYValue((AtQuadPlotWidget) lp, i);
				z1 = AtQuadPlotGetZ1Value((AtQuadPlotWidget) lp, i);
		    log_modify(ys,&z1);
				PIX[i].z1 = (lp->lplot.err_bar_side == AtSideUPPER) ?
							AtScaleUserToPixel(ys, y + z1) :
							AtScaleUserToPixel(ys, y - z1);
			}
		}
	}

	/* Calc the impulse height */
	if (ys->low <= 0.0)
		lp->lplot.impulse_y = AtScaleUserToPixel(ys, 0.0);
	else
		lp->lplot.impulse_y = AtScaleUserToPixel(ys, ys->low);
}
#undef lp


/*   Resource converters */

typedef struct {
	char *string;
	int val;
} AtGenericRec;

static AtGenericRec type_table[] =
{
			{"LINES", (int) AtTypeLINES},
			{"POINTS", (int) AtTypePOINTS},
			{"IMPULSES", (int) AtTypeIMPULSES},
			{"SEGMENTS", (int) AtTypeSEGMENTS},
			{"LINEPOINTS", (int) AtTypeLINEPOINTS},
			{"LINEIMPULSES", (int) AtTypeLINEIMPULSES},
			{"SEGMENTPOINTS", (int) AtTypeSEGMENTPOINTS},
			{NULL, -1}
};

static AtGenericRec line_table[] =
{
			{"SOLID", (int) AtLineSOLID},
			{"DOTTED", (int) AtLineDOTTED},
			{"DASHED", (int) AtLineDASHED},
			{"DOTDASHED", (int) AtLineDOTDASHED},
			{"DOTTED2", (int) AtLineDOTTED2},
			{"DOTTED3", (int) AtLineDOTTED3},
			{"DOTTED4", (int) AtLineDOTTED4},
			{"DOTTED5", (int) AtLineDOTTED5},
			{"DASHED3", (int) AtLineDASHED3},
			{"DASHED4", (int) AtLineDASHED4},
			{"DASHED5", (int) AtLineDASHED5},
			{"DOTDASHED2", (int) AtLineDOTDASHED2},
			{NULL, -1}
};

static AtGenericRec mark_table[] =
{
			{"SQUARE", (int) AtMarkSQUARE},
			{"DOT", (int) AtMarkDOT},
			{"CIRCLE", (int) AtMarkCIRCLE},
			{"PLUS", (int) AtMarkPLUS},
			{"XMARK", (int) AtMarkXMARK},
			{"STAR", (int) AtMarkSTAR},
			{"DIAMOND", (int) AtMarkDIAMOND},
			{"TRIANGLE1", (int) AtMarkTRIANGLE1},
			{"TRIANGLE2", (int) AtMarkTRIANGLE2},
			{"TRIANGLE3", (int) AtMarkTRIANGLE3},
			{"TRIANGLE4", (int) AtMarkTRIANGLE4},
			{"HBAR", (int) AtMarkHBAR},
			{"VBAR", (int) AtMarkVBAR},
			{NULL, -1}
};

static AtGenericRec bar_side_table[] =
{
      {"UPPER", (int) AtSideUPPER},
      {"LOWER", (int) AtSideLOWER},
      {"BOTH", (int) AtSideBOTH},
      {NULL, -1}
};

static int parse_generic(char *var, AtGenericRec * table)
{
	int i;

	i = 0;
	while (table[i].string != NULL) {
		if (!strcasecmp(table[i].string, var))
			return (table[i].val);
		i++;
	}

	return (-1);
}

static Boolean AtCvtStringToPlotType(Display * dpy, XrmValue * args,
			Cardinal * num_args, XrmValue * from, XrmValue * to,
			XtPointer * converter_data)
{
	static AtPlotType type;

	assert(to->addr == NULL);

	type = parse_generic(from->addr, type_table);

	if (type == -1)
		XtDisplayStringConversionWarning(dpy, from->addr, XtRPlotType);
	else {
		to->addr = (caddr_t) & type;
		to->size = sizeof(AtPlotType);
	}

	return (True);
}

static Boolean AtCvtStringToPlotLine(Display * dpy, XrmValue * args,
			Cardinal * num_args, XrmValue * from, XrmValue * to,
			XtPointer * converter_data)
{
	static AtPlotLine line;

	assert(to->addr == NULL);

	line = parse_generic(from->addr, line_table);

	if (line == -1)
		XtDisplayStringConversionWarning(dpy, from->addr, XtRPlotLine);
	else {
		to->addr = (caddr_t) & line;
		to->size = sizeof(AtPlotLine);
	}

	return (True);
}

static Boolean AtCvtStringToPlotMark(Display * dpy, XrmValue * args,
			Cardinal * num_args, XrmValue * from, XrmValue * to,
			XtPointer * converter_data)
{
	static AtPlotMark mark;

	assert(to->addr == NULL);

	mark = parse_generic(from->addr, mark_table);

	if (mark == -1)
		XtDisplayStringConversionWarning(dpy, from->addr, XtRPlotMark);
	else {
		to->addr = (caddr_t) & mark;
		to->size = sizeof(AtPlotMark);
	}

	return (True);
}

static Boolean AtCvtStringToErrBarSide(Display * dpy, XrmValue * args,
      Cardinal * num_args, XrmValue * from, XrmValue * to,
      XtPointer * converter_data)
{
  static AtPlotSide bar_side;

  assert(to->addr == NULL);

  bar_side = parse_generic(from->addr, bar_side_table);

  if (bar_side == -1)
    XtDisplayStringConversionWarning(dpy, from->addr, XtRErrBarSide);
  else {
    to->addr = (caddr_t) & bar_side;
    to->size = sizeof(AtPlotSide);
  }

  return (True);
}

/*   Creation function */

Widget AtCreateQuadLinePlot(parent, name, arglist, argcount)
		 Widget parent;
		 char *name;
		 Arg *arglist;
		 Cardinal argcount;
{
	return (XtCreateWidget(name, atQuadLinePlotWidgetClass, parent,
							arglist, argcount));
}
