function monograting
% Diffraction grating geometry with monochromatic light, orders +4 to -4 shown.
% Cross section of the geometry of a diffraction grating, a common
% illustration in textbooks of optics, spectroscopy, and analytical chemistry.
% The grating surface is at the bottom of the diagram, along the x axis.
% The line labeled "Incident beam" is the direction of the incoming light
% beam.  The line labeled "0" is the direction of the zeroth-order
% diffraction, at the angle of specular reflection from the grating
% surface.  
% Tom O'Haver, toh@umd.edu, November 2011

global alphar wavelength d RulingDensity

% User-modifiable parameters:
wavelength = 187;  % initial value of wavelength
alphar = .75;   % initial value of angle of incidence
RulingDensity=300;       % initial value of grating ruling density, lines/mm,
d=1000000/RulingDensity;  % initial value of groove spacing, in mn

plotangles(alphar,wavelength,d)
whitebg('w')

% Attaches KeyPress test function to the figure.
set(gcf,'KeyPressFcn',@ReadKey)
uicontrol('Style','text')
% end of outer function
% ----------------------------SUBFUNCTIONS--------------------------------
function ReadKey(obj,eventdata)
global alphar wavelength d RulingDensity
key=get(gcf,'CurrentCharacter');
if isscalar(key),
  switch double(key),
    case 28
        % angle down when right arrow pressed.
          alphar=alphar-alphar/10;
          if alphar>1.55,alphar=1.55;end
          plotangles(alphar,wavelength,d);
    case 29
        % angle up when left arrow pressed.    
        alphar=alphar+alphar/10;
         if alphar>1.55,alphar=1.55;end
        plotangles(alphar,wavelength,d);
    case 30
        % Zooms up when up arrow pressed.
        wavelength=wavelength+wavelength/10;
        plotangles(alphar,wavelength,d);
    case 31
        % Zooms down when down arrow pressed.
        wavelength=wavelength-wavelength/10;
        if wavelength<0,wavelength=0;end
        plotangles(alphar,wavelength,d);
    case 97
        % When 'a' key is pressed, INcreases "RulingDensity" by 1 or 10%
        if RulingDensity==0,RulingDensity=1;end
        if RulingDensity>20,
            RulingDensity=round(RulingDensity+RulingDensity./10);
        else
            RulingDensity=RulingDensity+1;
        end
        d=1000000./RulingDensity;
        plotangles(alphar,wavelength,d);
    case 122
        % When 'z' key is pressed, DEcreases "RulingDensity" by 1 or 10%
        if RulingDensity==0,RulingDensity=1;end
        if RulingDensity>20,
            RulingDensity=round(RulingDensity-RulingDensity./10);
        else
            RulingDensity=RulingDensity-1;
        end
        d=1000000./RulingDensity;
        plotangles(alphar,wavelength,d);
    case 107
        % When 'K' key is pressed, prints out keyboard commands
        disp('Keyboard commands:')
        disp('Angle of incidence, degrees...left and right cursor arrows')
        disp('Wavelength, nm................up and down cursor arrows')
        disp('Ruling density, lines/mm......A/Z')
    otherwise  
       UnassignedKey=double(key)
       disp('Press k to print out list of keyboard commands')
   end % switch
end % if

function plotangles(alphar,wavelength,d)

r=pi/2; % Constant used below
% Compute the angle of the zero-order beam and the X and Y coordinates for
% the endoint of the incident and zero-order beams. 
xi=-cos(r-alphar);yi=sin(r-alphar); % Incident beam
angle0 = asin(-sin(alphar));x0=-cos(r-angle0);y0=sin(r-angle0); % Zero order

% Compute the angle of each diffracted beam and the X and Y coordinates for
% the endpoint of the line if the angle is on scale (90 degrees or less)
m=1;          % Diffraction Order 
angle1 = asin(m*wavelength/d-sin(alphar));if imag(angle1)==0;x1=-cos(r-angle1);y1=sin(r-angle1);else x1=0;y1=0;end;
m=-1;          % Diffraction Order 
angle2 = asin(m*wavelength/d-sin(alphar));if imag(angle2)==0;x2=-cos(r-angle2);y2=sin(r-angle2);else x2=0;y2=0;end;
m=2;          % Diffraction Order 
angle3 = asin(m*wavelength/d-sin(alphar));if imag(angle3)==0;x3=-.8.*cos(r-angle3);y3=.8.*sin(r-angle3);else x3=0;y3=0;end;
m=-2;          % Diffraction Order 
angle4 = asin(m*wavelength/d-sin(alphar));if imag(angle4)==0;x4=-.8.*cos(r-angle4);y4=.8.*sin(r-angle4);else x4=0;y4=0;end;
m=3;          % Diffraction Order 
angle5 = asin(m*wavelength/d-sin(alphar));if imag(angle5)==0;x5=-.6.*cos(r-angle5);y5=.6.*sin(r-angle5);else x5=0;y5=0;end;
m=-3;          % Diffraction Order 
angle6 = asin(m*wavelength/d-sin(alphar));if imag(angle6)==0;x6=-.6.*cos(r-angle6);y6=.6.*sin(r-angle6);else x6=0;y6=0;end;
m=4;          % Diffraction Order 
angle7 = asin(m*wavelength/d-sin(alphar));if imag(angle7)==0;x7=-.4.*cos(r-angle7);y7=.4.*sin(r-angle7);else x7=0;y7=0;end;
m=-4;          % Diffraction Order 
angle8 = asin(m*wavelength/d-sin(alphar));if imag(angle8)==0;x8=-.4.*cos(r-angle8);y8=.4.*sin(r-angle8);else x8=0;y8=0;end;

% Draw lines from 0,0 to the endpoints
clf;figure(1);
plot([0 xi],[0 yi],'k',[0 x0],[0 y0],'k--',...
    [0 x1],[0 y1],'k:',[0 x2],[0 y2],'k:',[0 x3],[0 y3],'k:',...
    [0 x4],[0 y4],'k:',[0 x5],[0 y5],'k:',[0 x6],[0 y6],'k:',...
    [0 x7],[0 y7],'k:',[0 x8],[0 y8],'k:',[0 0],[0 1.5],'m')

% Add labels and title
text(xi,yi,'Incident beam');
if imag(angle0)==0;text(x0,y0,'0');end;
if imag(angle1)==0;text(x1,y1,' 1');end;
if imag(angle2)==0;text(x2,y2,' -1');end;
if imag(angle3)==0;text(x3,y3,' 2');end;
if imag(angle4)==0;text(x4,y4,' -2');end;
if imag(angle5)==0;text(x5,y5,' 3');end;
if imag(angle6)==0;text(x6,y6,' -3');end;
if imag(angle7)==0;text(x7,y7,' 4');end;
if imag(angle8)==0;text(x8,y8,' -4');end;
title('Diffraction grating with monochromatic incident beam, orders -4 to +4.');
degrees=alphar*360/(2*pi);
xlabel(['Angle {\leftarrow \rightarrow} ' num2str(round(10*degrees)/10) ' degrees.   Wavelength {\uparrow \downarrow} '  num2str(round(10*wavelength)/10) ' nm.  Lines/mm A/Z ' num2str(round(10*1000000/d)/10) ])
whitebg('w')
axis([-1 1 0 1.5]);